/* Copyright 2009 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdint.h>
#include <stdio.h>
#include <string.h>

struct TIFF_TAG {
	uint16_t tag;
	uint16_t type;
	uint32_t count;
	uint32_t value;
} __attribute__((__packed__));

struct TIFF_RATIONAL {
	uint32_t n;
	uint32_t d;
} __attribute__((__packed__));

struct CANON_HEADER {
	uint16_t tiffByteOrder;
	uint16_t tiffMagicWord;
	uint32_t tiffIfdOffset;
	uint16_t crMagicWord;
	uint16_t crVersion;
	uint32_t crIfdOffset;
} __attribute__((__packed__));

struct CANON_GPS {
	uint16_t tagCount;
	struct TIFF_TAG tags[31];
	uint32_t nextIfd;
	struct TIFF_RATIONAL tag02[3];
	struct TIFF_RATIONAL tag04[3];
	struct TIFF_RATIONAL tag06;
	struct TIFF_RATIONAL tag07[3];
	uint8_t tag08[460];
	struct TIFF_RATIONAL tag0b;
	struct TIFF_RATIONAL tag0d;
	struct TIFF_RATIONAL tag0f;
	struct TIFF_RATIONAL tag11;
	uint8_t tag12[256];
	struct TIFF_RATIONAL tag14[3];
	struct TIFF_RATIONAL tag16[3];
	struct TIFF_RATIONAL tag18;
	struct TIFF_RATIONAL tag1a;
	uint8_t tag1b[256];
	uint8_t tag1c[256];
	uint8_t tag1d[11];
} __attribute__((__packed__));

const char* gpsTagNames[] = {
	"GPSVersionID",
	"GPSLatitudeRef",
	"GPSLatitude",
	"GPSLongitudeRef",
	"GPSLongitude",
	"GPSAltitudeRef",
	"GPSAltitude",
	"GPSTimeStamp",
	"GPSSatellites",
	"GPSStatus",
	"GPSMeasureMode",
	"GPSDOP",
	"GPSSpeedRef",
	"GPSSpeed",
	"GPSTrackRef",
	"GPSTrack",
	"GPSImgDirectionRef",
	"GPSImgDirection",
	"GPSMapDatum",
	"GPSDestLatitudeRef",
	"GPSDestLatitude",
	"GPSDestLongitudeRef",
	"GPSDestLongitude",
	"GPSDestBearingRef",
	"GPSDestBearing",
	"GPSDestDistanceRef",
	"GPSDestDistance",
	"GPSProcessingMethod",
	"GPSAreaInformation",
	"GPSDateStamp",
	"GPSDifferential"
};

void printValueHex(struct TIFF_TAG* tag) {
	printf("0x");
	for(unsigned int i=0; i<tag->count*8; i+=4) printf("%0x", (tag->value>>i)&0xf);
}

void printValueChar(struct TIFF_TAG* tag) {
	for(unsigned int i=0; i<tag->count*8; i+=8) printf("%c", (tag->value>>i)&0xff);
}

void printValueShort(struct TIFF_TAG* tag) {
	for(unsigned int i=0; i<tag->count*16; i+=16) printf("0x%04x", (tag->value>>i)&0xffff);
}

void printValueExChar(struct TIFF_TAG* tag, uint8_t* chars) {
	for(unsigned int i=0; i<tag->count; i++) {
		if(chars[i] == 0)
			break;
		printf("%c", chars[i]);
	}
}

void printValueExRational(struct TIFF_TAG* tag, struct TIFF_RATIONAL* rational) {
	for(unsigned int i=0; i<tag->count; i++) printf("%d/%d ", rational[i].n, rational[i].d);
}

void printCanonGps(struct CANON_GPS* canonGps) {
	printf("tag count: %d\n", canonGps->tagCount);
	printf("next ifd: %08x\n", canonGps->nextIfd);
	printf("\n");

	for(int i=0; i<canonGps->tagCount; i++) {
		printf("%s(%02X,%0X,%d): ",
			gpsTagNames[canonGps->tags[i].tag],
			canonGps->tags[i].tag,
			canonGps->tags[i].type,
			canonGps->tags[i].count);

		switch(i) {
			case 0x02: printValueExRational(&canonGps->tags[2], canonGps->tag02); break;
			case 0x04: printValueExRational(&canonGps->tags[4], canonGps->tag04); break;
			case 0x06: printValueExRational(&canonGps->tags[6], &canonGps->tag06); break;
			case 0x07: printValueExRational(&canonGps->tags[7], canonGps->tag07); break;
			case 0x08: printValueExChar(&canonGps->tags[8], canonGps->tag08); break;
			case 0x0b: printValueExRational(&canonGps->tags[11], &canonGps->tag0b); break;
			case 0x0d: printValueExRational(&canonGps->tags[13], &canonGps->tag0d); break;
			case 0x0f: printValueExRational(&canonGps->tags[15], &canonGps->tag0f); break;
			case 0x11: printValueExRational(&canonGps->tags[17], &canonGps->tag11); break;
			case 0x12: printValueExChar(&canonGps->tags[18], canonGps->tag12); break;
			case 0x14: printValueExRational(&canonGps->tags[20], canonGps->tag14); break;
			case 0x16: printValueExRational(&canonGps->tags[22], canonGps->tag16); break;
			case 0x18: printValueExRational(&canonGps->tags[24], &canonGps->tag18); break;
			case 0x1a: printValueExRational(&canonGps->tags[26], &canonGps->tag1a); break;
			case 0x1b: printValueExChar(&canonGps->tags[27], canonGps->tag1b); break;
			case 0x1c: printValueExChar(&canonGps->tags[28], canonGps->tag1c); break;
			case 0x1d: printValueExChar(&canonGps->tags[29], canonGps->tag1d); break;

			default:
				switch(canonGps->tags[i].type) {
					case 1: printValueHex(&canonGps->tags[i]); break;
					case 2: printValueChar(&canonGps->tags[i]); break;
					case 3: printValueShort(&canonGps->tags[i]); break;
					default: break;
				}
		}

		printf("\n");
	}
}

int isCanonGpsEmpty(struct CANON_GPS* gps) {
	int offset = sizeof(gps->tagCount) + sizeof(gps->tags[0]);
	int size = sizeof(struct CANON_GPS) - offset;
	uint8_t* data = (uint8_t*)gps;

	for(int i=offset; i<size; i++)
		if(data[i] != 0) return 0;

	return 1;
}

int isCanonGpsUsed(struct CANON_GPS* gps) {
	if(gps->tagCount != 31)
		return 0;

	return 1;
}

void clearCanonGps(struct CANON_GPS* gps) {
	memset(gps, 0, sizeof(struct CANON_GPS));
	gps->tagCount = 1;
	gps->tags[0].tag = 0;
	gps->tags[0].type = 1;
	gps->tags[0].count = 4;
	gps->tags[0].value = 0x00000202;
}

void defaultCanonGps(struct CANON_GPS* gps, uint32_t gpsIfd) {
	memset(gps, 0, sizeof(struct CANON_GPS));
	gps->tagCount = 31;

	for(int i=0; i<31; i++) gps->tags[i].tag = i;

	gps->tags[0x00].type = 1;
	gps->tags[0x00].count = 4;
	gps->tags[0x00].value = 0x00000202;

	gps->tags[0x01].type = 2;
	gps->tags[0x01].count = 2;
	gps->tags[0x01].value = 0;

	gps->tags[0x02].type = 5;
	gps->tags[0x02].count = 3;
	gps->tags[0x02].value = gpsIfd + 0x017a;

	gps->tags[0x03].type = 2;
	gps->tags[0x03].count = 2;
	gps->tags[0x03].value = 0;

	gps->tags[0x04].type = 5;
	gps->tags[0x04].count = 3;
	gps->tags[0x04].value = gpsIfd + 0x0192;

	gps->tags[0x05].type = 1;
	gps->tags[0x05].count = 1;
	gps->tags[0x05].value = 0;

	gps->tags[0x06].type = 5;
	gps->tags[0x06].count = 1;
	gps->tags[0x06].value = gpsIfd + 0x01aa;

	gps->tags[0x07].type = 5;
	gps->tags[0x07].count = 3;
	gps->tags[0x07].value = gpsIfd + 0x01b2;

	gps->tags[0x08].type = 2;
	gps->tags[0x08].count = 460;
	gps->tags[0x08].value = gpsIfd + 0x01ca;

	gps->tags[0x09].type = 2;
	gps->tags[0x09].count = 2;
	gps->tags[0x09].value = 0;

	gps->tags[0x0a].type = 2;
	gps->tags[0x0a].count = 2;
	gps->tags[0x0a].value = 0;

	gps->tags[0x0b].type = 5;
	gps->tags[0x0b].count = 1;
	gps->tags[0x0b].value = gpsIfd + 0x0396;

	gps->tags[0x0c].type = 2;
	gps->tags[0x0c].count = 2;
	gps->tags[0x0c].value = 0;

	gps->tags[0x0d].type = 5;
	gps->tags[0x0d].count = 1;
	gps->tags[0x0d].value = gpsIfd + 0x039e;

	gps->tags[0x0e].type = 2;
	gps->tags[0x0e].count = 2;
	gps->tags[0x0e].value = 0;

	gps->tags[0x0f].type = 5;
	gps->tags[0x0f].count = 1;
	gps->tags[0x0f].value = gpsIfd + 0x03a6;

	gps->tags[0x10].type = 2;
	gps->tags[0x10].count = 2;
	gps->tags[0x10].value = 0;

	gps->tags[0x11].type = 5;
	gps->tags[0x11].count = 1;
	gps->tags[0x11].value = gpsIfd + 0x03ae;

	gps->tags[0x12].type = 2;
	gps->tags[0x12].count = 256;
	gps->tags[0x12].value = gpsIfd + 0x03b6;

	gps->tags[0x13].type = 2;
	gps->tags[0x13].count = 2;
	gps->tags[0x13].value = 0;

	gps->tags[0x14].type = 5;
	gps->tags[0x14].count = 3;
	gps->tags[0x14].value = gpsIfd + 0x04b6;

	gps->tags[0x15].type = 2;
	gps->tags[0x15].count = 2;
	gps->tags[0x15].value = 0;

	gps->tags[0x16].type = 5;
	gps->tags[0x16].count = 3;
	gps->tags[0x16].value = gpsIfd + 0x04ce;

	gps->tags[0x17].type = 2;
	gps->tags[0x17].count = 2;
	gps->tags[0x17].value = 0;

	gps->tags[0x18].type = 5;
	gps->tags[0x18].count = 1;
	gps->tags[0x18].value = gpsIfd + 0x04e6;

	gps->tags[0x19].type = 2;
	gps->tags[0x19].count = 2;
	gps->tags[0x19].value = 0;

	gps->tags[0x1a].type = 5;
	gps->tags[0x1a].count = 1;
	gps->tags[0x1a].value = gpsIfd + 0x04ee;

	gps->tags[0x1b].type = 7;
	gps->tags[0x1b].count = 256;
	gps->tags[0x1b].value = gpsIfd + 0x04f6;

	gps->tags[0x1c].type = 7;
	gps->tags[0x1c].count = 256;
	gps->tags[0x1c].value = gpsIfd + 0x05f6;

	gps->tags[0x1d].type = 2;
	gps->tags[0x1d].count = 11;
	gps->tags[0x1d].value = gpsIfd + 0x06f6;

	gps->tags[0x1e].type = 3;
	gps->tags[0x1e].count = 1;
	gps->tags[0x1e].value = 0;
}

int readCanonHeader(struct CANON_HEADER* header, FILE* file) {
	fseek(file, 0, SEEK_SET);
	fread(header, sizeof(struct CANON_HEADER), 1, file);

	if(header->tiffByteOrder != 0x4949 || header->tiffMagicWord != 0x002a) {
		fprintf(stderr, "unknown tiff header\n");
		return -1;
	}

	if(header->crMagicWord != 0x5243 || header->crVersion != 0x0002) {
		fprintf(stderr, "unknown canon raw header\n");
		return -1;
	}

	return 0;
}

uint32_t findGpsIfd(struct CANON_HEADER* header, FILE* file) {
	uint16_t tagCount;

	fseek(file, header->tiffIfdOffset, SEEK_SET);
	fread(&tagCount, sizeof(uint16_t), 1, file);

	// read tags
	for(int i=0; i<tagCount; i++) {
		struct TIFF_TAG tag;

		fread(&tag, sizeof(tag), 1, file);

		if(tag.tag == 0x8825)
			return tag.value;
	}

	return 0;
}

int info(char* filename) {
	FILE* file;
	struct CANON_GPS gps;
	struct CANON_HEADER header;

	if((file = fopen(filename, "rb")) == NULL)
		return -1;

	if(readCanonHeader(&header, file))
		return -1;

	printf("tiff ifd offset: %08x\n", header.tiffIfdOffset);
	printf("canon raw ifd offset: %08x\n", header.crIfdOffset);

	// search gps ifd
	uint32_t gpsIfd = findGpsIfd(&header, file);

	// check gps ifd
	if(gpsIfd == 0) {
		fprintf(stderr, "no gps ifd found\n");
		return -1;
	}

	printf("gps ifd: %08x\n", gpsIfd);

	// read gps tags
	fseek(file, gpsIfd, SEEK_SET);
	fread(&gps, sizeof(gps), 1, file);

	int canonGpsUsed = isCanonGpsUsed(&gps);

	if(canonGpsUsed) {
		printf("canon gps in use: yes\n");
	} else {
		printf("canon gps in use: no\n");

		if(isCanonGpsEmpty(&gps))
			printf("canon gps space: yes\n");
		else
			printf("canon gps space: no\n");
	}

	printCanonGps(&gps);

	fclose(file);

	return 0;
}

int clear(char* filename) {
	FILE* file;
	struct CANON_GPS gps;
	struct CANON_HEADER header;

	if((file = fopen(filename, "r+b")) == NULL)
		return -1;

	if(readCanonHeader(&header, file))
		return -1;

	// search gps ifd
	uint32_t gpsIfd = findGpsIfd(&header, file);

	// check gps ifd
	if(gpsIfd == 0) {
		fprintf(stderr, "no gps ifd found\n");
		return -1;
	}

	// read gps tags
	fseek(file, gpsIfd, SEEK_SET);
	fread(&gps, sizeof(gps), 1, file);

	int canonGpsUsed = isCanonGpsUsed(&gps);

	if(!canonGpsUsed) {
		fprintf(stderr, "canon gps not in use\n");
		return -1;
	}

	clearCanonGps(&gps);

	// write gps tags
	fseek(file, gpsIfd, SEEK_SET);
	fwrite(&gps, sizeof(gps), 1, file);

	fclose(file);

	return 0;
}

int writeGpsTags(char* filename, char* latitude, char* latitudeRef, char* longitude, char* longitudeRef, char* altitude, char* altitudeRef) {
	FILE* file;
	struct CANON_GPS gps;
	struct CANON_HEADER header;

	if((file = fopen(filename, "r+b")) == NULL)
		return -1;

	if(readCanonHeader(&header, file))
		return -1;

	// search gps ifd
	uint32_t gpsIfd = findGpsIfd(&header, file);

	// check gps ifd
	if(gpsIfd == 0) {
		fprintf(stderr, "no gps ifd found\n");
		return -1;
	}

	// read gps tags
	fseek(file, gpsIfd, SEEK_SET);
	fread(&gps, sizeof(gps), 1, file);

	int canonGpsUsed = isCanonGpsUsed(&gps);

	if(!canonGpsUsed) {
		if(!isCanonGpsEmpty(&gps)) {
			fprintf(stderr, "canon gps space not found\n");
			return -1;
		}
	}

	defaultCanonGps(&gps, gpsIfd);

	if(latitude != 0) {
		int h, m;
		float s;

		sscanf(latitude, "'%d %d %f'", &h, &m, &s);

		gps.tag02[0].n = h;
		gps.tag02[0].d = 1;
		gps.tag02[1].n = m*100000 + (int)(s*100000.0/60.0);
		gps.tag02[1].d = 100000;
		gps.tag02[2].n = 0;
		gps.tag02[2].d = 1;
	}

	if(latitudeRef != 0) {
		gps.tags[1].value = latitudeRef[0];
	}

	if(longitude != 0) {
		int h, m;
		float s;

		sscanf(longitude, "'%d %d %f'", &h, &m, &s);

		gps.tag04[0].n = h;
		gps.tag04[0].d = 1;
		gps.tag04[1].n = m*100000 + (int)(s*100000.0/60.0);
		gps.tag04[1].d = 100000;
		gps.tag04[2].n = 0;
		gps.tag04[2].d = 1;
	}

	if(longitudeRef != 0) {
		gps.tags[3].value = longitudeRef[0];
	}

	if(altitude != 0) {
		int m;

		sscanf(altitude, "%d", &m);

		gps.tag06.n = m * 10;
		gps.tag06.d = 10;
	}

	if(altitudeRef != 0) {
		if(!strcmp(altitudeRef, "'Below Sea Level'"))
			gps.tags[5].value = 1;
		else
			gps.tags[5].value = 0;
	}

	// write gps tags
	fseek(file, gpsIfd, SEEK_SET);
	fwrite(&gps, sizeof(gps), 1, file);

	fclose(file);

	return 0;
}

int main(int argc, char** argv) {
	if(argc == 3) {
		if(!strcmp(argv[1], "-info"))
			return info(argv[2]);
		else if(!strcmp(argv[1], "-clear"))
			return clear(argv[2]);
	} else if(argc >= 3) {
		char* latitude = 0;
		char* latitudeRef = 0;
		char* longitude = 0;
		char* longitudeRef = 0;
		char* altitude = 0;
		char* altitudeRef = 0;

		for(int i=1; i<argc; i++) {
			if(!strncmp(argv[i], "-GPSLatitude=", 13))
				latitude = argv[i] + 13;
			else if(!strncmp(argv[i], "-GPSLatitudeRef=", 16))
				latitudeRef = argv[i] + 16;
			else if(!strncmp(argv[i], "-GPSLongitude=", 14))
				longitude = argv[i] + 14;
			else if(!strncmp(argv[i], "-GPSLongitudeRef=", 17))
				longitudeRef = argv[i] + 17;
			else if(!strncmp(argv[i], "-GPSAltitude=", 13))
				altitude = argv[i] + 13;
			else if(!strncmp(argv[i], "-GPSAltitudeRef=", 16))
				altitudeRef = argv[i] + 16;
		}

		char* filename = argv[argc-1];
		int l = strlen(filename);
		filename[l-3] = 'C';
		filename[l-2] = 'R';
		filename[l-1] = '2';

		return writeGpsTags(argv[argc-1], latitude, latitudeRef, longitude, longitudeRef, altitude, altitudeRef);
	} else {
		return -1;
	}
}
