/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "ImageFilter.h"

#include <QByteArray>
#include <QFile>
#include <QImageWriter>

ImageFilter::~ImageFilter() {
}

BergPhoto::Parameters ImageFilter::parameters() {
	return _parameters;
}

void ImageFilter::setParameters(BergPhoto::Parameters parameters) {
	_parameters = parameters;
}

QString ImageWriter::parameterImageWriter = "imageWriter";
QString ImageWriter::parameterFileFormat = "fileFormat";
QString ImageWriter::parameterCompress = "compress";
QString ImageWriter::parameterQuality = "quality";

bool ImageWriter::write(QString fileName, Image* image, MetadataResource* resource, BergPhoto::Parameters parameter) {
	QFile file(fileName);

	file.open(QIODevice::WriteOnly);

	bool r = write(&file, image, resource, parameter);

	file.close();

	return r;
}

ResizeFilter::ResizeFilter() {
}

ResizeFilter::~ResizeFilter() {
}

QString ResizeFilter::name() {
	return QString("ResizeFilter");
}

bool ResizeFilter::accepts(QString inputFormat, QString outputFormat) {
	return inputFormat == RgbImage48::format() && outputFormat == RgbImage48::format();
}

Image* ResizeFilter::filter(Image* input, QString outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	int sourceWidth = input->width();
	int sourceHeight = input->height();
	int targetWidth = _parameters.value("targetWidth").toInt();
	int targetHeight = _parameters.value("targetHeight").toInt();

	RgbImage48* sourceImage = qobject_cast<RgbImage48*>(input);
	RgbImage48* targetImage = new RgbImage48(targetWidth, targetHeight);

	targetImage->setMetadata(sourceImage->metadata());

	for(int y=0; y<targetHeight; y++) {
		quint16* sourceLine = sourceImage->dataLine(y*sourceHeight/targetHeight);
		quint16* targetLine = targetImage->dataLine(y);

		for(int x=0; x<targetWidth; x++) {
			int offset = x*sourceWidth/targetWidth*3;

			targetLine[0] = sourceLine[offset];
			targetLine[1] = sourceLine[offset+1];
			targetLine[2] = sourceLine[offset+2];

			targetLine += 3;
		}
	}

	return targetImage;
}

QString QtImageWriter::name = "QtImageWriter";

bool QtImageWriter::accepts(QString format) {
	return format == RgbImage32::format();
}

bool QtImageWriter::write(QIODevice* device, Image* image, MetadataResource* resource, BergPhoto::Parameters parameters) {
	if(!accepts(image->format()))
		return false;

	QImageWriter imageWriter(device, QByteArray());

	if(parameters.contains(ImageWriter::parameterCompress))
		imageWriter.setCompression(parameters.value(ImageWriter::parameterCompress).toBool() ? 1 : 0);
	else
		imageWriter.setCompression(0);

	if(parameters.contains(ImageWriter::parameterQuality))
		imageWriter.setQuality(parameters.value(ImageWriter::parameterQuality).toInt());
	else
		imageWriter.setQuality(100);

	return imageWriter.write(*qobject_cast<RgbImage32*>(image)->toQImage());
}
