/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "LosslessJpegDecoder.h"

HuffmanNode::HuffmanNode() {
	_nodes[0] = 0;
	_nodes[1] = 0;
	_hasValue = false;
	_value = -1;
}

HuffmanNode::~HuffmanNode() {
	if(_nodes[0] != 0)
		delete _nodes[0];

	if(_nodes[1] != 0)
		delete _nodes[1];
}

void HuffmanNode::setValue(int value) {
	_hasValue = true;
	_value = value;
}

HuffmanNode* HuffmanNode::getMostLeft(int level) {
	if(_hasValue)
		return 0;

	if(level == 0)
		return this;

	HuffmanNode* nextNode = 0;

	for(int i=0; i<2; i++) {
		if(_nodes[i] == 0)
			_nodes[i] = new HuffmanNode();

		if((nextNode = _nodes[i]->getMostLeft(level-1)) != 0)
			return nextNode;
	}

	return 0;
}

LosslessJpegDecoder::LosslessJpegDecoder(QIODevice* device) : _dataStream(device), _bitStream(device) {
	_dataStream.setByteOrder(QDataStream::BigEndian);

	for(int i=0; i<4; i++)
		_huffmanTrees[i] = 0;
}

LosslessJpegDecoder::~LosslessJpegDecoder() {
	for(int i=0; i<4; i++)
		if(_huffmanTrees[i] != 0)
			delete _huffmanTrees[i];
}

int LosslessJpegDecoder::precision() {
	return _precision;
}

int LosslessJpegDecoder::lines() {
	return _lines;
}

int LosslessJpegDecoder::samples() {
	return _samples;
}

int LosslessJpegDecoder::components() {
	return _components;
}

void LosslessJpegDecoder::decodeHeader() {
	bool done = false;

	quint16 marker;
	_dataStream >> marker;

	if(marker != MarkerSOI)
		return;

	do {
		quint16 length;
		_dataStream >> marker;
		_dataStream >> length;
		length -= 2;

		switch(marker) {
			case MarkerSOF3:
				decodeSOF3(length);
				break;

			case MarkerDHT:
				decodeDHT(length);
				break;

			case MarkerSOS:
				decodeSOS(length);
				done = true;
				break;

			default:
				_dataStream.skipRawData(length);
				break;
		}
	} while(!done);
}

void LosslessJpegDecoder::decodeImage(quint16* data) {
	int width = _samples*_components;

	for(int i=0; i<_lines; i++, data+=width)
		decodeRow(data);
}

int LosslessJpegDecoder::buildTree() {
	int tableLength = 0;

	quint8 tableId;
	_dataStream >> tableId;
	_huffmanTrees[tableId] = new HuffmanNode();

	quint8 codeLengthList[16];
	for(int i=0; i<16; i++) {
		_dataStream >> codeLengthList[i];
		tableLength += codeLengthList[i];
	}

	for(int i=0; i<16; i++) {
		for(int j=0; j<codeLengthList[i]; j++) {
			quint8 value;
			_dataStream >> value;
			_huffmanTrees[tableId]->getMostLeft(i+1)->setValue(value);
		}
	}

	return tableLength + 17;
}

void LosslessJpegDecoder::decodeDHT(quint16 length) {
	if(length <= 0)
		return;

	do {
		length -= buildTree();
	} while(length > 0);
}

void LosslessJpegDecoder::decodeSOF3(quint16 length) {
	Q_UNUSED(length);

	_dataStream >> _precision;;
	_dataStream >> _lines;
	_dataStream >> _samples;
	_dataStream >> _components;

	// TODO: other prediction modes for non canon ljpegs
	for(int i=0; i<_components; i++)
		_prediction[i] = 1 << (_precision - 1);

	_dataStream.skipRawData(_components*3);
}

void LosslessJpegDecoder::decodeSOS(quint16 length) {
	_dataStream.skipRawData(length);
}

int LosslessJpegDecoder::decodeDiff(HuffmanNode* node) {
	int length = node->decode(_bitStream);

	int diff = (int)_bitStream.bits(length);

	if((diff & (1 << (length-1))) == 0)
		diff -= (1 << length) - 1;

	return diff;
}

void LosslessJpegDecoder::decodeRow(quint16* row) {
	for(int j=0; j<_components; j++)
		row[j] = _prediction[j] += decodeDiff(_huffmanTrees[j]);

	row += _components;

	for(int i=1; i<_samples; i++) {
		for(int j=0; j<_components; j++)
			row[j] = row[-_components+j] + decodeDiff(_huffmanTrees[j]);

		row += _components;
	}
}
