/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef LOSSLESSJPEGDECODER_H_
#define LOSSLESSJPEGDECODER_H_

#include <QDataStream>
#include <QIODevice>

class JpegBitStream {
public:
	JpegBitStream(QIODevice* device) : _dataStream(device) {
		_pos = 0;
	}

	inline int bit() {
		if(_pos == 0) {
			_dataStream >> _byte;
			if(_byte == 0xff)
				_dataStream.skipRawData(1);

			_pos = 8;
		}

		return (_byte >> --_pos) & 1;
	}

	inline quint32 bits(int length) {
		quint32 bits = 0;

		int nextLength = qMin(_pos, length);
		length -= nextLength;
		_pos -= nextLength;
		bits = (_byte >> _pos) & ((1<<nextLength) - 1);

		while(length > 0) {
			_dataStream >> _byte;
			if(_byte == 0xff)
				_dataStream.skipRawData(1);

			nextLength = qMin(8, length);
			length -= nextLength;
			_pos = 8 - nextLength;

			bits <<= nextLength;
			bits |= (_byte >> _pos) & ((1 << nextLength) - 1);
		}

		return bits;
	}

private:
	QDataStream _dataStream;
	quint8 _byte;
	int _pos;
};

class HuffmanNode {
public:
	HuffmanNode();
	~HuffmanNode();

	void setValue(int value);

	HuffmanNode* getMostLeft(int level);

	inline int decode(JpegBitStream& bitStream) {
		HuffmanNode* nextNode = _nodes[bitStream.bit()];

		if(nextNode == 0)
			return -1;

		if(nextNode->_hasValue)
			return nextNode->_value;

		return nextNode->decode(bitStream);
	}

private:
	HuffmanNode* _nodes[2];
	bool _hasValue;
	int _value;
};

class LosslessJpegDecoder {
public:
	enum Marker {
		MarkerSOF3 = 0xffc3,
		MarkerDHT = 0xffc4,
		MarkerSOI = 0xffd8,
		MarkerSOS = 0xffda
	};

	LosslessJpegDecoder(QIODevice* device);
	virtual ~LosslessJpegDecoder();

	int precision();
	int lines();
	int samples();
	int components();

	void decodeHeader();
	void decodeImage(quint16* data);

private:
	QDataStream _dataStream;
	JpegBitStream _bitStream;
	HuffmanNode* _huffmanTrees[4];
	quint8 _precision;
	quint16 _lines;
	quint16 _samples;
	quint8 _components;
	int _prediction[4];

	int buildTree();
	void decodeDHT(quint16 length);
	void decodeSOF3(quint16 length);
	void decodeSOS(quint16 length);
	int decodeDiff(HuffmanNode* node);
	void decodeRow(quint16* row);
};

#endif /*LOSSLESSJPEGDECODER_H_*/
