/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef METADATA_H_
#define METADATA_H_

#include <QList>
#include <QVariant>

class Rational {
public:
	Rational();
	Rational(qlonglong numerator, qlonglong denominator);

	qlonglong numerator();
	void setNumerator(qlonglong numerator);

	qlonglong denominator();
	void setDenominator(qlonglong denominator);

	QString toString();

private:
	qlonglong _numerator;
	qlonglong _denominator;
};

Q_DECLARE_METATYPE(Rational)

class MetadataQName {
public:
	MetadataQName();
	MetadataQName(QString namespaceUri, QString localName);

	QString namespaceUri() const;
	void setNamespaceUri(QString namespaceUri);

	QString localName() const;
	void setLocalName(QString localName);

	bool operator !=(const MetadataQName other) const;
	bool operator <(const MetadataQName other) const;
	bool operator ==(const MetadataQName other) const;

private:
	QString _namespaceUri;
	QString _localName;
};

Q_DECLARE_METATYPE(MetadataQName)

class MetadataNode;
class MetadataDocument;
class MetadataResource;
class MetadataProperty;
class MetadataSimpleProperty;
class MetadataListProperty;
class MetadataListItem;
class MetadataComplexProperty;

class MetadataNode {
public:
	static MetadataDocument* toDocument(MetadataNode* node);
	static MetadataResource* toResource(MetadataNode* node);
	static MetadataProperty* toProperty(MetadataNode* node);
	static MetadataSimpleProperty* toSimpleProperty(MetadataNode* node);
	static MetadataListProperty* toListProperty(MetadataNode* node);
	static MetadataListItem* toListItem(MetadataNode* node);
	static MetadataComplexProperty* toComplexProperty(MetadataNode* node);

	MetadataNode(MetadataNode* parent=0);
	virtual ~MetadataNode();

	virtual bool isDocument();
	virtual bool isResource();
	virtual bool isProperty();
	virtual bool isSimpleProperty();
	virtual bool isListProperty();
	virtual bool isListItem();
	virtual bool isComplexProperty();

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true);

	virtual QList<MetadataResource*> queryResources(QString resourceUri);
	virtual MetadataResource* queryResource(QString resourceUri);

	virtual QList<MetadataProperty*> queryProperties(QString resourceUri, MetadataQName qName, QVariant value=QVariant());
	virtual MetadataProperty* queryProperty(QString resourceUri, MetadataQName qName, QVariant value=QVariant());

	virtual QList<MetadataProperty*> queryProperties(MetadataQName qName, QVariant value=QVariant());
	virtual MetadataProperty* queryProperty(MetadataQName qName, QVariant value=QVariant());

	virtual QList<QVariant> queryValues(QString resourceUri, MetadataQName qName);
	virtual QVariant queryValue(QString resourceUri, MetadataQName qName);

	virtual QList<QVariant> queryValues(MetadataQName qName);
	virtual QVariant queryValue(MetadataQName qName);

	MetadataDocument* toDocument();
	MetadataResource* toResource();
	MetadataProperty* toProperty();
	MetadataSimpleProperty* toSimpleProperty();
	MetadataListProperty* toListProperty();
	MetadataListItem* toListItem();
	MetadataComplexProperty* toComplexProperty();

	MetadataNode* parent();
	void setParent(MetadataNode* parent);

	QList<MetadataNode*> children();

	void removeChild(MetadataNode* child);

protected:
	QList<MetadataNode*> cloneChildren(MetadataNode* parent, bool deep=true);

	MetadataNode* _parent;
	QList<MetadataNode*> _children;
};

class MetadataDocument : public MetadataNode {
public:
	MetadataDocument(MetadataNode* parent=0);

	virtual bool isDocument();

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true);
};

class MetadataResource : public MetadataNode {
public:
	MetadataResource(MetadataNode* parent, QString resourceUri=QString());

	virtual bool isResource();

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true);

	QString resourceUri();
	void setResourceUri(QString resourceUri);

protected:
	QString _resourceUri;
};

class MetadataProperty : public MetadataNode {
public:
	MetadataProperty(MetadataNode* parent, MetadataQName qName);

	virtual bool isProperty();

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true);

	MetadataQName qName();
	void setQName(MetadataQName qName);

	virtual QList<QVariant> values();

protected:
	MetadataQName _qName;
};

class MetadataSimpleProperty : public MetadataProperty {
public:
	MetadataSimpleProperty(MetadataNode* parent, MetadataQName qName, QVariant value=QVariant());

	virtual bool isSimpleProperty();

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true);

	virtual QList<QVariant> values();

	QVariant value();
	void setValue(QVariant value);

private:
	QVariant _value;
};

class MetadataListProperty : public MetadataProperty {
public:
	MetadataListProperty(MetadataNode* parent, MetadataQName qName);

	virtual bool isListProperty();

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true);

	virtual QList<QVariant> values();

	void append(QList<QVariant> values);
};

class MetadataListItem : public MetadataNode {
public:
	MetadataListItem(MetadataNode* parent, QVariant value=QVariant());

	virtual bool isListItem();

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true);

	QVariant value();
	void setValue(QVariant value);

private:
	QVariant _value;
};

class MetadataComplexProperty : public MetadataProperty {
public:
	MetadataComplexProperty(MetadataNode* parent, MetadataQName qName);

	virtual bool isComplexProperty();

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true);

	virtual QList<QVariant> values();
};

class MetadataTransform {
public:
	MetadataTransform();
	MetadataTransform(QVariant parameter);
	virtual ~MetadataTransform();

	QVariant parameter();
	void setParameter(QVariant parameter);

	virtual MetadataNode* transform(MetadataNode* source) = 0;

protected:
	QVariant _parameter;
};

#endif /*METADATA_H_*/
