/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "CanonCr2.h"

#include <QBuffer>
#include <QTextStream>

#include <LosslessJpegDecoder.h>
#include <QtUtils.h>

QString CanonCr2::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/makernote/canon/1.0/";
MetadataQName CanonCr2::propertyColorBalance = MetadataQName(CanonCr2::namespaceUri, "ColorBalance");
MetadataQName CanonCr2::propertyBodyInfo = MetadataQName(CanonCr2::namespaceUri, "BodyInfo");
MetadataQName CanonCr2::propertySensorInfo = MetadataQName(CanonCr2::namespaceUri, "SensorInfo");
MetadataQName CanonCr2::propertyShotInfo = MetadataQName(CanonCr2::namespaceUri, "ShotInfo");
MetadataQName CanonCr2::propertySensorPlanes = MetadataQName(CanonCr2::namespaceUri, "SensorPlanes");
MetadataQName CanonCr2::propertyDefaultWhiteBalanceNeutral5200K = MetadataQName(CanonCr2::namespaceUri, "DefaultWhiteBalanceNeutral5200K");

QMap<int, QString> CanonCr2Parser::_makerNodeTagNameMap = CanonCr2Parser::createMakerNodeTagNameMap();


CanonCr2Image::CanonCr2Image(int width, int height) : BayerImage16(width, height) {
	_format = format();
}

CanonCr2Image::~CanonCr2Image() {
}

QMap<int, QString> CanonCr2Parser::createMakerNodeTagNameMap() {
	QMap<int, QString> tagNameMap;

	tagNameMap.insert(0x4, CanonCr2::propertyShotInfo.localName());
	tagNameMap.insert(0xc, CanonCr2::propertyBodyInfo.localName());
	tagNameMap.insert(0xe0, CanonCr2::propertySensorInfo.localName());
	tagNameMap.insert(0x4001, CanonCr2::propertyColorBalance.localName());
	tagNameMap.insert(0xc640, CanonCr2::propertySensorPlanes.localName());

	return tagNameMap;
}

CanonCr2Parser::CanonCr2Parser(QIODevice* device) : TiffParser(device) {
}

MetadataQName CanonCr2Parser::tagQName(quint16 tag) {
	if(_makerNodeTagNameMap.contains(tag))
		return MetadataQName(CanonCr2::namespaceUri, _makerNodeTagNameMap.value(tag));
	else
		return TiffParser::tagQName(tag);
}

void CanonCr2Parser::parseMakerNote(MetadataNode* parent, quint32 offset) {
	parseDirectory(parent, offset);
}

CanonCr2File::CanonCr2File(QIODevice* device) : RawFile() {
	_device = device;
	_canonCr2Parser = new CanonCr2Parser(_device);
}

CanonCr2File::~CanonCr2File() {
	delete _canonCr2Parser;
}

MetadataDocument* CanonCr2File::readMetadata() {
	if(_metadata == 0) {
		CanonCr2MetadataCleanupTransform transform;

		_metadata = transform.transform(_canonCr2Parser->parse())->toDocument();
	}

	return _metadata;
}

Image* CanonCr2File::readImage() {
	if(_image == 0) {
		readMetadata();

		_device->seek(_canonCr2Parser->directoryOffset(3));

		LosslessJpegDecoder losslessJpegDecoder(_device);
		losslessJpegDecoder.decodeHeader();

		int width = losslessJpegDecoder.samples() * losslessJpegDecoder.components();
		int height = losslessJpegDecoder.lines();

		CanonCr2Image canonCr2Image(width, height);

		MetadataResource* resource = _metadata->queryResource("*");
		canonCr2Image.setMetadata(resource);

		losslessJpegDecoder.decodeImage(canonCr2Image.data());

		int whiteLevel = (1 << losslessJpegDecoder.precision()) - 1;
		new MetadataSimpleProperty(resource, RawFile::propertyWhiteLevel, QVariant(whiteLevel));

		// TODO: read black level from masked sensor area
		int blackLevel = 1 << (losslessJpegDecoder.precision() - 4);
		new MetadataSimpleProperty(resource, RawFile::propertyBlackLevel, QVariant(blackLevel));

		Cr2BayerConverter cr2BayerConverter;
		_image = cr2BayerConverter.filter(&canonCr2Image, BayerImage16::format());
	}

	return _image;
}

QString Cr2BayerConverter::name() {
	return QString("Cr2BayerFilter");
}

bool Cr2BayerConverter::accepts(QString inputFormat, QString outputFormat) {
	return inputFormat == CanonCr2Image::format() && outputFormat == BayerImage16::format();
}

Image* Cr2BayerConverter::filter(Image* input, QString outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	CanonCr2Image* canonCr2Image = qobject_cast<CanonCr2Image*>(input);

	int width = canonCr2Image->width();
	int height = canonCr2Image->height();

	MetadataResource* metadataResource = canonCr2Image->metadata();
	QList<QVariant> sensorPlanes = metadataResource->queryValues(CanonCr2::propertySensorPlanes);

	int strideR = sensorPlanes.at(0).toInt();
	int strideA = sensorPlanes.at(1).toInt();
	int strideB = sensorPlanes.at(2).toInt();

	BayerImage16* output = new BayerImage16(width, height);
	output->setMetadata(canonCr2Image->metadata());
	output->setPrecision(canonCr2Image->precision());

	QVector<int> pattern;
	pattern << Image::ColorRed << Image::ColorGreen << Image::ColorGreen << Image::ColorBlue;
	output->setPattern(pattern);

	quint16* inputData = canonCr2Image->data();

#if 0
	quint16* outputData = output->data();

	memcpy(outputData, inputData, width*height*2);
#else
	quint16* outputData0 = 0;
	quint16* outputData1 = 0;

	for(int r=0; r<strideR; r++) {
		outputData0 = output->data() + r*strideA;
		outputData1 = outputData0 + width;

		for(int i=0; i<height/2; i++) {
			memcpy(outputData0, inputData, strideA*2);
			inputData += strideA;
			outputData0 += width*2;

			memcpy(outputData1, inputData, strideA*2);
			inputData += strideA;
			outputData1 += width*2;
		}
	}

	outputData0 = output->data() + strideR*strideA;
	outputData1 = outputData0 + width;

	for(int i=0; i<height/2; i++) {
		memcpy(outputData0, inputData, strideB*2);
		inputData += strideB;
		outputData0 += width*2;

		memcpy(outputData1, inputData, strideB*2);
		inputData += strideB;
		outputData1 += width*2;
	}
#endif
	return output;
}

CanonCr2MetadataCleanupTransform::~CanonCr2MetadataCleanupTransform() {
}

MetadataNode* CanonCr2MetadataCleanupTransform::transform(MetadataNode* source) {
	MetadataResource* resource = source->children().at(0)->toResource();
	resource->setResourceUri(QString());

	// reassign properties from other ifds
	MetadataResource* resourceDir3 = source->children().at(3)->toResource();
	MetadataProperty* sensorPlanesProperty = resourceDir3->queryProperty(CanonCr2::propertySensorPlanes);
	sensorPlanesProperty->setParent(resource);

	// remove other ifds
	for(int i=1; i<4; i++)
		source->removeChild(source->children().at(1));

	// remove needless properties
	resource->removeChild(resource->queryProperty(Tiff::propertyBitsPerSample));
	resource->removeChild(resource->queryProperty(Tiff::propertyCompression));

	QString make = resource->queryValue(Tiff::propertyMake).toString();
	QString model = resource->queryValue(Tiff::propertyModel).toString();

	RawUtils::appendCameraDefinition(resource, model);

	// body id
	new MetadataSimpleProperty(resource, RawFile::propertyBodyId, resource->queryValue(CanonCr2::propertyBodyInfo));

	// sensor crop
	QList<int> cropValues = QtUtils::toTypedList<int>(resource->queryValues(CanonCr2::propertySensorInfo));
	MetadataListProperty* cropProperty = new MetadataListProperty(resource, RawFile::propertySensorCrop);
	new MetadataListItem(cropProperty, cropValues.at(5));
	new MetadataListItem(cropProperty, cropValues.at(6));
	new MetadataListItem(cropProperty, cropValues.at(7)-cropValues.at(5)+1);
	new MetadataListItem(cropProperty, cropValues.at(8)-cropValues.at(6)+1);

	// shot infos
	QList<QVariant> shotInfoValues = resource->queryValues(CanonCr2::propertyShotInfo);

	// white balance
	QList<QVariant> colorBalanceValues = resource->queryValues(CanonCr2::propertyColorBalance);
	int whiteBalanceOffset = -1;
	double whiteBalanceNeutral[3] = { 1.0, 1.0, 1.0 };

	QList<QVariant> defaultWhiteBalanceNeutral5200KValues = resource->queryValues(CanonCr2::propertyDefaultWhiteBalanceNeutral5200K);
	int whiteBalanceNeutral5200KOffset = -1;
	double cameraCalibration[3] = { 1.0, 1.0, 1.0 };

	if(colorBalanceValues.size() == 582) {
		whiteBalanceOffset = 25;
		whiteBalanceNeutral5200KOffset = 35;
	}

	if(colorBalanceValues.size() == 653) {
		whiteBalanceOffset = 24;
		whiteBalanceNeutral5200KOffset = 39;
	}

	if(colorBalanceValues.size() == 674 || colorBalanceValues.size() == 692 || colorBalanceValues.size() == 702 || colorBalanceValues.size() == 1227) {
		whiteBalanceOffset = 63;
		whiteBalanceNeutral5200KOffset = 87;
	}

	if(colorBalanceValues.size() == 796) {
		whiteBalanceOffset = 63;
		whiteBalanceNeutral5200KOffset = 78;
	}

	if(whiteBalanceOffset > 0) {
		double green = (colorBalanceValues.at(whiteBalanceOffset+1).toDouble() + colorBalanceValues.at(whiteBalanceOffset+2).toDouble()) / 2.0;

		whiteBalanceNeutral[0] = green / colorBalanceValues.at(whiteBalanceOffset).toDouble();
		whiteBalanceNeutral[2] = green / colorBalanceValues.at(whiteBalanceOffset+3).toDouble();
	}

	MetadataListProperty* whiteBalanceNeutralProperty = new MetadataListProperty(resource, RawFile::propertyWhiteBalanceNeutral);
	new MetadataListItem(whiteBalanceNeutralProperty, QVariant(whiteBalanceNeutral[0]));
	new MetadataListItem(whiteBalanceNeutralProperty, QVariant(whiteBalanceNeutral[1]));
	new MetadataListItem(whiteBalanceNeutralProperty, QVariant(whiteBalanceNeutral[2]));

	if(!defaultWhiteBalanceNeutral5200KValues.isEmpty()) {
		cameraCalibration[0] = defaultWhiteBalanceNeutral5200KValues.at(0).toDouble() / colorBalanceValues.at(whiteBalanceNeutral5200KOffset).toDouble();
		cameraCalibration[1] = defaultWhiteBalanceNeutral5200KValues.at(1).toDouble() / (colorBalanceValues.at(whiteBalanceNeutral5200KOffset+1).toDouble() + colorBalanceValues.at(whiteBalanceNeutral5200KOffset+2).toDouble()) * 2;
		cameraCalibration[2] = defaultWhiteBalanceNeutral5200KValues.at(2).toDouble() / colorBalanceValues.at(whiteBalanceNeutral5200KOffset+3).toDouble();
	}

	MetadataListProperty* cameraCalibration1Property = new MetadataListProperty(resource, RawFile::propertyCameraCalibration1);
	new MetadataListItem(cameraCalibration1Property, QVariant(cameraCalibration[0]));
	new MetadataListItem(cameraCalibration1Property, QVariant(cameraCalibration[1]));
	new MetadataListItem(cameraCalibration1Property, QVariant(cameraCalibration[2]));

	MetadataListProperty* cameraCalibration2Property = new MetadataListProperty(resource, RawFile::propertyCameraCalibration2);
	new MetadataListItem(cameraCalibration2Property, QVariant(cameraCalibration[0]));
	new MetadataListItem(cameraCalibration2Property, QVariant(cameraCalibration[1]));
	new MetadataListItem(cameraCalibration2Property, QVariant(cameraCalibration[2]));

	MetadataComplexProperty* settingsDefault = new MetadataComplexProperty(resource, RawExposure::propertySettings);
	new MetadataSimpleProperty(settingsDefault, RawExposure::propertyName, QVariant("default"));

	new MetadataSimpleProperty(settingsDefault, RawExposure::propertyWhiteBalance, QVariant("asShot"));

	double temperature, tint;
	TemperatureUtils::fromxy(RawUtils::cameraNeutralWhiteBalance(resource), &temperature, &tint);

	new MetadataSimpleProperty(settingsDefault, RawExposure::propertyTemperature, QVariant((int)temperature));
	new MetadataSimpleProperty(settingsDefault, RawExposure::propertyTint, QVariant((int)tint));

	MetadataComplexProperty* settingsCurrent = settingsDefault->clone(resource)->toComplexProperty();
	settingsCurrent->queryProperty(RawExposure::propertyName)->toSimpleProperty()->setValue(QVariant("current"));

	return source;
}
