/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "ColorMath.h"

#include <QStringList>

#include <stdio.h>
#include <math.h>

Matrix3x3::Matrix3x3() {
	for(int i=0; i<3; i++)
		for(int j=0; j<3; j++)
			_c[j][i] = i == j ? 1.0 : 0.0;
}

Matrix3x3::Matrix3x3(double* c) {
	for(int i=0; i<9; i++)
		_c[i/3][i%3] = c[i];
}

Matrix3x3::Matrix3x3(double c0, double c1, double c2) {
	for(int i=0; i<9; i++)
		_c[i/3][i%3] = 0.0;

	_c[0][0] = c0;
	_c[1][1] = c1;
	_c[2][2] = c2;
}

Matrix3x3::Matrix3x3(QList<double> list) {
	if(list.size() == 9) {
		for(int i=0; i<9; i++)
			_c[i/3][i%3] = list.at(i);
	} else {
		for(int i=0; i<3; i++)
			for(int j=0; j<3; j++)
				_c[j][i] = i == j ? 1.0 : 0.0;

		if(list.size() == 3) {
			_c[0][0] = list.at(0);
			_c[1][1] = list.at(1);
			_c[2][2] = list.at(2);
		}
	}
}

Matrix3x3::~Matrix3x3() {
}

Matrix3x3 Matrix3x3::operator+(Matrix3x3 m) {
	Matrix3x3 o;

	for(int i=0; i<3; i++)
		for(int j=0; j<3; j++)
			o._c[i][j] = _c[i][j] + m._c[i][j];

	return o;
}

Matrix3x3 Matrix3x3::operator*(Matrix3x3 m) {
	Matrix3x3 o;

	for(int i=0; i<3; i++) {
		double a = _c[i][0];
		double b = _c[i][1];
		double c = _c[i][2];

		for(int j=0; j<3; j++) {
			o._c[i][j]  = a * m._c[0][j];
			o._c[i][j] += b * m._c[1][j];
			o._c[i][j] += c * m._c[2][j];
		}
	}

	return o;
}

void Matrix3x3::debug() const {
	for(int i=0; i<3; i++) {
		for(int j=0; j<3; j++)
			printf("%f ", _c[i][j]);

		printf("\n");
	}

	printf("\n");
}

Matrix3x3 Matrix3x3::inverse() {
	Matrix3x3 t, o;
	double w[3][6];

	for(int i=0; i<3; i++) {
		for(int j=0; j<7; j++)
			w[i][j] = j == i+3 ? 1.0 : 0.0;

		for(int j=0; j<3; j++)
			for(int k=0; k<3; k++)
				w[i][j] += _c[k][i] * _c[k][j];
	}

	for(int i=0; i<3; i++) {
		double n = w[i][i];

		for(int j=0; j<6; j++)
			w[i][j] /= n;

		for(int k=0; k<3; k++) {
			if(k == i)
				continue;

			double n = w[k][i];

			for(int j=0; j<6; j++)
				w[k][j] -= w[i][j] * n;
		}
	}

	for(int i=0; i<3; i++) {
		for(int j=0; j<3; j++) {
			t._c[i][j] = 0.0;

			for(int k=0; k<3; k++)
				t._c[i][j] += w[j][k+3] * _c[i][k];
		}
	}

	for(int i=0; i<3; i++)
		for(int j=0; j<3; j++)
			o._c[i][j] = t._c[j][i];

	return o;
}

double Matrix3x3::coeff(int index0, int index1) const {
	return _c[index0][index1];
}

QList<double> Matrix3x3::toList() const {
	QList<double> list;

	for(int i=0; i<3; i++)
		for(int j=0; j<3; j++)
			list.append(_c[i][j]);

	return list;
}

Matrix3x3 CIEUtils::fromProPhotoRgb2XYZ() {
	double values[] = {
		 0.797675, 0.135192, 0.0313534,
		 0.288040, 0.711874,  0.000086,
		      0.0,      0.0,  0.825210
	};

	return Matrix3x3(values);
}

Matrix3x3 CIEUtils::fromXYZ2ProPhotoRgb() {
	double values[] = {
		     1.34594, -0.255608, -0.0511118,
		   -0.544599,   1.50817,  0.0205351,
		         0.0,       0.0,    1.21181
	};

	return Matrix3x3(values);
}

Matrix3x3 CIEUtils::fromsRgb2XYZ() {
	double values[] = {
		 0.412424, 0.357579,  0.180464,
		 0.212656, 0.715158, 0.0721856,
		0.0193324, 0.119193,  0.950444
	};

	return Matrix3x3(values);
}

Matrix3x3 CIEUtils::fromXYZ2sRGB() {
	double values[] = {
		    3.24071,  -1.53726, -0.498571,
		  -0.969258,   1.87599, 0.0415557,
		  0.0556352, -0.203996,   1.05707
	};

	return Matrix3x3(values);
}

Matrix3x3 CIEUtils::cameraMatrix(Matrix3x3 c1, double t1, Matrix3x3 c2, double t2, double t) {
	if(t <= t1) {
		return c1;
	} else if(t >= t2) {
		return c2;
	} else {
		double weight = ((1.0 / t) - (1.0 / t2)) / ((1.0 / t1) - (1.0 / t2));
		double weight1 = 1.0 - weight;

		return c1 * Matrix3x3(weight,weight,weight) + c2 * Matrix3x3(weight1,weight1,weight1);
	}
}

Matrix3x3 CIEUtils::whitePointXYZ2WhitePointXYZ(ColorXYZ sWhitePoint, ColorXYZ tWhitePoint) {
	double bradfordValues[] = {
		 0.8951,  0.2664, -0.1614,
		-0.7502,  1.7135,  0.0367,
		 0.0389, -0.0685,  1.0296
	};

	Matrix3x3 bradfordMatrix(bradfordValues);

	ColorXYZ sxyz = sWhitePoint * bradfordMatrix;
	ColorXYZ txyz = tWhitePoint * bradfordMatrix;

	return bradfordMatrix.inverse() * Matrix3x3(txyz.x()/sxyz.x(), txyz.y()/sxyz.y(), txyz.z()/sxyz.z()) * bradfordMatrix;
}

const TemperatureUtils::ruvt TemperatureUtils::ruvtTable[] = {
	{   0.0, 0.18006, 0.26352, -0.24341 },
	{  10.0, 0.18066, 0.26589, -0.25479 },
	{  20.0, 0.18133, 0.26846, -0.26876 },
	{  30.0, 0.18208, 0.27119, -0.28539 },
	{  40.0, 0.18293, 0.27407, -0.30470 },
	{  50.0, 0.18388, 0.27709, -0.32675 },
	{  60.0, 0.18494, 0.28021, -0.35156 },
	{  70.0, 0.18611, 0.28342, -0.37915 },
	{  80.0, 0.18740, 0.28668, -0.40955 },
	{  90.0, 0.18880, 0.28997, -0.44278 },
	{ 100.0, 0.19032, 0.29326, -0.47888 },
	{ 125.0, 0.19462, 0.30141, -0.58204 },
	{ 150.0, 0.19962, 0.30921, -0.70471 },
	{ 175.0, 0.20525, 0.31647, -0.84901 },
	{ 200.0, 0.21142, 0.32312, -1.01820 },
	{ 225.0, 0.21807, 0.32909, -1.21680 },
	{ 250.0, 0.22511, 0.33439, -1.45120 },
	{ 275.0, 0.23247, 0.33904, -1.72980 },
	{ 300.0, 0.24010, 0.34308, -2.06370 },
	{ 325.0, 0.24792, 0.34655, -2.46810 },
	{ 350.0, 0.25591, 0.34951, -2.96410 },
	{ 375.0, 0.26400, 0.35200, -3.58140 },
	{ 400.0, 0.27218, 0.35407, -4.36330 },
	{ 425.0, 0.28039, 0.35577, -5.37620 },
	{ 450.0, 0.28863, 0.35714, -6.72620 },
	{ 475.0, 0.29685, 0.35823, -8.59550 },
	{ 500.0, 0.30505, 0.35907, -11.3240 },
	{ 525.0, 0.31320, 0.35968, -15.6280 },
	{ 550.0, 0.32129, 0.36011, -23.3250 },
	{ 575.0, 0.32931, 0.36038, -40.7700 },
	{ 600.0, 0.33724, 0.36051, -116.450 }
};

void TemperatureUtils::fromxy(Colorxy xy, double* temperature, double* tint) {
	double us = 2.0 * xy.x() / (1.5 - xy.x() + 6.0 * xy.y());
	double vs = 3.0 * xy.y() / (1.5 - xy.x() + 6.0 * xy.y());

	double di = 0.0;
	double dj = 0.0;

	int index;

	for(index=0; index<31; index++) {
		di = (vs-ruvtTable[index].v) - ruvtTable[index].t*(us-ruvtTable[index].u);

		if(index > 0 && di < 0.0)
			break;

		dj = di;
	}

	di /= sqrt(1.0 + ruvtTable[index].t * ruvtTable[index].t);
	dj /= sqrt(1.0 + ruvtTable[index-1].t * ruvtTable[index-1].t);

	double f = dj / (dj - di);

	*temperature = 1000000.0 / ((ruvtTable[index].r - ruvtTable[index-1].r) * f + ruvtTable[index-1].r);

	double ud = us - ((ruvtTable[index].u - ruvtTable[index-1].u) * f + ruvtTable[index-1].u);
	double vd = vs - ((ruvtTable[index].v - ruvtTable[index-1].v) * f + ruvtTable[index-1].v);

	double tli = sqrt(1.0 + ruvtTable[index].t * ruvtTable[index].t);
	double tui = 1.0 / tli;
	double tvi = ruvtTable[index].t / tli;

	double tlj = sqrt(1.0 + ruvtTable[index-1].t * ruvtTable[index-1].t);
	double tuj = 1.0 / tlj;
	double tvj = ruvtTable[index-1].t / tlj;

	double tu = (tui - tuj) * f + tuj;
	double tv = (tvi - tvj) * f + tvj;
	double tl = sqrt(tu * tu + tv * tv);

	tu /= tl;
	tv /= tl;

	*tint = (ud * tu + vd * tv) * -3000.0;
}

Colorxy TemperatureUtils::toxy(double temperature, double tint) {
	double r = 1000000.0 / temperature;

	int index;

	for(index=1; index<31; index++) {
		if(r < ruvtTable[index].r)
			break;
	}

	double f = (ruvtTable[index].r - r) / (ruvtTable[index].r - ruvtTable[index-1].r);

	double us = (ruvtTable[index-1].u - ruvtTable[index].u) * f + ruvtTable[index].u;
	double vs = (ruvtTable[index-1].v - ruvtTable[index].v) * f + ruvtTable[index].v;

	double tli = sqrt(1.0 + ruvtTable[index].t * ruvtTable[index].t);
	double tui = 1.0 / tli;
	double tvi = ruvtTable[index].t / tli;

	double tlj = sqrt(1.0 + ruvtTable[index-1].t * ruvtTable[index-1].t);
	double tuj = 1.0 / tlj;
	double tvj = ruvtTable[index-1].t / tlj;

	double tu = (tuj - tui) * f + tui;
	double tv = (tvj - tvi) * f + tvi;
	double tl = sqrt(tu * tu + tv * tv);

	tu /= tl;
	tv /= tl;
	
	us += tu * tint / -3000.0;
	vs += tv * tint / -3000.0;

	return Colorxy(1.5 * us / (us - 4.0 * vs + 2.0), vs / (us - 4.0 * vs + 2.0));
}

ColorXYZ::ColorXYZ() {
	_x = _y = _z = 0.0;
}

ColorXYZ::ColorXYZ(double x, double y, double z) {
	_x = x;
	_y = y;
	_z = z;
}

ColorXYZ::ColorXYZ(QList<double> l) {
	if(l.size() == 3) {
		_x = l.at(0);
		_y = l.at(1);
		_z = l.at(2);
	} else {
		_x = _y = _z = 0.0;
	}
}

ColorXYZ ColorXYZ::operator*(Matrix3x3 matrix) {
	double v[3] = { _x, _y, _z };

	matrix.multiplyVector(v);

	return ColorXYZ(v[0], v[1], v[2]);
}

double ColorXYZ::x() {
	return _x;
}

double ColorXYZ::y() {
	return _y;
}

double ColorXYZ::z() {
	return _z;
}

ColorXYZ ColorXYZ::inverse() {
	return ColorXYZ(1.0/_x, 1.0/_y, 1.0/_z);
}

Colorxy ColorXYZ::toColorxy() {
	double s = _x + _y + _z;

	return Colorxy(_x/s, _y/s);
}

Matrix3x3 ColorXYZ::toMatrix3x3() {
	return Matrix3x3(_x, _y, _z);
}

void ColorXYZ::debug() {
	printf("X: %7.5f Y: %7.5f Z: %7.5f\n", _x, _y, _z);
}

Colorxy::Colorxy() {
	_x = _y = 0.0;
}

Colorxy::Colorxy(double x, double y) {
	_x = x;
	_y = y;
}

double Colorxy::x() {
	return _x;
}

double Colorxy::y() {
	return _y;
}

ColorXYZ Colorxy::toColorXYZ() {
	return ColorXYZ(_x/_y, 1.0, (1.0-_x-_y)/_y);
}

void Colorxy::debug() {
	printf("x: %7.5f y: %7.5f\n", _x, _y);
}

Colorxy Colorxy::whitePointA() {
	return Colorxy(0.44757, 0.40745);
}

Colorxy Colorxy::whitePointB() {
	return Colorxy(0.34842, 0.35161);
}

Colorxy Colorxy::whitePointC() {
	return Colorxy(0.31006, 0.31616);
}

Colorxy Colorxy::whitePointD50() {
	return Colorxy(0.34567, 0.35850);
}

Colorxy Colorxy::whitePointD55() {
	return Colorxy(0.33242, 0.34743);
}

Colorxy Colorxy::whitePointD65() {
	return Colorxy(0.31271, 0.32902);
}

Colorxy Colorxy::whitePointD75() {
	return Colorxy(0.29902, 0.31485);
}

Spline::Spline() {
	solve();
}

Spline::Spline(QString values) {
	QList<QPointF> list;
	double max = 0.0;
	QStringList valuePairs = values.split(';');

	if(valuePairs.size() > 1) {
		foreach(QString valuePair, valuePairs) {
			QStringList valueList = valuePair.split(',');
			QPointF point = QPointF(valueList.at(0).toDouble(), valueList.at(1).toDouble());
			max = point.x();
			list.append(point);
		}
	}

	double scale = 1.0;
	if(max > 1.0)
		scale = 255.0;
	if(max > 255.0)
		scale = 65535.0;

	setPoints(list, scale);
}

Spline::Spline(QList<QPointF> list, double scale) {
	setPoints(list, scale);
}

Spline::~Spline() {
}

double Spline::evaluate(double x) {
	int size = _p.size();

	if (x <= _p.at(0).x())
		return _p.at(0).y();

	if (x >= _p.at(size-1).x())
		return _p.at(size-1).y();

	int low = 1;
	int high = size - 1;

	while(high > low) {
		int mid = (low + high) >> 1;

		if (x == _p.at(mid).x())
			return _p.at(mid).y();

		if (x > _p.at(mid).x())
			low = mid + 1;
		else
			high = mid;
	}

	return evaluateSegment(x, _p.at(low-1), _s.at(low-1), _p.at(low), _s.at(low));
}

int Spline::addPoint(QPointF p) {
	if(containsX(p.x()))
		return -1;

	p.setX(qBound(0.0, p.x(), 1.0));
	p.setY(qBound(0.0, p.y(), 1.0));

	int index = 0;

	while(p.x() > _p.at(index).x() && index < _p.size())
		index++;

	_p.insert(index, p);
	solve();

	return index;
}

void Spline::movePoint(QPointF p, int index) {
	if(containsX(p.x()))
		return;

	p.setX(qBound(0.0, p.x(), 1.0));
	p.setY(qBound(0.0, p.y(), 1.0));

	if(index > 0 && _p.at(index-1).x() >= p.x())
		return;

	if(index < _p.size()-1 && _p.at(index+1).x() <= p.x())
		return;

	_p[index] = p;

	solve();
}

void Spline::removePoint(int index) {
	_p.remove(index);

	solve();
}

QList<QPointF> Spline::points() {
	return _p.toList();
}

void Spline::setPoints(QList<QPointF> list, double scale) {
	_p.clear();

	foreach(QPointF p, list) {
		_p.append(p / scale);
	}

	solve();
}

bool Spline::containsX(double x) {
	for(int i=0; i<_p.size(); i++)
		if(_p.at(i).x() == x)
			return true;

	return false;
}

void Spline::solve() {
	int size = _p.size();

	if(size < 2) {
		_p.clear();
		_p.append(QPointF(0.0, 0.0));
		_p.append(QPointF(1.0, 1.0));
		size = 2;
	}

	double a = _p.at(1).x() - _p.at(0).x();
	double b = (_p.at(1).y() - _p.at(0).y()) / a;

	_s.resize(size);

	_s[0] = b;

	for(int i=2; i<size; i++) {
		double c = _p.at(i).x() - _p.at(i-1).x();
		double d = (_p.at(i).y() - _p.at(i-1).y()) / c;

		_s[i-1] = (b * c + d * a) / (a + c);

		a = c;
		b = d;
	}

	_s[size-1] = 2.0 * b - _s[size-2];
	_s[0] = 2.0 * _s[0] - _s[1];

	if(size > 2) {
		QVector<double> e(size);
		QVector<double> f(size);
		QVector<double> g(size);

		f[0] = 0.5;
		e[size-1] = 0.5;
		g[0] = 0.75 * (_s[0] + _s[1]);
		g[size-1] = 0.75 * (_s[size-2] + _s[size-1]);

		for(int i=1; i<size-1; i++) {
			double h = (_p.at(i+1).x() - _p.at(i-1).x()) * 2.0;

			e[i] = (_p.at(i+1).x() - _p.at(i).x()) / h;
			f[i] = (_p.at(i).x() - _p.at(i-1).x()) / h;
			g[i] = 1.5 * _s[i];
		}

		for(int i=1; i<size; i++) {
			double h = 1.0 - f[i-1] * e[i];

			if (i != size-1)
				f[i] /= h;

			g[i] = (g[i] - g[i-1] * e[i]) / h;
		}

		for(int i=size-2; i>=0; i--)
			g[i] = g[i] - f[i] * g[i+1];

		for(int i=0; i<size; i++)
			_s[i] = g[i];
	}
}

double Spline::evaluateSegment(double x, QPointF p0, double s0, QPointF p1, double s1) {
	double a = p1.x() - p0.x();
	double b = (x - p0.x()) / a;
	double c = (p1.x() - x) / a;

	return ((p0.y() * (2.0 - c + b) + (s0 * a * b)) * (c * c)) + ((p1.y() * (2.0 - b + c) - (s1 * a * c)) * (b * b));
}
