/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "DNG.h"

#include <QBuffer>

#include <LosslessJpegDecoder.h>
#include <MetadataRdf.h>
#include <QtUtils.h>

QString DNG::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/dng/1.0/";
MetadataQName DNG::propertyUniqueCameraModel = MetadataQName(DNG::namespaceUri, "UniqueCameraModel");
MetadataQName DNG::propertyLocalizedCameraModel = MetadataQName(DNG::namespaceUri, "LocalizedCameraModel");
MetadataQName DNG::propertyCFALayout = MetadataQName(DNG::namespaceUri, "CFALayout");
MetadataQName DNG::propertyBlackLevel = MetadataQName(DNG::namespaceUri, "BlackLevel");
MetadataQName DNG::propertyWhiteLevel = MetadataQName(DNG::namespaceUri, "WhiteLevel");
MetadataQName DNG::propertyBaselineExposure = MetadataQName(DNG::namespaceUri, "BaselineExposure");
MetadataQName DNG::propertyBaselineNoise = MetadataQName(DNG::namespaceUri, "BaselineNoise");
MetadataQName DNG::propertyBaselineSharpness = MetadataQName(DNG::namespaceUri, "BaselineSharpness");
MetadataQName DNG::propertyBayerGreenSplit = MetadataQName(DNG::namespaceUri, "BayerGreenSplit");
MetadataQName DNG::propertyLinearResponseLimit = MetadataQName(DNG::namespaceUri, "LinearResponseLimit");
MetadataQName DNG::propertyCameraSerialNumber = MetadataQName(DNG::namespaceUri, "CameraSerialNumber");
MetadataQName DNG::propertyAntiAliasStrength = MetadataQName(DNG::namespaceUri, "AntiAliasStrength");
MetadataQName DNG::propertyShadowScale = MetadataQName(DNG::namespaceUri, "ShadowScale");
MetadataQName DNG::propertyCalibrationIlluminant1 = MetadataQName(DNG::namespaceUri, "CalibrationIlluminant1");
MetadataQName DNG::propertyCalibrationIlluminant2 = MetadataQName(DNG::namespaceUri, "CalibrationIlluminant2");
MetadataQName DNG::propertyBestQualityScale = MetadataQName(DNG::namespaceUri, "BestQualityScale");
MetadataQName DNG::propertyOriginalRawFileName = MetadataQName(DNG::namespaceUri, "OriginalRawFileName");
MetadataQName DNG::propertyOriginalRawFileData = MetadataQName(DNG::namespaceUri, "OriginalRawFileData");
MetadataQName DNG::propertyDNGVersion = MetadataQName(DNG::namespaceUri, "DNGVersion");
MetadataQName DNG::propertyDNGBackwardVersion = MetadataQName(DNG::namespaceUri, "DNGBackwardVersion");
MetadataQName DNG::propertyCFAPlaneColor = MetadataQName(DNG::namespaceUri, "CFAPlaneColor");
MetadataQName DNG::propertyBlackLevelRepeatDim = MetadataQName(DNG::namespaceUri, "BlackLevelRepeatDim");
MetadataQName DNG::propertyBlackLevelDeltaV = MetadataQName(DNG::namespaceUri, "BlackLevelDeltaV");
MetadataQName DNG::propertyDefaultScale = MetadataQName(DNG::namespaceUri, "DefaultScale");
MetadataQName DNG::propertyDefaultCropOrigin = MetadataQName(DNG::namespaceUri, "DefaultCropOrigin");
MetadataQName DNG::propertyDefaultCropSize = MetadataQName(DNG::namespaceUri, "DefaultCropSize");
MetadataQName DNG::propertyColorMatrix1 = MetadataQName(DNG::namespaceUri, "ColorMatrix1");
MetadataQName DNG::propertyColorMatrix2 = MetadataQName(DNG::namespaceUri, "ColorMatrix2");
MetadataQName DNG::propertyCameraCalibration1 = MetadataQName(DNG::namespaceUri, "CameraCalibration1");
MetadataQName DNG::propertyCameraCalibration2 = MetadataQName(DNG::namespaceUri, "CameraCalibration2");
MetadataQName DNG::propertyAnalogBalance = MetadataQName(DNG::namespaceUri, "AnalogBalance");
MetadataQName DNG::propertyAsShotNeutral = MetadataQName(DNG::namespaceUri, "AsShotNeutral");
MetadataQName DNG::propertyLensInfo = MetadataQName(DNG::namespaceUri, "LensInfo");
MetadataQName DNG::propertyDNGPrivateData = MetadataQName(DNG::namespaceUri, "DNGPrivateData");
MetadataQName DNG::propertyRawDataUniqueID = MetadataQName(DNG::namespaceUri, "RawDataUniqueID");
MetadataQName DNG::propertyActiveArea = MetadataQName(DNG::namespaceUri, "ActiveArea");
MetadataQName DNG::propertyMaskedAreas = MetadataQName(DNG::namespaceUri, "MaskedAreas");

QMap<int, QString> DNGParser::_tagNameMap = DNGParser::createTagNameMap();

DNGParser::DNGParser(QIODevice* device) : TiffParser(device) {
}

MetadataQName DNGParser::tagQName(quint16 tag) {
	if(_tagNameMap.contains(tag))
		return MetadataQName(DNG::namespaceUri, _tagNameMap.value(tag));
	else
		return TiffParser::tagQName(tag);
}

QMap<int, QString> DNGParser::createTagNameMap() {
	QMap<int, QString> tagNameMap;

	tagNameMap.insert(0xc614, DNG::propertyUniqueCameraModel.localName());
	tagNameMap.insert(0xc615, DNG::propertyLocalizedCameraModel.localName());
	tagNameMap.insert(0xc617, DNG::propertyCFALayout.localName());
	tagNameMap.insert(0xc61a, DNG::propertyBlackLevel.localName());
	tagNameMap.insert(0xc61d, DNG::propertyWhiteLevel.localName());
	tagNameMap.insert(0xc62a, DNG::propertyBaselineExposure.localName());
	tagNameMap.insert(0xc62b, DNG::propertyBaselineNoise.localName());
	tagNameMap.insert(0xc62c, DNG::propertyBaselineSharpness.localName());
	tagNameMap.insert(0xc62d, DNG::propertyBayerGreenSplit.localName());
	tagNameMap.insert(0xc62e, DNG::propertyLinearResponseLimit.localName());
	tagNameMap.insert(0xc62f, DNG::propertyCameraSerialNumber.localName());
	tagNameMap.insert(0xc632, DNG::propertyAntiAliasStrength.localName());
	tagNameMap.insert(0xc633, DNG::propertyShadowScale.localName());
	tagNameMap.insert(0xc65a, DNG::propertyCalibrationIlluminant1.localName());
	tagNameMap.insert(0xc65b, DNG::propertyCalibrationIlluminant2.localName());
	tagNameMap.insert(0xc65c, DNG::propertyBestQualityScale.localName());
	tagNameMap.insert(0xc68b, DNG::propertyOriginalRawFileName.localName());
	tagNameMap.insert(0xc68c, DNG::propertyOriginalRawFileData.localName());
	tagNameMap.insert(0xc612, DNG::propertyDNGVersion.localName());
	tagNameMap.insert(0xc613, DNG::propertyDNGBackwardVersion.localName());
	tagNameMap.insert(0xc616, DNG::propertyCFAPlaneColor.localName());
	tagNameMap.insert(0xc619, DNG::propertyBlackLevelRepeatDim.localName());
	tagNameMap.insert(0xc61c, DNG::propertyBlackLevelDeltaV.localName());
	tagNameMap.insert(0xc61e, DNG::propertyDefaultScale.localName());
	tagNameMap.insert(0xc61f, DNG::propertyDefaultCropOrigin.localName());
	tagNameMap.insert(0xc620, DNG::propertyDefaultCropSize.localName());
	tagNameMap.insert(0xc621, DNG::propertyColorMatrix1.localName());
	tagNameMap.insert(0xc622, DNG::propertyColorMatrix2.localName());
	tagNameMap.insert(0xc623, DNG::propertyCameraCalibration1.localName());
	tagNameMap.insert(0xc624, DNG::propertyCameraCalibration2.localName());
	tagNameMap.insert(0xc627, DNG::propertyAnalogBalance.localName());
	tagNameMap.insert(0xc628, DNG::propertyAsShotNeutral.localName());
	tagNameMap.insert(0xc630, DNG::propertyLensInfo.localName());
	tagNameMap.insert(0xc634, DNG::propertyDNGPrivateData.localName());
	tagNameMap.insert(0xc65d, DNG::propertyRawDataUniqueID.localName());
	tagNameMap.insert(0xc68d, DNG::propertyActiveArea.localName());
	tagNameMap.insert(0xc68e, DNG::propertyMaskedAreas.localName());

	return tagNameMap;
}

DNGFile::DNGFile(QIODevice* device) : RawFile() {
	_device = device;
	_dngParser = new DNGParser(_device);
}

DNGFile::~DNGFile() {
	delete _dngParser;
}

MetadataDocument* DNGFile::readMetadata() {
	if(_metadata == 0) {
		DNGMetadataCleanupTransform transform;

		_metadata = transform.transform(_dngParser->parse())->toDocument();
	}

	return _metadata;
}

Image* DNGFile::readImage() {
	if(_image == 0) {
		readMetadata();

		MetadataResource* resource = _metadata->queryResource("*");

		int width = resource->queryValue(Tiff::propertyImageWidth).toInt();
		int height = resource->queryValue(Tiff::propertyImageLength).toInt();

		if(resource->queryProperty(Tiff::propertyTileWidth) != 0) {
			quint32 tileWidth = resource->queryValue(Tiff::propertyTileWidth).toUInt();
			quint32 tilesPerLine = (width+tileWidth-1) / tileWidth;
			quint32 fullWidth = tilesPerLine * tileWidth;
			quint32 tileLength = resource->queryValue(Tiff::propertyTileLength).toUInt();
			quint32 tilesPerRow = (height+tileLength-1) / tileLength;
			quint32 fullHeight = tilesPerRow * tileLength;
			QList<quint32> tileOffsets = QtUtils::toTypedList<quint32>(resource->queryValues(Tiff::propertyTileOffsets));
			QList<quint32> tileByteCounts = QtUtils::toTypedList<quint32>(resource->queryValues(Tiff::propertyTileByteCounts));

			BayerImage16* bayerImage = new BayerImage16(fullWidth, fullHeight);
			bayerImage->setMetadata(resource);

			// TODO: move pattern to metadata
			QVector<int> pattern;
			foreach(int c, QtUtils::toTypedList<quint32>(resource->queryValues(Tiff::propertyCFAPattern))) {
				pattern.append(c);
			}
			bayerImage->setPattern(pattern);

			for(int i=0; i<tileOffsets.size(); i++) {
				quint32 offset = tileOffsets.at(i);
				quint32 byteCounts = tileByteCounts.at(i);

				_device->seek(offset);
				QByteArray data = _device->read(byteCounts);
				QBuffer buffer(&data);
				buffer.open(QIODevice::ReadOnly);
				LosslessJpegDecoder losslessJpegDecoder(&buffer);
				losslessJpegDecoder.decodeHeader();

				quint16* imageOffset = bayerImage->dataLine((i/tilesPerLine)*tileLength) + (i%tilesPerLine)*tileWidth;
				losslessJpegDecoder.decodeImage(imageOffset, fullWidth);
				buffer.close();
			}

			_image = bayerImage;
		} else {
			quint32 dataOffset = resource->queryValue(Tiff::propertyStripOffsets).toInt();

			BayerImage16* bayerImage = new BayerImage16(width, height);
			bayerImage->setMetadata(resource);

			// TODO: move pattern to metadata
			QVector<int> pattern;
			foreach(int c, QtUtils::toTypedList<quint32>(resource->queryValues(Tiff::propertyCFAPattern))) {
				pattern.append(c);
			}
			bayerImage->setPattern(pattern);

			_device->seek(dataOffset);
			LosslessJpegDecoder losslessJpegDecoder(_device);
			losslessJpegDecoder.decodeHeader();
			losslessJpegDecoder.decodeImage(bayerImage->data());

			_image = bayerImage;
		}
	}

	return _image;
}

DNGMetadataCleanupTransform::~DNGMetadataCleanupTransform() {
}

MetadataNode* DNGMetadataCleanupTransform::transform(MetadataNode* source) {
	int imageNumber = 0;

	MetadataResource* resource = source->queryResource("*");
	resource->setResourceUri(QString());

	// remove root properties which will be replaced by sub ifd
	resource->removeChild(resource->queryProperty(Tiff::propertyImageWidth));
	resource->removeChild(resource->queryProperty(Tiff::propertyImageLength));
	resource->removeChild(resource->queryProperty(Tiff::propertyBitsPerSample));
	resource->removeChild(resource->queryProperty(Tiff::propertyCompression));
	resource->removeChild(resource->queryProperty(Tiff::propertyPhotometricInterpretation));
	resource->removeChild(resource->queryProperty(Tiff::propertyStripOffsets));
	resource->removeChild(resource->queryProperty(Tiff::propertySamplesPerPixel));
	resource->removeChild(resource->queryProperty(Tiff::propertyStripByteCounts));
	resource->removeChild(resource->queryProperty(Tiff::propertyPlanarConfiguration));

	// reassign properties from sub ifd
	MetadataProperty* subIfd = resource->queryProperties(Tiff::propertySubIFDs).at(imageNumber);

	foreach(MetadataNode* node, subIfd->children()) {
		node->setParent(resource);
	}

	resource->removeChild(subIfd);

	// body id
	//new MetadataSimpleProperty(resource, RawFile::propertyBodyId, resource->queryValue(CanonCr2::propertyBodyInfo));

	// sensor crop
	QList<quint32> activeArea = QtUtils::toTypedList<quint32>(resource->queryValues(DNG::propertyActiveArea));
	QList<Rational> cropOrigin = QtUtils::toTypedList<Rational>(resource->queryValues(DNG::propertyDefaultCropOrigin));
	QList<Rational> cropSize = QtUtils::toTypedList<Rational>(resource->queryValues(DNG::propertyDefaultCropSize));
	quint32 imageWidth = resource->queryValue(Tiff::propertyImageWidth).toUInt();
	quint32 imageLength = resource->queryValue(Tiff::propertyImageLength).toUInt();

	// TODO: use only crop if there is no active area
	MetadataListProperty* cropProperty = new MetadataListProperty(resource, RawFile::propertySensorCrop);
	if(!activeArea.empty() || !cropOrigin.empty() || !cropSize.empty()) {
		new MetadataListItem(cropProperty, activeArea.at(1)+(int)cropOrigin.at(0).toDouble());
		new MetadataListItem(cropProperty, activeArea.at(0)+(int)cropOrigin.at(1).toDouble());
		new MetadataListItem(cropProperty, (int)cropSize.at(0).toDouble());
		new MetadataListItem(cropProperty, (int)cropSize.at(1).toDouble());
	} else {
		new MetadataListItem(cropProperty, 0);
		new MetadataListItem(cropProperty, 0);
		new MetadataListItem(cropProperty, imageWidth);
		new MetadataListItem(cropProperty, imageLength);
	}

	// white level 
	new MetadataSimpleProperty(resource, RawFile::propertyWhiteLevel, resource->queryValue(DNG::propertyWhiteLevel));

	// black level
	// TODO: use all black level infos from dng
	QList<Rational> blackLevelDeltaV = QtUtils::toTypedList<Rational>(resource->queryValues(DNG::propertyBlackLevelDeltaV));
	double blackLevelDeltaVSum = 0.0;
	if(!blackLevelDeltaV.isEmpty()) {
		foreach(Rational item, blackLevelDeltaV) {
			blackLevelDeltaVSum += item.toDouble();
		}
		blackLevelDeltaVSum /= (double)blackLevelDeltaV.size();
	}
	new MetadataSimpleProperty(resource, RawFile::propertyBlackLevel, resource->queryValue(DNG::propertyBlackLevel).toDouble() + blackLevelDeltaVSum);

	// white balance neutral	
	MetadataListProperty* whiteBalanceNeutralProperty = new MetadataListProperty(resource, RawFile::propertyWhiteBalanceNeutral);
	foreach(Rational value, QtUtils::toTypedList<Rational>(resource->queryValues(DNG::propertyAsShotNeutral))) {
		new MetadataListItem(whiteBalanceNeutralProperty, value.toDouble());
	}

	// TODO: translate from dng
	// calibration illuminant
	new MetadataSimpleProperty(resource, RawFile::propertyCalibrationIlluminant1, 2856);
	new MetadataSimpleProperty(resource, RawFile::propertyCalibrationIlluminant2, 5503);
	
	// color matrix
	MetadataListProperty* colorMatrix1Property = new MetadataListProperty(resource, RawFile::propertyColorMatrix1);
	foreach(Rational value, QtUtils::toTypedList<Rational>(resource->queryValues(DNG::propertyColorMatrix1))) {
		new MetadataListItem(colorMatrix1Property, value.toDouble());
	}

	MetadataListProperty* colorMatrix2Property = new MetadataListProperty(resource, RawFile::propertyColorMatrix2);
	foreach(Rational value, QtUtils::toTypedList<Rational>(resource->queryValues(DNG::propertyColorMatrix2))) {
		new MetadataListItem(colorMatrix2Property, value.toDouble());
	}

	// camera calibration
	MetadataListProperty* cameraCalibration1Property = new MetadataListProperty(resource, RawFile::propertyCameraCalibration1);
	foreach(Rational value, QtUtils::toTypedList<Rational>(resource->queryValues(DNG::propertyCameraCalibration1))) {
		new MetadataListItem(cameraCalibration1Property, value.toDouble());
	}
	
	MetadataListProperty* cameraCalibration2Property = new MetadataListProperty(resource, RawFile::propertyCameraCalibration2);
	foreach(Rational value, QtUtils::toTypedList<Rational>(resource->queryValues(DNG::propertyCameraCalibration2))) {
		new MetadataListItem(cameraCalibration2Property, value.toDouble());
	}	

	// settings default
	MetadataComplexProperty* settingsDefault = new MetadataComplexProperty(resource, RawExposure::propertySettings);
	new MetadataSimpleProperty(settingsDefault, RawExposure::propertyName, "default");

	new MetadataSimpleProperty(settingsDefault, RawExposure::propertyWhiteBalance, "asShot");

	double temperature, tint;
	TemperatureUtils::fromxy(RawUtils::cameraNeutralWhiteBalance(resource), &temperature, &tint);

	new MetadataSimpleProperty(settingsDefault, RawExposure::propertyTemperature, (int)temperature);
	new MetadataSimpleProperty(settingsDefault, RawExposure::propertyTint, (int)tint);

	// settings current
	MetadataComplexProperty* settingsCurrent = settingsDefault->clone(resource)->toComplexProperty();
	settingsCurrent->queryProperty(RawExposure::propertyName)->toSimpleProperty()->setValue("current");

	return source;
}
