/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "LosslessJpegDecoder.h"

#include <QFile>
#include <QTextStream>

HuffmanNode::HuffmanNode() {
	_nodes[0] = 0;
	_nodes[1] = 0;
	_hasValue = false;
	_value = -1;
}

HuffmanNode::~HuffmanNode() {
	if(_nodes[0] != 0)
		delete _nodes[0];

	if(_nodes[1] != 0)
		delete _nodes[1];
}

void HuffmanNode::setValue(int value) {
	_hasValue = true;
	_value = value;
}

HuffmanNode* HuffmanNode::getMostLeft(int level) {
	if(_hasValue)
		return 0;

	if(level == 0)
		return this;

	HuffmanNode* nextNode = 0;

	for(int i=0; i<2; i++) {
		if(_nodes[i] == 0)
			_nodes[i] = new HuffmanNode();

		if((nextNode = _nodes[i]->getMostLeft(level-1)) != 0)
			return nextNode;
	}

	return 0;
}

LosslessJpegDecoder::LosslessJpegDecoder(QIODevice* device) : _dataStream(device), _bitStream(device) {
	_dataStream.setByteOrder(QDataStream::BigEndian);

	for(int i=0; i<4; i++)
		_huffmanTrees[i] = 0;
}

LosslessJpegDecoder::~LosslessJpegDecoder() {
	for(int i=0; i<4; i++)
		if(_huffmanTrees[i] != 0)
			delete _huffmanTrees[i];
}

int LosslessJpegDecoder::precision() {
	return _precision;
}

int LosslessJpegDecoder::lines() {
	return _lines;
}

int LosslessJpegDecoder::samples() {
	return _samples;
}

int LosslessJpegDecoder::components() {
	return _components;
}

void LosslessJpegDecoder::decodeHeader() {
	bool done = false;

	quint16 marker;
	_dataStream >> marker;

	if(marker != MarkerSOI)
		return;

	do {
		quint16 length;
		_dataStream >> marker;
		_dataStream >> length;
		length -= 2;

		switch(marker) {
			case MarkerSOF3:
				decodeSOF3(length);
				break;

			case MarkerDHT:
				decodeDHT(length);
				break;

			case MarkerSOS:
				decodeSOS(length);
				done = true;
				break;

			default:
				_dataStream.skipRawData(length);
				break;
		}
	} while(!done);
}

void LosslessJpegDecoder::decodeImage(quint16* data, int stride) {
	int width = _samples*_components;

	if(stride == -1)
		stride = width;

	if(_hSamplingFactor[0] == 2) {
		int p0, p1, p2;

		p0 = p1 = p2 = 1 << (_precision - 1);

		for(int y=0; y<_lines; y++) {			


			if(y != 0) {
				p0 = data[-width];
				p1 = data[-width+2];
				p2 = data[-width+4];
			}

			for(int x=0; x<_samples; x+=2) {
				*(data++) = p0 += decodeDiff(_huffmanTreesSelected[0]);
				*(data++) = p0 += decodeDiff(_huffmanTreesSelected[0]);
				*(data++) = p1 += decodeDiff(_huffmanTreesSelected[1]);
				*(data++) = data[-1];
				*(data++) = p2 += decodeDiff(_huffmanTreesSelected[2]);
				*(data++) = data[-1];
			}
		}
	} else {
		for(int j=0; j<_components; j++)
			data[j] = decodeDiff(_huffmanTreesSelected[j]) + (1 << (_precision - 1));

		for(int x=_components; x<_samples*_components; x+=_components)
			for(int j=0; j<_components; j++)
				data[x+j] = decodeDiff(_huffmanTreesSelected[j]) + data[x+j-_components];

		data += stride;

		for(int y=1; y<_lines; y++) {
			for(int j=0; j<_components; j++)
				data[j] = decodeDiff(_huffmanTreesSelected[j]) + data[j-stride];

			for(int x=_components; x<_samples*_components; x+=_components)
				for(int j=0; j<_components; j++) {
					switch(_predictor) {
						case 1:
							data[x+j] = decodeDiff(_huffmanTreesSelected[j]) + data[x+j-_components];
							break;

						case 6:
							data[x+j] = decodeDiff(_huffmanTreesSelected[j]) + data[x+j-stride] + ((data[x+j-_components] - data[x+j-_components-stride]) >> 1);
							break;
					}
				}

			data += stride;
		}
	}
}

int LosslessJpegDecoder::buildTree() {
	int tableLength = 0;

	quint8 tableId;
	_dataStream >> tableId;
	_huffmanTrees[tableId] = new HuffmanNode();

	quint8 codeLengthList[16];
	for(int i=0; i<16; i++) {
		_dataStream >> codeLengthList[i];
		tableLength += codeLengthList[i];
	}

	for(int i=0; i<16; i++) {
		for(int j=0; j<codeLengthList[i]; j++) {
			quint8 value;
			_dataStream >> value;
			_huffmanTrees[tableId]->getMostLeft(i+1)->setValue(value);
		}
	}

	return tableLength + 17;
}

void LosslessJpegDecoder::decodeDHT(quint16 length) {
	if(length <= 0)
		return;

	do {
		length -= buildTree();
	} while(length > 0);
}

void LosslessJpegDecoder::decodeSOF3(quint16 length) {
	Q_UNUSED(length);

	_dataStream >> _precision;
	_dataStream >> _lines;
	_dataStream >> _samples;
	_dataStream >> _components;

	for(int j=0; j<_components; j++) {
		quint8 component, samplingFactor, quantizationTable;

		_dataStream >> component;
		_dataStream >> samplingFactor;
		_dataStream >> quantizationTable;

		_componentIndex[component] = j;

		_hSamplingFactor[_componentIndex[component]] = samplingFactor>>4;
		_vSamplingFactor[_componentIndex[component]] = samplingFactor&0xf;
	}

	for(int i=6+_components*3; i<length; i++) {
		quint8 data;
		_dataStream >> data;
	}
}

void LosslessJpegDecoder::decodeSOS(quint16 length) {
	quint8 components;

	_dataStream >> components;

	for(int j=0; j<components; j++) {
		quint8 component, treeSelection;

		_dataStream >> component;
		_dataStream >> treeSelection;

		_huffmanTreesSelected[_componentIndex[component]] = _huffmanTrees[treeSelection>>4];
	}

	_dataStream >> _predictor;

	for(int i=2+components*2; i<length; i++) {
		quint8 data;
		_dataStream >> data;
	}
}

int LosslessJpegDecoder::decodeDiff(HuffmanNode* node) {
	int length = node->decode(_bitStream);

	int diff = (int)_bitStream.bits(length);

	if((diff & (1 << (length-1))) == 0)
		diff -= (1 << length) - 1;

	return diff;
}
