/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "Metadata.h"

#include <QHash>

Rational::Rational() {
	_numerator = 0;
	_denominator = 0;
}

Rational::Rational(qlonglong numerator, qlonglong denominator) {
	_numerator = numerator;
	_denominator = denominator;
}

qlonglong Rational::numerator() const {
	return _numerator;
}

void Rational::setNumerator(qlonglong numerator) {
	_numerator = numerator;
}

qlonglong Rational::denominator() const {
	return _denominator;
}

void Rational::setDenominator(qlonglong denominator) {
	_denominator = denominator;
}

QString Rational::toString() const {
	return QString("%1/%2").arg(_numerator).arg(_denominator);
}

double Rational::toDouble() const {
	return (double)_numerator / (double)_denominator;
}

MetadataQName::MetadataQName() {
}

MetadataQName::MetadataQName(QString namespaceUri, QString localName) {
	_namespaceUri = namespaceUri;
	_localName = localName;
}

QString MetadataQName::namespaceUri() const {
	return _namespaceUri;
}

void MetadataQName::setNamespaceUri(QString namespaceUri) {
	_namespaceUri = namespaceUri;
}

QString MetadataQName::localName() const {
	return _localName;
}

void MetadataQName::setLocalName(QString localName) {
	_localName = localName;
}

bool MetadataQName::operator !=(const MetadataQName other) const {
	return _namespaceUri != other._namespaceUri || _localName != other._localName;
}

bool MetadataQName::operator <(const MetadataQName other) const {
	if(_namespaceUri == other._namespaceUri)
		return _localName < other._localName;
	else
		return _namespaceUri < other._namespaceUri;
}

bool MetadataQName::operator ==(const MetadataQName other) const {
	return _namespaceUri == other._namespaceUri && _localName == other._localName;
}

MetadataDocument* MetadataNode::toDocument(MetadataNode* node) {
	if(node == 0)
		return 0;
	else
		return node->toDocument();
}

MetadataResource* MetadataNode::toResource(MetadataNode* node) {
	if(node == 0)
		return 0;
	else
		return node->toResource();
}

MetadataProperty* MetadataNode::toProperty(MetadataNode* node) {
	if(node == 0)
		return 0;
	else
		return node->toProperty();
}

MetadataSimpleProperty* MetadataNode::toSimpleProperty(MetadataNode* node) {
	if(node == 0)
		return 0;
	else
		return node->toSimpleProperty();
}

MetadataListProperty* MetadataNode::toListProperty(MetadataNode* node) {
	if(node == 0)
		return 0;
	else
		return node->toListProperty();
}

MetadataListItem* MetadataNode::toListItem(MetadataNode* node) {
	if(node == 0)
		return 0;
	else
		return node->toListItem();
}

MetadataComplexProperty* MetadataNode::toComplexProperty(MetadataNode* node) {
	if(node == 0)
		return 0;
	else
		return node->toComplexProperty();
}

MetadataNode::MetadataNode(MetadataNode* parent) {
	_parent = parent;

	if(_parent != 0)
		_parent->_children.append(this);
}

MetadataNode::~MetadataNode() {
	foreach(MetadataNode* node, _children) {
		delete node;
	}

	if(_parent != 0)
		_parent->_children.removeAll(this);
}

bool MetadataNode::isDocument() {
	return false;
}

bool MetadataNode::isResource() {
	return false;
}

bool MetadataNode::isProperty() {
	return false;
}

bool MetadataNode::isSimpleProperty() {
	return false;
}

bool MetadataNode::isListProperty() {
	return false;
}

bool MetadataNode::isListItem() {
	return false;
}

bool MetadataNode::isComplexProperty() {
	return false;
}

MetadataNode* MetadataNode::clone(MetadataNode* parent, bool deep) {
	MetadataNode* clone = new MetadataNode(parent);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

QList<MetadataResource*> MetadataNode::queryResources(QString resourceUri) {
	QList<MetadataResource*> list;

	if(resourceUri == "*") {
		foreach(MetadataNode* node, children()) {
			if(node->isResource())
				list.append(node->toResource());
		}

		foreach(MetadataNode* node, children()) {
			if(!node->isResource())
				list += node->queryResources(resourceUri);
		}
	} else {
		foreach(MetadataNode* node, children()) {
			if(node->isResource() && node->toResource()->resourceUri() == resourceUri)
				list.append(node->toResource());
		}

		foreach(MetadataNode* node, children()) {
			if(!(node->isResource() && node->toResource()->resourceUri() == resourceUri))
				list += node->queryResources(resourceUri);
		}
	}

	return list;
}

MetadataResource* MetadataNode::queryResource(QString resourceUri) {
	QList<MetadataResource*> list = queryResources(resourceUri);

	if(list.isEmpty())
		return 0;
	else
		return list.at(0);
}

QList<MetadataProperty*> MetadataNode::queryProperties(QString resourceUri, MetadataQName qName, QVariant value) {
	QList<MetadataProperty*> list;

	foreach(MetadataResource* resource, queryResources(resourceUri)) {
		list += resource->queryProperties(qName, value);
	}

	return list;
}

MetadataProperty* MetadataNode::queryProperty(QString resourceUri, MetadataQName qName, QVariant value) {
	QList<MetadataProperty*> list = queryProperties(resourceUri, qName, value);

	if(list.isEmpty())
		return 0;
	else
		return list.at(0);
}

QList<MetadataProperty*> MetadataNode::queryProperties(MetadataQName qName, QVariant value) {
	QList<MetadataProperty*> list;

	foreach(MetadataNode* node, children()) {
		if(node->isProperty()) {
			bool accept = true;
			MetadataProperty* propertyNode = node->toProperty();

			if(qName.namespaceUri() != "*" && propertyNode->qName().namespaceUri() != qName.namespaceUri())
				accept = false;

			if(qName.localName() != "*" && propertyNode->qName().localName() != qName.localName())
				accept = false;

			if(!value.isNull() && !propertyNode->values().contains(value))
				accept = false;

			if(accept)
				list.append(propertyNode);
		}

		if(node->isDocument() || node->isResource() || node->isComplexProperty())
			list += node->queryProperties(qName, value);
	}

	return list;
}

MetadataProperty* MetadataNode::queryProperty(MetadataQName qName, QVariant value) {
	QList<MetadataProperty*> list = queryProperties(qName, value);

	if(list.isEmpty())
		return 0;
	else
		return list.at(0);
}

QList<QVariant> MetadataNode::queryValues(QString resourceUri, MetadataQName qName) {
	QList<QVariant> list;

	foreach(MetadataResource* resource, queryResources(resourceUri)) {
		list += resource->queryValues(qName);
	}

	return list;
}

QVariant MetadataNode::queryValue(QString resourceUri, MetadataQName qName) {
	QList<QVariant> list = queryValues(resourceUri, qName);

	if(list.isEmpty())
		return 0;
	else
		return list.at(0);
}

QList<QVariant> MetadataNode::queryValues(MetadataQName qName) {
	QList<QVariant> list;

	foreach(MetadataProperty* property, queryProperties(qName)) {
		list += property->values();
	}

	return list;
}

QVariant MetadataNode::queryValue(MetadataQName qName) {
	QList<QVariant> list = queryValues(qName);

	if(list.isEmpty())
		return 0;
	else
		return list.at(0);
}

MetadataDocument* MetadataNode::toDocument() {
	return static_cast<MetadataDocument*>(this);
}

MetadataResource* MetadataNode::toResource() {
	return static_cast<MetadataResource*>(this);
}

MetadataProperty* MetadataNode::toProperty() {
	return static_cast<MetadataProperty*>(this);
}

MetadataSimpleProperty* MetadataNode::toSimpleProperty() {
	return static_cast<MetadataSimpleProperty*>(this);
}

MetadataListProperty* MetadataNode::toListProperty() {
	return static_cast<MetadataListProperty*>(this);
}

MetadataListItem* MetadataNode::toListItem() {
	return static_cast<MetadataListItem*>(this);
}

MetadataComplexProperty* MetadataNode::toComplexProperty() {
	return static_cast<MetadataComplexProperty*>(this);
}

MetadataNode* MetadataNode::parent() {
	return _parent;
}

void MetadataNode::setParent(MetadataNode* parent) {
	if(_parent != 0)
		_parent->_children.removeAll(this);

	_parent = parent;
	_parent->_children.append(this);
}

QList<MetadataNode*> MetadataNode::children() {
	return _children;
}

void MetadataNode::removeChild(MetadataNode* child) {
	_children.removeAll(child);
}

QList<MetadataNode*> MetadataNode::cloneChildren(MetadataNode* parent, bool deep) {
	QList<MetadataNode*> clones;

	foreach(MetadataNode* child, _children) {
		 clones.append(child->clone(parent, deep));
	}

	return clones;
}

MetadataDocument::MetadataDocument(MetadataNode* parent) : MetadataNode(parent) {
}

bool MetadataDocument::isDocument() {
	return true;
}

MetadataNode* MetadataDocument::clone(MetadataNode* parent, bool deep) {
	MetadataDocument* clone = new MetadataDocument(parent);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

MetadataResource::MetadataResource(MetadataNode* parent, QString resourceUri) : MetadataNode(parent) {
	_resourceUri = resourceUri;
}

bool MetadataResource::isResource() {
	return true;
}

MetadataNode* MetadataResource::clone(MetadataNode* parent, bool deep) {
	MetadataResource* clone = new MetadataResource(parent, _resourceUri);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

QString MetadataResource::resourceUri() {
	return _resourceUri;
}

void MetadataResource::setResourceUri(QString resourceUri) {
	_resourceUri = resourceUri;
}

MetadataProperty::MetadataProperty(MetadataNode* parent, MetadataQName qName) : MetadataNode(parent) {
	_qName = qName;
}

bool MetadataProperty::isProperty() {
	return true;
}

MetadataNode* MetadataProperty::clone(MetadataNode* parent, bool deep) {
	MetadataProperty* clone = new MetadataProperty(parent, _qName);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

MetadataQName MetadataProperty::qName() {
	return _qName;
}

void MetadataProperty::setQName(MetadataQName qName) {
	_qName = qName;
}

QList<QVariant> MetadataProperty::values() {
	return QList<QVariant>();
}

MetadataSimpleProperty::MetadataSimpleProperty(MetadataNode* parent, MetadataQName qName, QVariant value) : MetadataProperty(parent, qName) {
	_value = value;
}

bool MetadataSimpleProperty::isSimpleProperty() {
	return true;
}

MetadataNode* MetadataSimpleProperty::clone(MetadataNode* parent, bool deep) {
	MetadataSimpleProperty* clone = new MetadataSimpleProperty(parent, _qName, _value);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

QList<QVariant> MetadataSimpleProperty::values() {
	QList<QVariant> values;

	values.append(_value);

	return values;
}

QVariant MetadataSimpleProperty::value() {
	return _value;
}

void MetadataSimpleProperty::setValue(QVariant value) {
	_value = value;
}

MetadataListProperty::MetadataListProperty(MetadataNode* parent, MetadataQName qName) : MetadataProperty(parent, qName) {
}

bool MetadataListProperty::isListProperty() {
	return true;
}

MetadataNode* MetadataListProperty::clone(MetadataNode* parent, bool deep) {
	MetadataListProperty* clone = new MetadataListProperty(parent, _qName);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

QList<QVariant> MetadataListProperty::values() {
	QList<QVariant> values;

	foreach(MetadataNode* node, children()) {
		if(node->isListItem())
			values.append(node->toListItem()->value());
	}

	return values;
}

void MetadataListProperty::append(QList<QVariant> values) {
	foreach(QVariant value, values) {
		new MetadataListItem(this, value);
	}
}

MetadataListItem::MetadataListItem(MetadataNode* parent, QVariant value) : MetadataNode(parent) {
	_value = value;
}

bool MetadataListItem::isListItem() {
	return true;
}

MetadataNode* MetadataListItem::clone(MetadataNode* parent, bool deep) {
	MetadataListItem* clone = new MetadataListItem(parent, _value);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

QVariant MetadataListItem::value() {
	return _value;
}

void MetadataListItem::setValue(QVariant value) {
	_value = value;
}

MetadataComplexProperty::MetadataComplexProperty(MetadataNode* parent, MetadataQName qName) : MetadataProperty(parent, qName) {
}

bool MetadataComplexProperty::isComplexProperty() {
	return true;
}

MetadataNode* MetadataComplexProperty::clone(MetadataNode* parent, bool deep) {
	MetadataComplexProperty* clone = new MetadataComplexProperty(parent, _qName);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

QList<QVariant> MetadataComplexProperty::values() {
	QList<QVariant> values;

	foreach(MetadataNode* node, children()) {
		if(node->isProperty())
			values += node->toProperty()->values();
	}

	return values;
}

MetadataTransform::MetadataTransform() {
}

MetadataTransform::MetadataTransform(QVariant parameter) {
	_parameter = parameter;
}

MetadataTransform::~MetadataTransform() {
	
}

QVariant MetadataTransform::parameter() {
	return _parameter;
}
void MetadataTransform::setParameter(QVariant parameter) {
	_parameter = parameter;
}
