/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef METADATARDF_H_
#define METADATARDF_H_

#include <QIODevice>
#include <QList>
#include <QMap>
#include <QString>
#include <QVariant>
#include <QXmlStreamReader>
#include <QXmlStreamWriter>

#include <Metadata.h>

class MetadataRdfType {
public:
	virtual ~MetadataRdfType() {
	}

	virtual MetadataQName qName() = 0;

	virtual void write(QVariant value, QXmlStreamWriter* writer) = 0;
	virtual QVariant read(QXmlStreamReader* reader) = 0;
};

class MetadataRdfStringType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfBinaryType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfIntegerType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfDoubleType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfRationalType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfPointType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfSchema : QObject {

	Q_OBJECT

public:
	static QString namespaceUriXmlSchema;
	static QString namespaceUriBergnetTypes;
	static QString namespaceUriRdf;
	static QString namespaceUriRdfDefinition;

	static MetadataRdfSchema* instance();

	~MetadataRdfSchema();

	MetadataRdfType* type(MetadataQName qName);
	void insertType(MetadataRdfType* type);

	MetadataRdfType* property(MetadataQName qName);
	void insertProperty(MetadataQName qName, MetadataRdfType* type);
	void insertProperty(MetadataQName qName, MetadataQName typeQName);
	bool containsProperty(MetadataQName qName);
	void removeProperty(MetadataQName qName);

	bool isListProperty(MetadataQName qName);
	void setIsListProperty(MetadataQName qName);
	bool isComplexProperty(MetadataQName qName);
	void setIsComplexProperty(MetadataQName qName);

private:
	static MetadataRdfSchema* _instance;

	MetadataRdfStringType* _stringType;
	MetadataRdfBinaryType* _binaryType;
	MetadataRdfIntegerType* _integerType;
	MetadataRdfDoubleType* _doubleType;
	MetadataRdfRationalType* _rationalType;
	MetadataRdfPointType* _pointType;
	QMap<MetadataQName, MetadataRdfType*> _types;
	QMap<MetadataQName, MetadataRdfType*> _properties;
	QList<MetadataQName> _listProperties;
	QList<MetadataQName> _complexProperties;

	MetadataRdfSchema();
};

class MetadataRdfDefinitionReader {
public:
	static void readFolder(QString folder);

	MetadataRdfDefinitionReader(QIODevice* device);

	bool read();

private:
	QIODevice* _device;
	QMap<QString, QString> _typeNamespaceMap;
	QMap<QString, QString> _typeLocalNameMap;
};

class MetadataRdfReader {
public:
	static MetadataNode* readFile(QString fileName);
	static MetadataNode* readFolder(QString folder, MetadataNode* parent=0);

	MetadataRdfReader(QIODevice* device);
	virtual ~MetadataRdfReader();

	MetadataNode* read(MetadataNode* parent=0);

protected:
	QXmlStreamReader* _reader;
};

class MetadataRdfWriter {
public:
	static void writeToFile(MetadataNode* node, QString filename);

	MetadataRdfWriter(QIODevice* device);
	virtual ~MetadataRdfWriter();

	void setPrefix(QString namespaceUri, QString prefix);

	void write(MetadataNode* node);
	void write(QList<MetadataNode*> nodes);

protected:
	QXmlStreamWriter* _writer;
	QMap<QString, QString> _prefix;
};

#ifdef METADATAXPATHSUPPORT

// TODO: use QXmlSerializer

#endif

#endif /*METADATARDF_H_*/
