/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "RawImporter.h"

#include <QApplication>
#include <QColor>
#include <QDebug>
#include <QFileDialog>
#include <QTextStream>
#include <QTime>

#include <math.h>

#include <CanonCr2.h>
#include <ColorManagement.h>
#include <Dialogs.h>
#include <MetadataRdf.h>
#include <QtUtils.h>
#include <Tiff.h>

#include <ui_rawdev.h>

RawdevCore::RawdevCore() {
	_rawFile = 0;
	_metadataResource = 0;
	_debayerImage = 0;

	MetadataRdfDefinitionReader::readFolder("./rdfdefinitions/");
	RawFile::readCameraDefinitions("./camdefinitions/");
}

RawdevCore::~RawdevCore() {
	if(_rawFile != 0)
		delete _rawFile;

	if(_debayerImage != 0)
		delete _debayerImage;
}

bool RawdevCore::fileOpen(QString fileName) {
	QFile inputFile(fileName);

	if(!inputFile.open(QIODevice::ReadOnly))
		return false;

	if((_rawFile = RawFile::fromFile(&inputFile)) == 0)
		return false;

	Image* bayerImage = _rawFile->readImage();
	inputFile.close();

	_metadataResource = _rawFile->readMetadata()->queryResource(QString("*"));

	_settingsDefault = _metadataResource->queryProperty(RawExposure::propertyName, QVariant("default"))->parent();
	_settingsCurrent = _metadataResource->queryProperty(RawExposure::propertyName, QVariant("current"))->parent();

	if(_debayerImage != 0)
		delete _debayerImage;

	_debayerImage = _demosaicFilter.filter(bayerImage, RgbImage48::format());

	return true;
}

bool RawdevCore::fileClose() {
	if(_rawFile != 0) {
		delete _rawFile;

		_rawFile = 0;
	}

	if(_debayerImage != 0) {
		delete _debayerImage;

		_debayerImage = 0;
	}

	return true;
}

void RawdevCore::saveImage(QString fileName) {
	if(_debayerImage == 0)
		return;

	Image* exposedImage = _exposureFilter.filter(_debayerImage, RgbImage48::format());

	QString outputProfile = DevelopFilter::profileSRgb;
	if(_developParameters.contains(DevelopFilter::parameterOutputProfile))
		outputProfile = _developParameters.value(DevelopFilter::parameterOutputProfile).toString();

	QString imageWriterName = QtImageWriter::name;		
	if(_imageWriterParamters.contains(ImageWriter::parameterImageWriter))
		imageWriterName = _imageWriterParamters.value(ImageWriter::parameterImageWriter).toString();

	if(imageWriterName == TiffWriter::name) {
		MetadataTransform* metadataTransform = new TiffSimpleMetadataTransform();
		MetadataResource* resource = metadataTransform->transform(exposedImage->metadata())->toResource();

		ImageWriter* imageWriter = new TiffWriter();

		if(outputProfile == DevelopFilter::profileProPhotoLinearRgb) {
			imageWriter->write(fileName, exposedImage, resource, _imageWriterParamters);
		} else {
			_developFilter.setParameters(_developParameters);
			Image* developedImage = _developFilter.filter(exposedImage, RgbImage32::format());

			imageWriter->write(fileName, developedImage, resource, _imageWriterParamters);

			delete developedImage;
		}

		delete imageWriter;
		delete metadataTransform;
	} else if(imageWriterName == QtImageWriter::name) {
		_developFilter.setParameters(_developParameters);
		Image* developedImage = _developFilter.filter(exposedImage, RgbImage32::format());

		ImageWriter* imageWriter = new QtImageWriter();
		imageWriter->write(fileName, developedImage, developedImage->metadata(), _imageWriterParamters);

		delete imageWriter;
		delete developedImage;
	}

	delete exposedImage;
}

void RawdevCore::saveBayerImage(QString fileName) {
	if(_rawFile == 0)
		return;

	ImageWriter* imageWriter = new TiffWriter();

	imageWriter->write(fileName, _rawFile->image(), (MetadataResource*)0, _imageWriterParamters);

	delete imageWriter;
}

void RawdevCore::saveMetadata(QString fileName) {
	if(_rawFile == 0)
		return;

	QFile file(fileName);
	MetadataRdfWriter rdfWriter(&file);
	rdfWriter.setPrefix(Tiff::namespaceUri, "tiff");
	rdfWriter.setPrefix(Exif::namespaceUri, "exif");
	rdfWriter.setPrefix(RawFile::namespaceUri, "raw");
	rdfWriter.setPrefix(RawExposure::namespaceUri, "rawexposure");
	rdfWriter.setPrefix(CanonCr2::namespaceUri, "canoncr2");
	file.open(QIODevice::WriteOnly);
	rdfWriter.write(_rawFile->metadata());
	file.close();
}

bool RawdevCore::imageLoaded() {
	return _rawFile != 0;
}

int RawdevCore::sourceWidth() {
	if(_rawFile == 0)
		return -1;

	return _debayerImage->width();
}

int RawdevCore::sourceHeight() {
	if(_rawFile == 0)
		return -1;

	return _debayerImage->height();
}

QList<QPointF> RawdevCore::toneCurve() {
	if(_rawFile == 0)
		return QList<QPointF>();

	MetadataListProperty* toneCurveProperty = MetadataNode::toListProperty(_settingsCurrent->queryProperty(RawExposure::propertyToneCurve));
 
	if(toneCurveProperty == 0)
		return QList<QPointF>();

	return QtUtils::toTypedList<QPointF>(toneCurveProperty->values());
}

void RawdevCore::setToneCurve(Spline curve) {
	if(_metadataResource == 0)
		return;

	MetadataListProperty* toneCurveProperty = MetadataNode::toListProperty(_settingsCurrent->queryProperty(RawExposure::propertyToneCurve));

	if(toneCurveProperty == 0)
		toneCurveProperty = new MetadataListProperty(_settingsCurrent, RawExposure::propertyToneCurve);

	foreach(MetadataNode* child, toneCurveProperty->children()) {
		toneCurveProperty->removeChild(child);
	}

	toneCurveProperty->append(QtUtils::toVariantList(curve.points()));
}

QString RawdevCore::whiteBalance() {
	return _settingsCurrent->queryValue(RawExposure::propertyWhiteBalance).toString();
}

void RawdevCore::setWhiteBalance(QString whiteBalance) {
	if(_metadataResource == 0)
		return;

	if(whiteBalance.isEmpty())
		whiteBalance = "asShot";
	else if(whiteBalance.compare("asShot", Qt::CaseInsensitive) == 0)
		whiteBalance = "asShot";
	else if(whiteBalance.compare("manual", Qt::CaseInsensitive) == 0)
		whiteBalance = "manual";

	MetadataSimpleProperty* whiteBalanceModeProperty = MetadataNode::toSimpleProperty(_settingsCurrent->queryProperty(RawExposure::propertyWhiteBalance));

	if(whiteBalanceModeProperty == 0)
		whiteBalanceModeProperty = new MetadataSimpleProperty(_settingsCurrent, RawExposure::propertyWhiteBalance);

	whiteBalanceModeProperty->setValue(QVariant(whiteBalance));
}

int RawdevCore::whiteBalanceTemperature() {
	return _settingsCurrent->queryValue(RawExposure::propertyTemperature).toInt();
}

void RawdevCore::setWhiteBalanceTemperature(int temperature) {
	if(_metadataResource == 0)
		return;

	MetadataSimpleProperty* temperatureProperty = MetadataNode::toSimpleProperty(_settingsCurrent->queryProperty(RawExposure::propertyTemperature));

	if(temperatureProperty == 0)
		temperatureProperty = new MetadataSimpleProperty(_settingsCurrent, RawExposure::propertyTemperature);

	temperatureProperty->setValue(QVariant(temperature));
}

int RawdevCore::whiteBalanceTint() {
	return _settingsCurrent->queryValue(RawExposure::propertyTint).toInt();
}

void RawdevCore::setWhiteBalanceTint(int tint) {
	if(_metadataResource == 0)
		return;

	MetadataSimpleProperty* tintProperty = MetadataNode::toSimpleProperty(_settingsCurrent->queryProperty(RawExposure::propertyTint));

	if(tintProperty == 0)
		tintProperty = new MetadataSimpleProperty(_settingsCurrent, RawExposure::propertyTint);

	tintProperty->setValue(QVariant(tint));	
}

BergPhoto::Parameters RawdevCore::developParameters() {
	return _developParameters;
}

void RawdevCore::setDevelopParameters(BergPhoto::Parameters parameters) {
	_developParameters = parameters;
}

BergPhoto::Parameters RawdevCore::imageWriterParameters() {
	return _imageWriterParamters;
}

void RawdevCore::setImageWriterParameters(BergPhoto::Parameters parameters) {
	_imageWriterParamters = parameters;
}

Image* RawdevCore::preview(int width, int height) {
	if(_debayerImage == 0)
		return 0;

	BergPhoto::Parameters resizeParameters;
	resizeParameters.insert("targetWidth", width);
	resizeParameters.insert("targetHeight", height);

	ResizeFilter resizeFilter;
	resizeFilter.setParameters(resizeParameters);

	Image* resizedImage = resizeFilter.filter(_debayerImage, RgbImage48::format());
	Image* exposedImage = _exposureFilter.filter(resizedImage, RgbImage48::format());

	BergPhoto::Parameters developParameters = _developParameters;
	developParameters.insert(DevelopFilter::parameterOutputProfile, DevelopFilter::profileSRgb);
	_developFilter.setParameters(developParameters);

	Image* previewImage =  _developFilter.filter(exposedImage, RgbImage32::format());

	delete resizedImage;
	delete exposedImage;

	return previewImage;
}

RawdevGui::RawdevGui(QMainWindow* mainWindow, RawdevCore* rawdevCore) {
	_previewImage = 0;

	_rawdevCore = rawdevCore;

	_mainWindow = mainWindow;
	_mainWindowRawdev.setupUi(_mainWindow);

	connect(_mainWindowRawdev.actionFileOpen, SIGNAL(triggered()), this, SLOT(fileOpen()));
	connect(_mainWindowRawdev.actionExit, SIGNAL(triggered()), QCoreApplication::instance(), SLOT(quit()));
	connect(_mainWindowRawdev.actionPreviewImage, SIGNAL(triggered()), this, SLOT(previewImage()));
	connect(_mainWindowRawdev.actionSaveImage, SIGNAL(triggered()), this, SLOT(saveImage()));
	connect(_mainWindowRawdev.actionSetOutputParameters, SIGNAL(triggered()), this, SLOT(setOutputParameters()));
	connect(_mainWindowRawdev.actionSaveBayerImage, SIGNAL(triggered()), this, SLOT(saveBayerImage()));
	connect(_mainWindowRawdev.actionSaveMetadata, SIGNAL(triggered()), this, SLOT(saveMetadata()));

	connect(_mainWindowRawdev.pushButtonRealtimePreview, SIGNAL(toggled(bool)), this, SLOT(setRealtimePreview(bool)));

	connect(_mainWindowRawdev.widgetPreviewImage, SIGNAL(outputSizeChanged(int, int)), this, SLOT(previewImageSizeChanged(int, int)));	

	connect(_mainWindowRawdev.comboBoxWhiteBalanceMode, SIGNAL(currentIndexChanged(QString)), this, SLOT(setWhiteBalanceMode(QString)));
	connect(_mainWindowRawdev.spinBoxTemperature, SIGNAL(valueChanged(int)), this, SLOT(setWhiteBalanceTemperature(int)));
	connect(_mainWindowRawdev.spinBoxTint, SIGNAL(valueChanged(int)), this, SLOT(setWhiteBalanceTint(int)));

	connect(_mainWindowRawdev.widgetToneCurve, SIGNAL(curveChanged(Spline)), this, SLOT(setToneCurve(Spline)));

	syncGui();
}

RawdevGui::~RawdevGui() {
	if(_previewImage != 0)
		delete _previewImage;
}

void RawdevGui::fileOpen() {
	QString fileName = QFileDialog::getOpenFileName(_mainWindow, "Open raw file", QString(), "Raw files (*.cr2 *.dng)");

	if(fileName.isEmpty())
		return;

	_rawdevCore->fileOpen(fileName);

	syncGui();
}

void RawdevGui::fileClose() {
	_rawdevCore->fileClose();

	syncGui();
}

void RawdevGui::previewImage() {
	if(!_rawdevCore->imageLoaded()) {
		_mainWindowRawdev.widgetPreviewImage->setImage(0);
	} else {
		_previewImage = _rawdevCore->preview(_previewWidth, _previewHeight);

		QImage* outputQImage = qobject_cast<RgbImage32*>(_previewImage)->toQImage();

		_mainWindowRawdev.widgetPreviewImage->setImage(outputQImage);
	}
}

void RawdevGui::previewImageSizeChanged(int width, int height) {
	if(_rawdevCore->imageLoaded()) {
		double widgetWidth = (double)width;
		double widgetHeight = (double)height;
		double imageWidth = (double)_rawdevCore->sourceWidth();
		double imageHeight = (double)_rawdevCore->sourceHeight();

		if((imageWidth/imageHeight) > (widgetWidth/widgetHeight)) {
			_previewWidth = (int)widgetWidth;
			_previewHeight = (int)(widgetWidth*imageHeight/imageWidth);
		} else {
			_previewWidth = (int)(widgetHeight*imageWidth/imageHeight);
			_previewHeight = (int)widgetHeight;
		}
	}

	previewImage();
}

void RawdevGui::setRealtimePreview(bool flag) {
	if(flag) {
		connect(this, SIGNAL(parametersChanged()), this, SLOT(previewImage()));
		emit parametersChanged();
	} else {
		disconnect(SIGNAL(parametersChanged()), this, SLOT(previewImage()));
	}
}

void RawdevGui::saveImage() {
	if(!_rawdevCore->imageLoaded())
		return;

	QString fileFormat = "jpg";

	if(_rawdevCore->imageWriterParameters().contains(ImageWriter::parameterFileFormat))
		fileFormat = _rawdevCore->imageWriterParameters().value(ImageWriter::parameterFileFormat).toString();

	QString fileName = QFileDialog::getSaveFileName(_mainWindow, "Save image", QString(), QString("Image file (*.%1)").arg(fileFormat));

	if(fileName.isEmpty())
		return;

	_rawdevCore->saveImage(fileName);
}

void RawdevGui::setOutputParameters() {
	OutputParametersDialog dialog(_mainWindow);

	dialog.setDevelopParameters(_rawdevCore->developParameters());
	dialog.setImageWriterParameters(_rawdevCore->imageWriterParameters());

	if(dialog.exec() == QDialog::Accepted) {
		_rawdevCore->setDevelopParameters(dialog.developParameters());
		_rawdevCore->setImageWriterParameters(dialog.imageWriterParameters());
	}
}

void RawdevGui::saveBayerImage() {
	if(!_rawdevCore->imageLoaded())
		return;

	QString fileName = QFileDialog::getSaveFileName(_mainWindow, "Save bayer image", QString(), "Tiff files (*.tiff)");

	if(fileName.isEmpty())
		return;

	_rawdevCore->saveBayerImage(fileName);
}

void RawdevGui::saveMetadata() {
	if(!_rawdevCore->imageLoaded())
		return;

	QString fileName = QFileDialog::getSaveFileName(_mainWindow, "Save metadata", QString(), "XML file (*.xml)");

	if(fileName.isEmpty())
			return;

	_rawdevCore->saveMetadata(fileName);
}

void RawdevGui::setToneCurve(Spline curve) {
	_rawdevCore->setToneCurve(curve);

	emit parametersChanged();
}

void RawdevGui::setWhiteBalanceMode(QString mode) {
	_rawdevCore->setWhiteBalance(mode);

	emit parametersChanged();
}

void RawdevGui::setWhiteBalanceTemperature(int temperature) {
	_rawdevCore->setWhiteBalanceTemperature(temperature);

	emit parametersChanged();
}

void RawdevGui::setWhiteBalanceTint(int tint) {
	_rawdevCore->setWhiteBalanceTint(tint);

	emit parametersChanged();
}

void RawdevGui::syncGui() {
	if(!_rawdevCore->imageLoaded()) {
		_mainWindowRawdev.pushButtonRealtimePreview->setChecked(false);

		previewImageSizeChanged(-1, -1);

		_mainWindowRawdev.comboBoxWhiteBalanceMode->setEnabled(false);
		_mainWindowRawdev.horizontalSliderTemperature->setEnabled(false);
		_mainWindowRawdev.horizontalSliderTint->setEnabled(false);
		_mainWindowRawdev.pushButtonPreviewImage->setEnabled(false);
		_mainWindowRawdev.pushButtonRealtimePreview->setEnabled(false);
		_mainWindowRawdev.pushButtonSaveBayerImage->setEnabled(false);
		_mainWindowRawdev.pushButtonSaveImage->setEnabled(false);
		_mainWindowRawdev.pushButtonSaveMetadata->setEnabled(false);
		_mainWindowRawdev.pushButtonSetOutputParameters->setEnabled(false);
		_mainWindowRawdev.spinBoxTemperature->setEnabled(false);
		_mainWindowRawdev.spinBoxTint->setEnabled(false);
		_mainWindowRawdev.widgetToneCurve->setEnabled(false);
	} else {
		_mainWindowRawdev.comboBoxWhiteBalanceMode->setEnabled(true);
		_mainWindowRawdev.horizontalSliderTemperature->setEnabled(true);
		_mainWindowRawdev.horizontalSliderTint->setEnabled(true);
		_mainWindowRawdev.pushButtonPreviewImage->setEnabled(true);
		_mainWindowRawdev.pushButtonRealtimePreview->setEnabled(true);
		_mainWindowRawdev.pushButtonSaveBayerImage->setEnabled(true);
		_mainWindowRawdev.pushButtonSaveImage->setEnabled(true);
		_mainWindowRawdev.pushButtonSaveMetadata->setEnabled(true);
		_mainWindowRawdev.pushButtonSetOutputParameters->setEnabled(true);
		_mainWindowRawdev.spinBoxTemperature->setEnabled(true);
		_mainWindowRawdev.spinBoxTint->setEnabled(true);
		_mainWindowRawdev.widgetToneCurve->setEnabled(true);

		_mainWindowRawdev.comboBoxWhiteBalanceMode->setCurrentIndex(0);
		_mainWindowRawdev.spinBoxTemperature->setValue(_rawdevCore->whiteBalanceTemperature());
		_mainWindowRawdev.spinBoxTint->setValue(_rawdevCore->whiteBalanceTint());
		_mainWindowRawdev.widgetToneCurve->setPoints(_rawdevCore->toneCurve());

		previewImageSizeChanged(_mainWindowRawdev.widgetPreviewImage->width(), _mainWindowRawdev.widgetPreviewImage->height());
	}
}

#if 0

int main(int argc, char** argv) {
	QApplication application(argc, argv);

	MetadataRdfDefinitionReader::readFolder("./rdfdefinitions/");

	MetadataNode* node = MetadataRdfReader::readFile("test.xml");
	MetadataRdfWriter::writeToFile(node, "test.out.xml");

	return 0;
}

#endif

#if 1

int main(int argc, char** argv) {
	QApplication application(argc, argv);

	QTextStream streamOut(stdout);
	QTextStream streamErr(stderr);

	QString inputFilename;
	QString outputFilename;
	QString rdfOutputFilename;
	QString bayerOutputFilename;
	QString toneCurve;
	QString whiteBalance;
	QString wbTemperature;
	QString wbTint;
	QString developOutputProfile;
	QString imageWriter;
	bool imageWriterCompress = false;
	QString imageWriterQuality;
	bool gui = false;

	for(int i=1; i<argc; i++) {
		QString parameter = QString::fromAscii(argv[i]);

		if(parameter.startsWith("--input="))
			inputFilename = parameter.mid(8);

		if(parameter.startsWith("--output="))
			outputFilename = parameter.mid(9);

		if(parameter.startsWith("--rdfoutput="))
			rdfOutputFilename = parameter.mid(12);

		if(parameter.startsWith("--dumpbayer="))
			bayerOutputFilename = parameter.mid(12);

		if(parameter.startsWith("--tonecurve="))
			toneCurve = parameter.mid(12);

		if(parameter.startsWith("--wb="))
			whiteBalance = parameter.mid(5);

		if(parameter.startsWith("--wb-temperature="))
			wbTemperature = parameter.mid(17);

		if(parameter.startsWith("--wb-tint="))
			wbTint = parameter.mid(10);

		if(parameter.startsWith("--develop-outputprofile="))
			developOutputProfile = parameter.mid(24);

		if(parameter.startsWith("--imagewriter="))
			imageWriter = parameter.mid(14);

		if(parameter.startsWith("--imagewriter-compress="))
			imageWriterCompress = parameter.mid(23).compare("true", Qt::CaseInsensitive) == 0;

		if(parameter.startsWith("--imagewriter-quality="))
			imageWriterQuality = parameter.mid(22);

		if(parameter.startsWith("--gui="))
			gui = parameter.mid(6).compare("true", Qt::CaseInsensitive) == 0;
	}

	if(!gui && inputFilename.isEmpty()) {
		streamErr << "no input file defined\n";
		return -1;
	}

	RawdevCore rawdevCore;

	if(!inputFilename.isEmpty())
		rawdevCore.fileOpen(inputFilename);

	if(!whiteBalance.isEmpty())
		rawdevCore.setWhiteBalance(whiteBalance);

	if(!wbTemperature.isEmpty())
		rawdevCore.setWhiteBalanceTemperature(wbTemperature.toInt());

	if(!wbTint.isEmpty())
		rawdevCore.setWhiteBalanceTint(wbTint.toInt());

	if(!toneCurve.isEmpty())
		rawdevCore.setToneCurve(Spline(toneCurve));

	if(!developOutputProfile.isEmpty()) {
		BergPhoto::Parameters parameters = rawdevCore.developParameters();
		parameters.insert(DevelopFilter::parameterOutputProfile, developOutputProfile);
		rawdevCore.setDevelopParameters(parameters);
	}

	if(!imageWriter.isEmpty()) {
		BergPhoto::Parameters parameters = rawdevCore.imageWriterParameters();
		parameters.insert(ImageWriter::parameterImageWriter, imageWriter);
		rawdevCore.setImageWriterParameters(parameters);
	}

	if(imageWriterCompress) {
		BergPhoto::Parameters parameters = rawdevCore.imageWriterParameters();
		parameters.insert(ImageWriter::parameterCompress, imageWriterCompress);
		rawdevCore.setImageWriterParameters(parameters);
	}

	if(!imageWriterQuality.isEmpty()) {
		BergPhoto::Parameters parameters = rawdevCore.imageWriterParameters();
		parameters.insert(ImageWriter::parameterQuality, imageWriterQuality);
		rawdevCore.setImageWriterParameters(parameters);
	}

	if(!rdfOutputFilename.isEmpty())
		rawdevCore.saveMetadata(rdfOutputFilename);

	if(!bayerOutputFilename.isEmpty())
		rawdevCore.saveBayerImage(bayerOutputFilename);

	if(!outputFilename.isEmpty())
		rawdevCore.saveImage(outputFilename);

	if(gui) {
		QMainWindow mainWindow;
		RawdevGui rawdev(&mainWindow, &rawdevCore);
		mainWindow.show();
		return application.exec();
	}

	return 0;
}

#endif
