/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef COLORMATH_H_
#define COLORMATH_H_

#include <QList>
#include <QPointF>
#include <QVector>

class ColorXYZ;
class Colorxy;

class Matrix3x3 {
public:
	Matrix3x3();
	Matrix3x3(double* c);
	Matrix3x3(double c0, double c1, double c2);
	Matrix3x3(QList<double> l);

	virtual ~Matrix3x3();

	Matrix3x3 operator+(Matrix3x3 m);

	Matrix3x3 operator*(Matrix3x3 m);

	inline void multiplyVector(double* v) const {
		double a = v[0]*_c[0][0] + v[1]*_c[0][1] + v[2]*_c[0][2];
		double b = v[0]*_c[1][0] + v[1]*_c[1][1] + v[2]*_c[1][2];
		double c = v[0]*_c[2][0] + v[1]*_c[2][1] + v[2]*_c[2][2];

		v[0] = a;
		v[1] = b;
		v[2] = c;
	}

	inline void multiplyVectorLine(double* v, int length) const {
		for(int i=0; i<length; i++) {
			double a = v[0]*_c[0][0] + v[1]*_c[0][1] + v[2]*_c[0][2];
			double b = v[0]*_c[1][0] + v[1]*_c[1][1] + v[2]*_c[1][2];
			double c = v[0]*_c[2][0] + v[1]*_c[2][1] + v[2]*_c[2][2];

			v[0] = a;
			v[1] = b;
			v[2] = c;

			v += 3;
		}
	}

	Matrix3x3 inverse();

	double coeff(int index0, int index1) const;

	QList<double> toList() const;

	void debug() const;

private:
	double _c[3][3];
};

class ColorXYZ {
public:
	ColorXYZ();
	ColorXYZ(double x, double y, double z);
	ColorXYZ(QList<double> l);

	ColorXYZ operator*(Matrix3x3 matrix);

	double x();
	double y();
	double z();

	ColorXYZ inverse();

	Colorxy toColorxy();
	Matrix3x3 toMatrix3x3();

	void debug();

private:
	double _x, _y, _z;
};

class Colorxy {
public:
	Colorxy();
	Colorxy(double x, double y);

	double x();
	double y();

	ColorXYZ toColorXYZ();

	void debug();

	static Colorxy whitePointA();
	static Colorxy whitePointB();
	static Colorxy whitePointC();
	static Colorxy whitePointD50();
	static Colorxy whitePointD55();
	static Colorxy whitePointD65();
	static Colorxy whitePointD75();

private:
	double _x, _y;
};

class CIEUtils {
public:
	static Matrix3x3 fromProPhotoRgb2XYZ();

	static Matrix3x3 fromXYZ2ProPhotoRgb();

	static Matrix3x3 fromsRgb2XYZ();

	static Matrix3x3 fromXYZ2sRGB();

	static Matrix3x3 cameraMatrix(Matrix3x3 c1, double t1, Matrix3x3 c2, double t2, double t);

	static Matrix3x3 whitePointXYZ2WhitePointXYZ(ColorXYZ sWhitePoint, ColorXYZ dWhitePoint);
};

class TemperatureUtils {
public:
	static void fromxy(Colorxy xy, double* temperature, double* tint);
	static Colorxy toxy(double temperature, double tint);

private:
	struct ruvt {
		double r;
		double u;
		double v;
		double t;
	};

	static const ruvt ruvtTable[];
};

class Spline {
public:
	Spline();
	Spline(QString values);
	Spline(QList<QPointF> list, double scale=1.0);
	virtual ~Spline();

	double evaluate(double x);

	int addPoint(QPointF p);
	void movePoint(QPointF p, int index);
	void removePoint(int index);

	QList<QPointF> points();
	virtual void setPoints(QList<QPointF> list, double scale=1.0);

protected:
	QVector<QPointF> _p;

private:
	QVector<double> _s;

	bool containsX(double x);

	void solve();

	double evaluateSegment(double x, QPointF p0, double s0, QPointF p1, double s1);
};

#endif /*COLORMATH_H_*/
