/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "MetadataRdf.h"

#include <QDebug>
#include <QDir>
#include <QPointF>
#include <QSet>
#include <QStringList>

#include <QtUtils.h>

QString MetadataRdfStringType::name = "string";

MetadataQName MetadataRdfStringType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriXmlSchema, name);
}

void MetadataRdfStringType::write(QVariant value, QXmlStreamWriter* writer) {
	writer->writeCharacters(value.toString());
}

QVariant MetadataRdfStringType::read(QXmlStreamReader* reader) {
	return QVariant(reader->readElementText());
}

QString MetadataRdfBinaryType::name = "base64Binary";

MetadataQName MetadataRdfBinaryType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriXmlSchema, name);
}

void MetadataRdfBinaryType::write(QVariant value, QXmlStreamWriter* writer) {
	writer->writeCharacters(value.toByteArray().toBase64());
}

QVariant MetadataRdfBinaryType::read(QXmlStreamReader* reader) {
	return QVariant(QByteArray::fromBase64(reader->readElementText().toAscii()));
}

QString MetadataRdfIntegerType::name = "integer";

MetadataQName MetadataRdfIntegerType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriXmlSchema, name);
}

void MetadataRdfIntegerType::write(QVariant value, QXmlStreamWriter* writer) {
	writer->writeCharacters(value.toString());
}

QVariant MetadataRdfIntegerType::read(QXmlStreamReader* reader) {
	QString text = reader->readElementText();

	if(text.toLongLong() == text.toInt())
		return QVariant(text.toInt());
	else
		return QVariant(text.toLongLong());
}

QString MetadataRdfDoubleType::name = "double";

MetadataQName MetadataRdfDoubleType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriXmlSchema, name);
}

void MetadataRdfDoubleType::write(QVariant value, QXmlStreamWriter* writer) {
	writer->writeCharacters(value.toString());
}

QVariant MetadataRdfDoubleType::read(QXmlStreamReader* reader) {
	QString text = reader->readElementText();

	return QVariant(text.toDouble());
}

QString MetadataRdfRationalType::name = "rational";

MetadataQName MetadataRdfRationalType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriBergnetTypes, name);
}

void MetadataRdfRationalType::write(QVariant value, QXmlStreamWriter* writer) {
	writer->writeCharacters(value.value<Rational>().toString());
}

QVariant MetadataRdfRationalType::read(QXmlStreamReader* reader) {
	QStringList strings = reader->readElementText().split('/');

	QVariant value;
	value.setValue(Rational(strings.at(0).toLongLong(), strings.at(1).toLongLong()));

	return value;
}

QString MetadataRdfPointType::name = "point";

MetadataQName MetadataRdfPointType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriBergnetTypes, name);
}

void MetadataRdfPointType::write(QVariant value, QXmlStreamWriter* writer) {
	QPointF point = value.toPointF();

	writer->writeCharacters(QString("%1, %2").arg(point.x()).arg(point.y()));
}

QVariant MetadataRdfPointType::read(QXmlStreamReader* reader) {
	QStringList list = reader->readElementText().split(',');

	return QVariant(QPointF(list.at(0).toDouble(), list.at(1).toDouble()));
}

QString MetadataRdfSchema::namespaceUriXmlSchema = "http://www.w3.org/2001/XMLSchema";
QString MetadataRdfSchema::namespaceUriRdf = "http://www.w3.org/1999/02/22-rdf-syntax-ns#";
QString MetadataRdfSchema::namespaceUriBergnetTypes = "http://ns.bergnet.org/bergphoto/rdf/types/1.0/";
QString MetadataRdfSchema::namespaceUriRdfDefinition = "http://ns.bergnet.org/bergphoto/rdf/definition/1.0/";

MetadataRdfSchema* MetadataRdfSchema::_instance = 0;

MetadataRdfSchema* MetadataRdfSchema::instance() {
	if(_instance == 0)
		_instance = new MetadataRdfSchema();

	return _instance;
}

MetadataRdfSchema::MetadataRdfSchema() {
	_stringType = new MetadataRdfStringType();
	insertType(_stringType);

	_binaryType = new MetadataRdfBinaryType();
	insertType(_binaryType);

	_integerType = new MetadataRdfIntegerType();
	insertType(_integerType);

	_doubleType = new MetadataRdfDoubleType();
	insertType(_doubleType);

	_rationalType = new MetadataRdfRationalType();
	insertType(_rationalType);

	_pointType = new MetadataRdfPointType();
	insertType(_pointType);
}

MetadataRdfSchema::~MetadataRdfSchema() {
	delete _stringType;
	delete _binaryType;
	delete _integerType;
	delete _doubleType;
	delete _rationalType;
	delete _pointType;
}

MetadataRdfType* MetadataRdfSchema::type(MetadataQName qName) {
	return _types.value(qName);
}

void MetadataRdfSchema::insertType(MetadataRdfType* type) {
	_types.insert(type->qName(), type);
}

MetadataRdfType* MetadataRdfSchema::property(MetadataQName qName) {
	return _properties.value(qName);
}

void MetadataRdfSchema::insertProperty(MetadataQName qName, MetadataRdfType* type) {
	_properties.insert(qName, type);
}

void MetadataRdfSchema::insertProperty(MetadataQName qName, MetadataQName typeQName) {
	_properties.insert(qName, type(typeQName));
}

bool MetadataRdfSchema::containsProperty(MetadataQName qName) {
	return _properties.contains(qName);
}

void MetadataRdfSchema::removeProperty(MetadataQName qName) {
	_properties.remove(qName);
	_listProperties.removeAll(qName);
	_complexProperties.removeAll(qName);
}

bool MetadataRdfSchema::isListProperty(MetadataQName qName) {
	return _listProperties.contains(qName);
}

void MetadataRdfSchema::setIsListProperty(MetadataQName qName) {
	_listProperties.append(qName);
}

bool MetadataRdfSchema::isComplexProperty(MetadataQName qName) {
	return _complexProperties.contains(qName);
}

void MetadataRdfSchema::setIsComplexProperty(MetadataQName qName) {
	_complexProperties.append(qName);
}

void MetadataRdfDefinitionReader::readFolder(QString folder) {
	QDir dir(folder);

	foreach(QString filename, dir.entryList(QStringList("*.xml"))) {
		QFile file(dir.filePath(filename));
		file.open(QIODevice::ReadOnly);
		MetadataRdfDefinitionReader definitionReader(&file);
		definitionReader.read();
		file.close();
	}
}

MetadataRdfDefinitionReader::MetadataRdfDefinitionReader(QIODevice* device) {
	_device = device;
}

bool MetadataRdfDefinitionReader::read() {
	QXmlStreamReader reader(_device);

	MetadataRdfSchema* schema = MetadataRdfSchema::instance();

	QString typeNamespaceUri;
	QString propertyNamespaceUri;
	QList<MetadataQName> propertyDefinitions;

	while(!reader.atEnd()) {
		reader.readNext();

		if(reader.isStartElement() && reader.namespaceUri() == MetadataRdfSchema::namespaceUriRdfDefinition) {
			QString localName = reader.name().toString();

			if(localName == "Types") {
				typeNamespaceUri = reader.attributes().value("namespace").toString();
			} else if(localName == "Type") {
				QString id = reader.attributes().value("id").toString();
				QString typeName = reader.attributes().value("name").toString();

				_typeNamespaceMap.insert(id, typeNamespaceUri);
				_typeLocalNameMap.insert(id, typeName);
			} else if(localName == "Properties") {
				propertyNamespaceUri = reader.attributes().value("namespace").toString();
			} else if(localName == "Property") {
				QString typeId = reader.attributes().value("typeId").toString();
				QString propertyName = reader.attributes().value("name").toString();

				MetadataQName typeQName(_typeNamespaceMap.value(typeId), _typeLocalNameMap.value(typeId));
				MetadataQName qName(propertyNamespaceUri, propertyName);

				bool isListProperty = reader.attributes().value("isList") == "true";
				bool isComplexProperty = reader.attributes().value("isComplex") == "true";

				schema->insertProperty(qName, typeQName);
				propertyDefinitions.append(qName);

				if(isListProperty)
					schema->setIsListProperty(qName);

				if(isComplexProperty)
					schema->setIsComplexProperty(qName);
			}
		} else if(reader.isEndElement() && reader.namespaceUri() == MetadataRdfSchema::namespaceUriRdfDefinition) {
			QString localName = reader.name().toString();

			if(localName == "Types")
				typeNamespaceUri = QString();
			else if(localName == "Properties")
				propertyNamespaceUri = QString();
		}
	}

	if(reader.hasError()) {
		foreach(MetadataQName qName, propertyDefinitions) {
			schema->removeProperty(qName);
		}

		qWarning() << QtUtils::fileName(reader.device()) << "error parsing xml document" << reader.errorString();

		return false;
	}

	return true;
}

MetadataNode* MetadataRdfReader::readFile(QString fileName) {
	QFile file(fileName);
	file.open(QIODevice::ReadOnly);
	MetadataRdfReader rdfReader(&file);
	MetadataNode* parent = rdfReader.read();
	file.close();

	return parent;
}

MetadataNode* MetadataRdfReader::readFolder(QString folder, MetadataNode* parent) {
	QDir dir(folder);

	foreach(QString filename, dir.entryList(QStringList("*.xml"))) {
		QFile file(dir.filePath(filename));
		file.open(QIODevice::ReadOnly);
		MetadataRdfReader rdfReader(&file);
		parent = rdfReader.read(parent);
		file.close();
	}

	return parent;
}

MetadataRdfReader::MetadataRdfReader(QIODevice* device) {
	_reader = new QXmlStreamReader(device);
}

MetadataRdfReader::~MetadataRdfReader() {
	delete _reader;
}

MetadataNode* MetadataRdfReader::read(MetadataNode* parent) {
	MetadataRdfSchema* schema = MetadataRdfSchema::instance();
	MetadataNode* root = 0;

	while(!_reader->atEnd()) {
		_reader->readNext();

		if(_reader->isStartElement()) {
			QString namespaceUri = _reader->namespaceUri().toString();
			QString localName = _reader->name().toString();

			if(namespaceUri == MetadataRdfSchema::namespaceUriRdf) {
				if(localName == "RDF") {
					if(parent == 0 || !parent->isDocument())
						root = parent = new MetadataDocument();
				} else if(localName == "Description") {
					QString resourceUri = _reader->attributes().value(MetadataRdfSchema::namespaceUriRdf, "about").toString();
					MetadataResource* resource = new MetadataResource(parent, resourceUri);
					parent = resource;
					if(root == 0)
						root = resource;
				} else if(localName == "li") {
					if(parent->isListProperty()) {
						MetadataListProperty* listProperty = parent->toListProperty();
						MetadataRdfType* type = schema->property(listProperty->qName());
						MetadataListItem* listItem = new MetadataListItem(parent);
						listItem->setValue(type->read(_reader));
						if(root == 0)
							root = listItem;
					}
				}
			} else {
				MetadataQName qName(namespaceUri, localName);

				if(schema->containsProperty(qName)) {
					if(schema->isListProperty(qName)) {
						parent = new MetadataListProperty(parent, qName);
						if(root == 0)
							root = parent;
					} else if(schema->isComplexProperty(qName)) {
						parent = new MetadataComplexProperty(parent, qName);
						if(root == 0)
							root = parent;
					} else {
						MetadataRdfType* type = schema->property(qName);
						MetadataSimpleProperty* simpleProperty = new MetadataSimpleProperty(parent, qName);
						simpleProperty->setValue(type->read(_reader));
						if(root == 0)
							root = simpleProperty;
					}
				}
			}
		} else if(_reader->isEndElement()) {
			QString namespaceUri = _reader->namespaceUri().toString();
			QString localName = _reader->name().toString();

			if(namespaceUri == MetadataRdfSchema::namespaceUriRdf && localName == "Description")
				parent = parent->parent();
			else if(schema->isListProperty(MetadataQName(namespaceUri, localName)))
				parent = parent->parent();
			else if(schema->isComplexProperty(MetadataQName(namespaceUri, localName)))
				parent = parent->parent();
		}
	}

	if(_reader->hasError()) {
		delete root;

		qWarning() << QtUtils::fileName(_reader->device()) << "error parsing xml document" << _reader->errorString();

		return 0;
	}

	return parent;
}

void MetadataRdfWriter::writeToFile(MetadataNode* node, QString filename) {
	QFile file(filename);
	file.open(QIODevice::WriteOnly);
	MetadataRdfWriter writer(&file);
	writer.write(node);
	file.close();
}

MetadataRdfWriter::MetadataRdfWriter(QIODevice* device) {
	_writer = new QXmlStreamWriter(device);

	_writer->setAutoFormatting(true);
	_writer->writeStartDocument();
}

MetadataRdfWriter::~MetadataRdfWriter() {
	_writer->writeEndElement();

	delete _writer;
}

void MetadataRdfWriter::setPrefix(QString namespaceUri, QString prefix) {
	_prefix.insert(namespaceUri, prefix);
}

void MetadataRdfWriter::write(MetadataNode* node) {
	MetadataRdfSchema* schema = MetadataRdfSchema::instance();

	if(node->isDocument()) {
		_writer->writeNamespace(MetadataRdfSchema::namespaceUriRdf, "rdf");
		_writer->writeStartElement(MetadataRdfSchema::namespaceUriRdf, "RDF");

		write(node->children());

		_writer->writeEndDocument();
	} else if(node->isResource()) {
		_writer->writeStartElement(MetadataRdfSchema::namespaceUriRdf, "Description");
		_writer->writeAttribute(MetadataRdfSchema::namespaceUriRdf, "about", node->toResource()->resourceUri());

		QSet<QString> namespaceUriSet;

		foreach(MetadataNode* child, node->children()) {
			if(child->isProperty()) {
				QString namespaceUri = child->toProperty()->qName().namespaceUri();

				if(!_prefix.contains(namespaceUri))
					namespaceUriSet.insert(namespaceUri);
			}
		}

		foreach(QString namespaceUri, _prefix.keys()) {
			_writer->writeNamespace(namespaceUri, _prefix.value(namespaceUri));
		}

		int i=0;
		foreach(QString namespaceUri, namespaceUriSet) {
			_writer->writeNamespace(namespaceUri, QString("t%1").arg(i));
			i++;
		}

		write(node->children());

		_writer->writeEndElement();
	} else if(node->isSimpleProperty()) {
		MetadataSimpleProperty* simpleProperty = node->toSimpleProperty();
		MetadataRdfType* rdfType = schema->property(simpleProperty->qName());

		if(rdfType != 0) {
			_writer->writeStartElement(simpleProperty->qName().namespaceUri(), simpleProperty->qName().localName());
			rdfType->write(simpleProperty->value(), _writer);
			_writer->writeEndElement();
		} else {
			qDebug() << "simple property type not defined in schema:" << simpleProperty->qName().namespaceUri() << simpleProperty->qName().localName();
		}
	} else if(node->isListProperty()) {
		MetadataListProperty* listProperty = node->toListProperty();

		if(schema->isListProperty(listProperty->qName())) {
			_writer->writeStartElement(listProperty->qName().namespaceUri(), listProperty->qName().localName());
			_writer->writeStartElement(MetadataRdfSchema::namespaceUriRdf, "Seq");

			write(node->children());

			_writer->writeEndElement();
			_writer->writeEndElement();
		} else {
			qDebug() << "list property type not defined in schema:" << listProperty->qName().namespaceUri() << listProperty->qName().localName();			
		}
	} else if(node->isListItem()) {
		MetadataListProperty* listProperty = node->parent()->toListProperty();
		MetadataRdfType* rdfType = schema->property(listProperty->qName());

		if(rdfType != 0) {
			_writer->writeStartElement(MetadataRdfSchema::namespaceUriRdf, "li");
			rdfType->write(node->toListItem()->value(), _writer);
			_writer->writeEndElement();
		} else {
			qDebug() << "list property type not defined in schema:" << listProperty->qName().namespaceUri() << listProperty->qName().localName();
		}
	} else if(node->isComplexProperty()) {
		MetadataComplexProperty* complexProperty = node->toComplexProperty();

		if(schema->isComplexProperty(complexProperty->qName())) {
			_writer->writeStartElement(complexProperty->qName().namespaceUri(), complexProperty->qName().localName());

			write(node->children());

			_writer->writeEndElement();
		} else {
			qDebug() << "complex property type not defined in schema:" << complexProperty->qName().namespaceUri() << complexProperty->qName().localName();
		}
	}
}

void MetadataRdfWriter::write(QList<MetadataNode*> nodes) {
	foreach(MetadataNode* node, nodes) {
		write(node);
	}
}
