/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "RawFile.h"

#include <CanonCr2.h>
#include <ColorManagement.h>
#include <DNG.h>
#include <MetadataRdf.h>
#include <QtUtils.h>

// TODO: input profile code
// TODO: output code without lcms
// TODO: output code with lcms (rgb 48)

QString RawFile::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/raw/1.0/";
MetadataQName RawFile::propertyBodyId = MetadataQName(RawFile::namespaceUri, "BodyId");
MetadataQName RawFile::propertySensorCrop = MetadataQName(RawFile::namespaceUri, "SensorCrop");
MetadataQName RawFile::propertyBlackLevel = MetadataQName(RawFile::namespaceUri, "BlackLevel");
MetadataQName RawFile::propertyWhiteLevel = MetadataQName(RawFile::namespaceUri, "WhiteLevel");
MetadataQName RawFile::propertyWhiteBalanceNeutral = MetadataQName(RawFile::namespaceUri, "WhiteBalanceNeutral");
MetadataQName RawFile::propertyCalibrationIlluminant1 = MetadataQName(RawFile::namespaceUri, "CalibrationIlluminant1");
MetadataQName RawFile::propertyCalibrationIlluminant2 = MetadataQName(RawFile::namespaceUri, "CalibrationIlluminant2");
MetadataQName RawFile::propertyColorMatrix1 = MetadataQName(RawFile::namespaceUri, "ColorMatrix1");
MetadataQName RawFile::propertyColorMatrix2 = MetadataQName(RawFile::namespaceUri, "ColorMatrix2");
MetadataQName RawFile::propertyCameraCalibration1 = MetadataQName(RawFile::namespaceUri, "CameraCalibration1");
MetadataQName RawFile::propertyCameraCalibration2 = MetadataQName(RawFile::namespaceUri, "CameraCalibration2");
MetadataQName RawFile::propertyAnalogBalance = MetadataQName(RawFile::namespaceUri, "AnalogBalance");
MetadataQName RawFile::propertyMaskedArea = MetadataQName(RawFile::namespaceUri, "MaskedArea");
MetadataDocument* RawFile::_cameraDefinitions = 0;

RawFile::~RawFile() {
	if(_metadata != 0)
		delete _metadata;

	if(_image != 0)
		delete _image;
}

MetadataDocument* RawFile::metadata() {
	return _metadata;
}

Image* RawFile::image() {
	return _image;
}

RawFile* RawFile::fromFile(QFile* file) {
	QString fileName = file->fileName();
	RawFile* rawFile = 0;

	if(fileName.endsWith(".cr2", Qt::CaseInsensitive))
		rawFile = new CanonCr2File(file);
	else if(fileName.endsWith(".dng", Qt::CaseInsensitive))
		rawFile = new DNGFile(file);

	return rawFile;
}

bool RawFile::readCameraDefinitions(QString folder) {
	_cameraDefinitions = MetadataNode::toDocument(MetadataRdfReader::readFolder(folder));

	return _cameraDefinitions != 0;
}

MetadataDocument* RawFile::cameraDefinitions() {
	return _cameraDefinitions;
}

RawFile::RawFile() {
	_metadata = 0;
	_image = 0;
}

MetadataResource* RawUtils::findCameraDefinition(QString model) {
	MetadataDocument* cameraDefinitions = RawFile::cameraDefinitions();

	if(cameraDefinitions == 0)
		return 0;

	MetadataProperty* modelProperty = cameraDefinitions->queryProperty("*", Tiff::propertyModel, QVariant(model));

	if(modelProperty == 0)
		return 0;

	return modelProperty->parent()->toResource();
}

void RawUtils::appendCameraDefinition(MetadataResource* resource, QString model) {
	MetadataResource* cameraDefinition = findCameraDefinition(model);

	if(cameraDefinition == 0)
		return;

	foreach(MetadataNode* child, cameraDefinition->children()) {
		if(child->isProperty()) {
			MetadataProperty* property = child->toProperty();

			if(Tiff::propertyMake == property->qName())
				continue;

			if(Tiff::propertyModel == property->qName())
				continue;
		}

		child->clone(resource);
	}
}

Matrix3x3 RawUtils::cameraMatrix(MetadataResource* metadataResource, double temperature) {
	return analogBalance(metadataResource) * cameraCalibration(metadataResource, temperature) * colorMatrix(metadataResource, temperature);
}

Colorxy RawUtils::cameraNeutralWhiteBalance(MetadataResource* metadataResource) {
	return cameraNeutralWhiteBalance(metadataResource, ColorXYZ(QtUtils::toTypedList<double>(metadataResource->queryValues(RawFile::propertyWhiteBalanceNeutral))));
}

Colorxy RawUtils::cameraNeutralWhiteBalance(MetadataResource* metadataResource, ColorXYZ neutralWhiteBalance) {
	double temperature;
	double tint;
	double d;

	Colorxy lastTest = Colorxy::whitePointD50();

	do {
		TemperatureUtils::fromxy(lastTest, &temperature, &tint);
		Colorxy test = (neutralWhiteBalance * cameraMatrix(metadataResource, temperature).inverse()).toColorxy();

		d = qAbs(lastTest.x()-test.x()) + qAbs(lastTest.y()-test.y());

		lastTest = test;
	} while(d > 0.0000001);

	return lastTest;
}

void RawUtils::appendBlackLevelFromMaskedArea(Image* image) {
	BayerImage16* bayerImage16;

	QList<int> maskedArea = QtUtils::toTypedList<int>(image->metadata()->queryValues(RawFile::propertyMaskedArea));

	if(maskedArea.isEmpty())
		return;

	int left = maskedArea.at(0);
	int top = maskedArea.at(1);
	int width = maskedArea.at(2);
	int height = maskedArea.at(3);

	if((bayerImage16 = qobject_cast<BayerImage16*>(image)) != 0) {
		double blackLevel = 0.0;

		for(int y=top; y<height; y++) {
			quint16* data = bayerImage16->dataLine(y);

			for(int x=left; x<width; x++)
				blackLevel += (double)data[x];
		}

		new MetadataSimpleProperty(image->metadata(), RawFile::propertyBlackLevel, (int)blackLevel / ((width-left)*(height-top)));
	}
}

double RawUtils::illuminant1(MetadataResource* metadataResource) {
	QVariant illuminant1Value = metadataResource->queryValue(RawFile::propertyCalibrationIlluminant1);

	if(!illuminant1Value.isNull())
		return illuminant1Value.toDouble();
	else
		return 0.0;
}

double RawUtils::illuminant2(MetadataResource* metadataResource) {
	QVariant illuminant2Value = metadataResource->queryValue(RawFile::propertyCalibrationIlluminant2);

	if(!illuminant2Value.isNull())
		return illuminant2Value.toDouble();
	else
		return 0.0;
}

Matrix3x3 RawUtils::colorMatrix(MetadataResource* metadataResource, double temperature) {
	QList<QVariant> colorMatrix1Values = metadataResource->queryValues(RawFile::propertyColorMatrix1);
	QList<QVariant> colorMatrix2Values = metadataResource->queryValues(RawFile::propertyColorMatrix2);

	double illuminant1Value = illuminant1(metadataResource);
	double illuminant2Value = illuminant2(metadataResource);

	Matrix3x3 colorMatrix1Matrix;
	Matrix3x3 colorMatrix2Matrix;

	if(!colorMatrix1Values.isEmpty())
		colorMatrix1Matrix = Matrix3x3(QtUtils::toTypedList<double>(colorMatrix1Values));

	if(!colorMatrix2Values.isEmpty())
		colorMatrix2Matrix = Matrix3x3(QtUtils::toTypedList<double>(colorMatrix2Values));

	if(illuminant1Value != 0.0 && illuminant2Value != 0.0) {
		double f = ((1.0 / temperature) - (1.0 / illuminant2Value)) / ((1.0 / illuminant1Value) - (1.0 / illuminant2Value));

		return interpolateMatrix(colorMatrix1Matrix, colorMatrix2Matrix, f);
	} else {
		return colorMatrix1Matrix;
	}
}

Matrix3x3 RawUtils::cameraCalibration(MetadataResource* metadataResource, double temperature) {
	QList<QVariant> cameraCalibration1Values = metadataResource->queryValues(RawFile::propertyCameraCalibration1);
	QList<QVariant> cameraCalibration2Values = metadataResource->queryValues(RawFile::propertyCameraCalibration2);

	double illuminant1Value = illuminant1(metadataResource);
	double illuminant2Value = illuminant2(metadataResource);

	Matrix3x3 cameraCalibration1Matrix;
	Matrix3x3 cameraCalibration2Matrix;

	if(!cameraCalibration1Values.isEmpty())
		cameraCalibration1Matrix = Matrix3x3(QtUtils::toTypedList<double>(cameraCalibration1Values));

	if(!cameraCalibration2Values.isEmpty())
		cameraCalibration2Matrix = Matrix3x3(QtUtils::toTypedList<double>(cameraCalibration2Values));

	if(illuminant1Value != 0.0 && illuminant2Value != 0.0) {
		double f = ((1.0 / temperature) - (1.0 / illuminant2Value)) / ((1.0 / illuminant1Value) - (1.0 / illuminant2Value));

		return interpolateMatrix(cameraCalibration1Matrix, cameraCalibration2Matrix, f);
	} else {
		return cameraCalibration1Matrix;
	}
}

Matrix3x3 RawUtils::analogBalance(MetadataResource* metadataResource) {
	QList<QVariant> analogBalanceValues = metadataResource->queryValues(RawFile::propertyAnalogBalance);

	return Matrix3x3(QtUtils::toTypedList<double>(analogBalanceValues));
}

Matrix3x3 RawUtils::interpolateMatrix(Matrix3x3 a, Matrix3x3 b, double f) {
	if(f <= 0.0)
		return a;
	else if(f >= 1.0)
		return b;
	else
		return a*Matrix3x3(f,f,f) + b*Matrix3x3(1.0-f,1.0-f,1.0-f);
}

QString RawExposure::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/raw-exposure/1.0/";
MetadataQName RawExposure::propertySettings = MetadataQName(RawExposure::namespaceUri, "Settings");
MetadataQName RawExposure::propertyName = MetadataQName(RawExposure::namespaceUri, "Name");
MetadataQName RawExposure::propertyDate = MetadataQName(RawExposure::namespaceUri, "Date");
MetadataQName RawExposure::propertyWhiteBalance = MetadataQName(RawExposure::namespaceUri, "WhiteBalance");
MetadataQName RawExposure::propertyTemperature = MetadataQName(RawExposure::namespaceUri, "Temperature");
MetadataQName RawExposure::propertyTint = MetadataQName(RawExposure::namespaceUri, "Tint");
MetadataQName RawExposure::propertyToneCurve = MetadataQName(RawExposure::namespaceUri, "ToneCurve");

BayerImage16::BayerImage16(int width, int height) : Image() {
	_width = width;
	_height = height;
	_format = BayerImage16::format();
	_data = new quint16[_width*_height];
}

BayerImage16::~BayerImage16() {
	delete _data;
}

quint16* BayerImage16::data() {
	return _data;
}

quint16* BayerImage16::dataLine(int y) {
	return _data+y*_width;
}

quint16* BayerImage16::dataOffset(int x, int y) {
	return _data+y*_width+x;
}

int BayerImage16::precision() {
	return _precision;
}

void BayerImage16::setPrecision(int precision) {
	_precision = precision;
}

QVector<int> BayerImage16::pattern() {
	return _pattern;
}

void BayerImage16::setPattern(QVector<int> pattern) {
	_pattern = pattern;
}

DemosaicFilter::DemosaicFilter() {
}

DemosaicFilter::~DemosaicFilter() {
}

QString DemosaicFilter::name() {
	return QString("DemosaicFilter");
}

bool DemosaicFilter::accepts(QString inputFormat, QString outputFormat) {
	return inputFormat == BayerImage16::format() && outputFormat == RgbImage48::format();
}

Image* DemosaicFilter::filter(Image* input, QString outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	BayerImage16* bayerImage = qobject_cast<BayerImage16*>(input);

	MetadataResource* metadataResource = input->metadata();

	int bayerWidth = bayerImage->width();
	int bayerHeight = bayerImage->height();
	int leftBorder = 0;
	int topBorder = 0;
	int rightBorder = bayerWidth-1;
	int bottomBorder = bayerHeight-1;

	QList<int> sensorCrop = QtUtils::toTypedList<int>(metadataResource->queryValues(RawFile::propertySensorCrop));

	if(!sensorCrop.isEmpty()) {
		leftBorder = sensorCrop.at(0);
		topBorder = sensorCrop.at(1);
		rightBorder = sensorCrop.at(2)+leftBorder-1;
		bottomBorder = sensorCrop.at(3)+topBorder-1;
	}

	int width = rightBorder - leftBorder + 1;
	int height = bottomBorder - topBorder + 1;

	RgbImage48* rgbImage = new RgbImage48(width, height);

	rgbImage->setMetadata(bayerImage->metadata());

	quint16* bayerData = bayerImage->data();

	QVector<int> pattern = bayerImage->pattern();

	int bayerMask[3][4];

	QVector<int> pattern0112;
	pattern0112 << Image::ColorRed << Image::ColorGreen << Image::ColorGreen << Image::ColorBlue;

	QVector<int> pattern1021;
	pattern1021 << Image::ColorGreen << Image::ColorRed << Image::ColorBlue << Image::ColorGreen;

#define FILLMASKROW(r,a,b,c,d) r[0]=a; r[1]=b; r[2]=c; r[3]=d;

	if(pattern == pattern0112) {
		FILLMASKROW(bayerMask[0], 0, 1, 2, 3);
		FILLMASKROW(bayerMask[1], 4, 0, 0, 4);
		FILLMASKROW(bayerMask[2], 3, 2, 1, 0);
	} else if(pattern == pattern1021) {
		FILLMASKROW(bayerMask[0], 1, 0, 3, 2);
		FILLMASKROW(bayerMask[1], 0, 4, 4, 0);
		FILLMASKROW(bayerMask[2], 2, 3, 0, 1);
	} else {
		qDebug("unkown bayer pattern");
		FILLMASKROW(bayerMask[0], 0, 0, 0, 0);
		FILLMASKROW(bayerMask[1], 0, 0, 0, 0);
		FILLMASKROW(bayerMask[2], 0, 0, 0, 0);
	}

#undef FILLMASKROW

	int border = 2;

	for(int y=topBorder; y<topBorder+border; y++) {
		quint16* data = rgbImage->dataLine(y-topBorder);

		for(int x=leftBorder; x<rightBorder; x++) {
			data[0] = bayerPixelSave(bayerData, bayerMask[0], x, y, bayerWidth, bayerWidth, bayerHeight);
			data[1] = bayerPixelSave(bayerData, bayerMask[1], x, y, bayerWidth, bayerWidth, bayerHeight);
			data[2] = bayerPixelSave(bayerData, bayerMask[2], x, y, bayerWidth, bayerWidth, bayerHeight);

			data += 3;
		}
	}

	for(int y=bottomBorder-border; y<bottomBorder; y++) {
		quint16* data = rgbImage->dataLine(y-topBorder);

		for(int x=leftBorder; x<rightBorder; x++) {
			data[0] = bayerPixelSave(bayerData, bayerMask[0], x, y, bayerWidth, bayerWidth, bayerHeight);
			data[1] = bayerPixelSave(bayerData, bayerMask[1], x, y, bayerWidth, bayerWidth, bayerHeight);
			data[2] = bayerPixelSave(bayerData, bayerMask[2], x, y, bayerWidth, bayerWidth, bayerHeight);

			data += 3;
		}
	}

	for(int y=topBorder+border; y<bottomBorder-border; y++) {
		quint16* data = rgbImage->dataLine(y-topBorder);

		for(int x=leftBorder; x<leftBorder+border; x++) {
			data[0] = bayerPixelSave(bayerData, bayerMask[0], x, y, bayerWidth, bayerWidth, bayerHeight);
			data[1] = bayerPixelSave(bayerData, bayerMask[1], x, y, bayerWidth, bayerWidth, bayerHeight);
			data[2] = bayerPixelSave(bayerData, bayerMask[2], x, y, bayerWidth, bayerWidth, bayerHeight);

			data += 3;
		}
	}

	for(int y=topBorder+border; y<bottomBorder-border; y++) {
		quint16* data = rgbImage->dataOffset(rightBorder-border-leftBorder, y-topBorder);

		for(int x=rightBorder-border; x<rightBorder; x++) {
			data[0] = bayerPixelSave(bayerData, bayerMask[0], x, y, bayerWidth, bayerWidth, bayerHeight);
			data[1] = bayerPixelSave(bayerData, bayerMask[1], x, y, bayerWidth, bayerWidth, bayerHeight);
			data[2] = bayerPixelSave(bayerData, bayerMask[2], x, y, bayerWidth, bayerWidth, bayerHeight);

			data += 3;
		}
	}

	for(int y=topBorder+border; y<=bottomBorder-border; y++) {
		quint16* data = rgbImage->dataOffset(border, y-topBorder);

		for(int x=leftBorder+border; x<=rightBorder-border; x++) {
			data[0] = bayerPixel(bayerData, bayerMask[0], x, y, bayerWidth);
			data[1] = bayerPixel(bayerData, bayerMask[1], x, y, bayerWidth);
			data[2] = bayerPixel(bayerData, bayerMask[2], x, y, bayerWidth);

			data += 3;
		}		
	}

	return rgbImage;
}

ExposureFilter::ExposureFilter() {
}

ExposureFilter::~ExposureFilter() {
}

QString ExposureFilter::name() {
	return QString("ExposureFilter");
}

bool ExposureFilter::accepts(QString inputFormat, QString outputFormat) {
	return inputFormat == RgbImage48::format() && outputFormat == RgbImage48::format();
}

Image* ExposureFilter::filter(Image* input, QString outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	RgbImage48* rgbInput = qobject_cast<RgbImage48*>(input);

	int width = rgbInput->width();
	int height = rgbInput->height();

	RgbImage48* rgbImageOutput = new RgbImage48(width, height);

	rgbImageOutput->setColorProfile(CmsProfile::proPhotoLinearRgbProfile());
	rgbImageOutput->setMetadata(input->metadata());

	MetadataResource* metadata = input->metadata();
	Matrix3x3 matrix;
	double blackLevel = 0.0;
	double whiteLevel = 65535.0;
	//QString cameraProfile;
	MetadataNode* settingsCurrent = metadata->queryProperty(RawExposure::propertyName, QVariant("current"))->parent();
	QString whiteBalance = settingsCurrent->queryValue(RawExposure::propertyWhiteBalance).toString();
	double temperature = settingsCurrent->queryValue(RawExposure::propertyTemperature).toDouble();
	double tint = settingsCurrent->queryValue(RawExposure::propertyTint).toDouble();
	Spline toneCurveSpline = Spline(QtUtils::toTypedList<QPointF>(settingsCurrent->queryValues(RawExposure::propertyToneCurve)));

	/*if(_parameters.contains("cameraProfile"))
		cameraProfile = _parameters.value("cameraProfile").toString();*/

	blackLevel = metadata->queryValue(RawFile::propertyBlackLevel).toDouble();
	whiteLevel = metadata->queryValue(RawFile::propertyWhiteLevel).toDouble();

	double divisor = 1.0 / (whiteLevel - blackLevel);

	blackLevel *= divisor;

	/*CmsProfile* inProfile = 0;
	CmsProfile* outProfile = 0;
	CmsTransform* transform = 0;

	if(!cameraProfile.isEmpty()) {
		inProfile = new CmsProfile(cameraProfile);
		outProfile = CmsProfile::proPhotoLinearProfile();
		transform = new CmsTransform(inProfile, CmsTransform::Rgb16, outProfile, CmsTransform::Rgb16);
	} else {
	}*/

	ColorXYZ whitePointXYZ;

	if(whiteBalance == "manual")
		whitePointXYZ = TemperatureUtils::toxy(temperature, tint).toColorXYZ();
	else
		whitePointXYZ = RawUtils::cameraNeutralWhiteBalance(metadata).toColorXYZ();

	TemperatureUtils::fromxy(whitePointXYZ.toColorxy(), &temperature, &tint);

	Matrix3x3 cameraMatrix = RawUtils::cameraMatrix(metadata, temperature);
	Matrix3x3 whitePointMatrix = whitePointXYZ.toMatrix3x3();
	Matrix3x3 whitePointShift = CIEUtils::whitePointXYZ2WhitePointXYZ(Colorxy::whitePointD50().toColorXYZ(), whitePointXYZ);
	Matrix3x3 toTargetColorSpace = CIEUtils::fromXYZ2ProPhotoRgb();

	matrix = toTargetColorSpace * (cameraMatrix * whitePointShift).inverse();

	//quint16 buffer[width*3];

	double curve[65536];
	double d = 1.0 / 65535.0;

	for(int i=0; i<65536; i++) {
		double v = (double)i * d;

		v = toneCurveSpline.evaluate(qBound(0.0, v, 1.0));

		curve[i] = qBound(0.0, v, 1.0);
	}

	for(int y=0; y<height; y++) {
		quint16* dataInput = rgbInput->dataLine(y);
		quint16* dataOutput = rgbImageOutput->dataLine(y);

		for(int x=0; x<width; x++) {
			double p[3];

			p[0] = (double)dataInput[0];
			p[1] = (double)dataInput[1];
			p[2] = (double)dataInput[2];

			p[0] *= divisor;
			p[1] *= divisor;
			p[2] *= divisor;

			p[0] -= blackLevel;
			p[1] -= blackLevel;
			p[2] -= blackLevel;

			matrix.multiplyVector(p);

			dataOutput[0] = (quint16)(curve[(int)qBound(0.0, p[0]*65535.0, 65535.0)] * 65535.0);
			dataOutput[1] = (quint16)(curve[(int)qBound(0.0, p[1]*65535.0, 65535.0)] * 65535.0);
			dataOutput[2] = (quint16)(curve[(int)qBound(0.0, p[2]*65535.0, 65535.0)] * 65535.0);

			/*if(transform == 0) {*/
				/*dataOutput[0] = (quint16)qBound(0.0, p[0] * 65535.0, 65535.0);
				dataOutput[1] = (quint16)qBound(0.0, p[1] * 65535.0, 65535.0);
				dataOutput[2] = (quint16)qBound(0.0, p[2] * 65535.0, 65535.0);*/
			/*} else {
				buffer[x*3+0] = (quint16)qBound(0.0, p[0] * 65535.0, 65535.0);
				buffer[x*3+1] = (quint16)qBound(0.0, p[1] * 65535.0, 65535.0);
				buffer[x*3+2] = (quint16)qBound(0.0, p[2] * 65535.0, 65535.0);
			}*/

			dataInput += 3;
			dataOutput += 3;
		}

		/*if(transform != 0)
			transform->transform((void*)buffer, (void*)dataOutput, width);*/
	}

	/*if(transform != 0) {
		delete inProfile;
		delete outProfile;
		delete transform;
	}*/

	return rgbImageOutput;
}

QString DevelopFilter::parameterOutputProfile = "outputProfile";
QString DevelopFilter::profileSRgb = "sRgb";
QString DevelopFilter::profileProPhotoLinearRgb = "proPhotoLinearRgb";
QString DevelopFilter::profileAdobeRgb = "adobeRgb";

DevelopFilter::DevelopFilter() {
}

DevelopFilter::~DevelopFilter() {
}

QString DevelopFilter::name() {
	return QString("DevelopFilter");
}

bool DevelopFilter::accepts(QString inputFormat, QString outputFormat) {
	return inputFormat == RgbImage48::format() && outputFormat == RgbImage32::format();
}

Image* DevelopFilter::filter(Image* input, QString outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	RgbImage48* rgb48Image = qobject_cast<RgbImage48*>(input);

	int width = rgb48Image->width();
	int height = rgb48Image->height();

	QString profile = profileSRgb;

	if(_parameters.contains(parameterOutputProfile))
		profile = _parameters.value(parameterOutputProfile).toString();

	RgbImage32* rgb32Image = new RgbImage32(width, height);
	rgb32Image->setMetadata(input->metadata());

	if(profile == profileSRgb || profile == profileAdobeRgb) {
		if(profile == profileSRgb)
			rgb32Image->setColorProfile(CmsProfile::sRgbProfile());
		else if(profile == profileAdobeRgb)
			rgb32Image->setColorProfile(CmsProfile::adobeRgbProfile());

		CmsProfile* inProfile = rgb48Image->colorProfile();
		CmsProfile* outProfile = rgb32Image->colorProfile();
		CmsTransform* transform = new CmsTransform(inProfile, CmsTransform::Rgb16, outProfile, CmsTransform::Bgra8); 

		for(int y=0; y<height; y++) {
			void* data48 = (void*)rgb48Image->dataLine(y);
			void* data32 = (void*)rgb32Image->dataLine(y);

			memset(data32, 0xff, width*4);

			transform->transform(data48, data32, width);
		}

		delete transform;
	}

	return rgb32Image;
}
