/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef RAWFILE_H_
#define RAWFILE_H_

#include <QFile>
#include <QList>
#include <QString>
#include <QVector>

#include <ColorMath.h>
#include <Image.h>
#include <ImageFilter.h>
#include <Metadata.h>

class RawFile {
public:
	static QString namespaceUri;

	static MetadataQName propertyBodyId;
	static MetadataQName propertySensorCrop;
	static MetadataQName propertyBlackLevel;
	static MetadataQName propertyWhiteLevel;
	static MetadataQName propertyWhiteBalanceNeutral;
	static MetadataQName propertyCalibrationIlluminant1;
	static MetadataQName propertyCalibrationIlluminant2;
	static MetadataQName propertyColorMatrix1;
	static MetadataQName propertyColorMatrix2;
	static MetadataQName propertyCameraCalibration1;
	static MetadataQName propertyCameraCalibration2;
	static MetadataQName propertyAnalogBalance;
	// TODO: can be different for sraw!!!
	static MetadataQName propertyMaskedArea;

	virtual ~RawFile();

	virtual MetadataDocument* readMetadata() = 0;
	virtual Image* readImage() = 0;

	MetadataDocument* metadata();
	Image* image();

	static RawFile* fromFile(QFile* file);

	static bool readCameraDefinitions(QString folder);
	static MetadataDocument* cameraDefinitions();

protected:
	RawFile();

	MetadataDocument* _metadata;
	Image* _image;

private:
	static MetadataDocument* _cameraDefinitions;
};

class RawUtils {
public:
	static MetadataResource* findCameraDefinition(QString model);
	static void appendCameraDefinition(MetadataResource* resource, QString model);

	static Matrix3x3 cameraMatrix(MetadataResource* metadataResource, double temperature);
	static Colorxy cameraNeutralWhiteBalance(MetadataResource* metadataResource);
	static Colorxy cameraNeutralWhiteBalance(MetadataResource* metadataResource, ColorXYZ neutralWhiteBalance);

	static void appendBlackLevelFromMaskedArea(Image* image);

private:
	static double illuminant1(MetadataResource* metadataResource);
	static double illuminant2(MetadataResource* metadataResource);
	static Matrix3x3 colorMatrix(MetadataResource* metadataResource, double temperature);
	static Matrix3x3 cameraCalibration(MetadataResource* metadataResource, double temperature);
	static Matrix3x3 analogBalance(MetadataResource* metadataResource);
	static Matrix3x3 interpolateMatrix(Matrix3x3 a, Matrix3x3 b, double f);
};

class RawExposure {
public:
	static QString namespaceUri;

	static MetadataQName propertySettings;
	static MetadataQName propertyName;
	static MetadataQName propertyDate;
	static MetadataQName propertyWhiteBalance;
	static MetadataQName propertyTemperature;
	static MetadataQName propertyTint;
	static MetadataQName propertyToneCurve;
};

class BayerImage16 : public Image {

	Q_OBJECT

public:
	BayerImage16(int width, int height);
	virtual ~BayerImage16();

	quint16* data();
	quint16* dataLine(int y);
	quint16* dataOffset(int x, int y);

	int precision();
	void setPrecision(int precision);

	QVector<int> pattern();
	void setPattern(QVector<int> pattern);

	static QString format() {
		return "Bayer16";
	}

private:
	quint16* _data;
	int _precision;
	QVector<int> _pattern;
};

class DemosaicFilter : public ImageFilter {
public:
	DemosaicFilter();

	virtual ~DemosaicFilter();

	virtual QString name();

	virtual bool accepts(QString inputFormat, QString outputFormat);

	virtual Image* filter(Image* input, QString outputFormat);

private:
	static inline quint16 bayerPixel(quint16* d, int* bm, int x, int y, int s) {
		quint16* p = d+x+y*s;

		switch(bm[(x&1) | (y<<1&2)]) {
			case 0:
				return p[0];
			case 1:
				return (p[-1]+p[1]+1)/2;
			case 2:
				return (p[-s]+p[s]+1)/2;
			case 3:
				return (p[-s-1]+p[-s+1]+p[s-1]+p[s+1]+2)/4;
			case 4:
				return (p[-s]+p[-1]+p[1]+p[s]+2)/4;
			default:
				return 0;
		}
	}

	static inline void getPixelSave(quint16* d, int* v, int* c, int x, int y, int s, int w, int h) {
		if(x >= w || x < 0 || y >= h || y < 0)
			return;

		(*c)++;
		*v += *(d+x+y*s);
	}

	static inline quint16 bayerPixelSave(quint16* d, int* bm, int x, int y, int s, int w, int h) {
		int c = 0, v = 0;

		switch(bm[(x&1) | (y<<1&2)]) {
			case 0:
				getPixelSave(d, &v, &c, x, y, s, w, h);
				break;
			case 1:
				getPixelSave(d, &v, &c, x-1, y, s, w, h);
				getPixelSave(d, &v, &c, x+1, y, s, w, h);
				break;
			case 2:
				getPixelSave(d, &v, &c, x, y-1, s, w, h);
				getPixelSave(d, &v, &c, x, y+1, s, w, h);
				break;
			case 3:
				getPixelSave(d, &v, &c, x-1, y-1, s, w, h);
				getPixelSave(d, &v, &c, x+1, y-1, s, w, h);
				getPixelSave(d, &v, &c, x-1, y+1, s, w, h);
				getPixelSave(d, &v, &c, x+1, y+1, s, w, h);
				break;
			case 4:
				getPixelSave(d, &v, &c, x, y-1, s, w, h);
				getPixelSave(d, &v, &c, x-1, y, s, w, h);
				getPixelSave(d, &v, &c, x+1, y, s, w, h);
				getPixelSave(d, &v, &c, x, y+1, s, w, h);
				break;
			default:
				return 0;
		}

		if(c == 1)
			return v;
		else if(c == 2)
			return (v+1)/2;
		else if(c == 3)
			return (v+1)/3;
		else if(c == 4)
			return (v+2)/4;
		else
			return 0;
	}
};

class ExposureFilter : public ImageFilter {
public:
	ExposureFilter();
	virtual ~ExposureFilter();

	virtual QString name();

	virtual bool accepts(QString inputFormat, QString outputFormat);

	virtual Image* filter(Image* input, QString outputFormat);
};

class DevelopFilter : public ImageFilter {
public:
	static QString parameterOutputProfile;
	static QString profileSRgb;
	static QString profileProPhotoLinearRgb;
	static QString profileAdobeRgb;

	DevelopFilter();

	virtual ~DevelopFilter();

	virtual QString name();

	virtual bool accepts(QString inputFormat, QString outputFormat);

	virtual Image* filter(Image* input, QString outputFormat);
};

#endif /*RAWFILE_H_*/
