/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef TIFF_H_
#define TIFF_H_

#include <QByteArray>
#include <QDataStream>
#include <QIODevice>
#include <QList>
#include <QMap>
#include <QString>
#include <QVariant>

#include <Image.h>
#include <ImageFilter.h>
#include <Metadata.h>

class Tiff {
public:
	static QString namespaceUri;
	static MetadataQName propertyNewSubfileType;
	static MetadataQName propertyImageWidth;
	static MetadataQName propertyImageLength;
	static MetadataQName propertyBitsPerSample;
	static MetadataQName propertyCompression;
	static MetadataQName propertyPhotometricInterpretation;
	static MetadataQName propertyImageDescription;
	static MetadataQName propertyMake;
	static MetadataQName propertyModel;
	static MetadataQName propertyStripOffsets;
	static MetadataQName propertyOrientation;
	static MetadataQName propertySamplesPerPixel;
	static MetadataQName propertyRowsPerStrip;
	static MetadataQName propertyStripByteCounts;
	static MetadataQName propertyXResolution;
	static MetadataQName propertyYResolution;
	static MetadataQName propertyPlanarConfiguration;
	static MetadataQName propertyResolutionUnit;
	static MetadataQName propertyDateTime;
	static MetadataQName propertyArtist;
	static MetadataQName propertyTileWidth;
	static MetadataQName propertyTileLength;
	static MetadataQName propertyTileOffsets;
	static MetadataQName propertyTileByteCounts;
	static MetadataQName propertySubIFDs;
	static MetadataQName propertyXMP;
	static MetadataQName propertyCFARepeatPatternDim;
	static MetadataQName propertyCFAPattern;
	static MetadataQName propertyCopyright;
	static MetadataQName propertyIPTCNAA;
	static MetadataQName propertyExifIFD;
	static MetadataQName propertyICCProfile;
	static MetadataQName propertyTimeZoneOffset;
	static MetadataQName propertyLightSource;
	static MetadataQName propertyImageNumber;
	static MetadataQName propertySensingMethod;
	static MetadataQName propertyMakerNote;

	enum ByteOrder {
		ByteOrderLittleEndian = 0x4949,
		ByteOrderBigEndian = 0x4d4d
	};

	enum {
		MagicNumber = 42
	};

	enum Tag {
		TagNewSubfileType = 0xfe,
		TagImageWidth = 0x100,
		TagImageLength = 0x101,
		TagBitsPerSample = 0x102,
		TagCompression = 0x103,
		TagPhotometricInterpretation = 0x106,
		TagImageDescription = 0x10e,
		TagMake = 0x10f,
		TagModel = 0x110,
		TagStripOffsets = 0x111,
		TagOrientation = 0x112,
		TagSamplesPerPixel = 0x115,
		TagRowsPerStrip = 0x116,
		TagStripByteCounts = 0x117,
		TagXResolution = 0x11a,
		TagYResolution = 0x11b,
		TagPlanarConfiguration = 0x11c,
		TagResolutionUnit = 0x128,
		TagDateTime = 0x132,
		TagArtist = 0x13b,
		TagTileWidth = 0x142,
		TagTileLength = 0x143,
		TagTileOffsets = 0x144,
		TagTileByteCounts = 0x145,
		TagSubIFDs = 0x14a,
		TagJPEGInterchangeFormat = 0x201,
		TagJPEGInterchangeFormatLength = 0x202,
		TagXMP = 0x2bc,
		TagCFARepeatPatternDim = 0x828d,
		TagCFAPattern = 0x828e,
		TagCopyright = 0x8298,
		TagIPTCNAA = 0x83bb,
		TagExifIFD = 0x8769,
		TagICCProfile = 0x8773,
		TagTimeZoneOffset = 0x882a,
		TagLightSource = 0x9208,
		TagImageNumber = 0x9211,
		TagSensingMethod = 0x9217,
		// TODO: part of tiff or exif?
		TagMakerNote = 0x927c
	};

	enum Type {
		TypeByte = 1,
		TypeAscii = 2,
		TypeShort = 3,
		TypeLong = 4,
		TypeRational = 5,
		TypeSByte = 6,
		TypeUndefined = 7,
		TypeSShort = 8,
		TypeSLong = 9,
		TypeSRational = 10,
		TypeFloat = 11,
		TypeDouble = 12
	};

	struct TagSet {
		Tag tag;
		QString property;
		Type defaultType;
	};

	static TagSet tagSetTable[];

	static int typeSize(Type type);

	static double lightSourceTemperature(quint16 value);
};

class Exif {
public:
	static QString namespaceUri;
	static MetadataQName propertyExposureTime;
	static MetadataQName propertyFNumber;
	static MetadataQName propertyExposureProgram;
	static MetadataQName propertyISOSpeedRatings;
	static MetadataQName propertyExifVersion;
	static MetadataQName propertyDateTimeOriginal;
	static MetadataQName propertyDateTimeDigitized;
	static MetadataQName propertyComponentsConfiguration;
	static MetadataQName propertyShutterSpeedValue;
	static MetadataQName propertyApertureValue;
	static MetadataQName propertyExposureBiasValue;
	static MetadataQName propertyMaxApertureValue;
	static MetadataQName propertyMeteringMode;
	static MetadataQName propertyFlash;
	static MetadataQName propertyFocalLength;
	static MetadataQName propertyUserComment;
	static MetadataQName propertySubsecTime;
	static MetadataQName propertySubsecTimeOriginal;
	static MetadataQName propertySubsecTimeDigitized;
	static MetadataQName propertyFlashPixVersion;
	static MetadataQName propertyColorSpace;
	static MetadataQName propertyPixelXDimension;
	static MetadataQName propertyPixelYDimension;
	static MetadataQName propertyFocalPlaneXResolution;
	static MetadataQName propertyFocalPlaneYResolution;
	static MetadataQName propertyFocalPlaneResolutionUnit;
	static MetadataQName propertyCustomRendered;
	static MetadataQName propertyExposureMode;
	static MetadataQName propertyWhiteBalance;
	static MetadataQName propertySceneCaptureType;

	enum Tag {
		TagExposureTime = 0x829a,
		TagFNumber = 0x829d,
		TagExposureProgram = 0x8822,
		TagISOSpeedRatings = 0x8827,
		TagExifVersion = 0x9000,
		TagDateTimeOriginal = 0x9003,
		TagDateTimeDigitized = 0x9004,
		TagComponentsConfiguration = 0x9101,
		TagShutterSpeedValue = 0x9201,
		TagApertureValue = 0x9202,
		TagExposureBiasValue = 0x9204,
		TagMaxApertureValue = 0x9205,
		TagMeteringMode = 0x9207,
		TagFlash = 0x9209,
		TagFocalLength = 0x920a,
		TagUserComment = 0x9286,
		TagSubsecTime = 0x9290,
		TagSubsecTimeOriginal = 0x9291,
		TagSubsecTimeDigitized = 0x9292,
		TagFlashPixVersion = 0xa000,
		TagColorSpace = 0xa001,
		TagPixelXDimension = 0xa002,
		TagPixelYDimension = 0xa003,
		TagInteroperabilityIFDPointer = 0xa005,
		TagFocalPlaneXResolution = 0xa20e,
		TagFocalPlaneYResolution = 0xa20f,
		TagFocalPlaneResolutionUnit = 0xa210,
		TagCustomRendered = 0xa401,
		TagExposureMode = 0xa402,
		TagWhiteBalance = 0xa403,
		TagSceneCaptureType = 0xa406
	};

	struct TagSet {
		Tag tag;
		QString property;
		Tiff::Type defaultType;
	};

	static TagSet tagSetTable[];
};

class TiffEntry {
public:
	TiffEntry(quint16 tag, Tiff::Type type, QVariant value);
	TiffEntry(quint16 tag, Tiff::Type type, QList<QVariant> values);

	quint16 tag();
	Tiff::Type type();
	QList<QVariant> values();

	quint32 valueCount();
	quint32 valueSize();

private:
	quint16 _tag;
	Tiff::Type _type;
	QList<QVariant> _values;
};

class TiffDirectory {
public:
	void addTag(TiffEntry* entry);

	QMap<quint16, TiffEntry*> entryMap();

	quint32 headerSize();
	quint32 valueSize();
	quint32 size();

private:
	QMap<quint16, TiffEntry*> _entries;
};

class TiffParser {
public:
	static QMap<int, QString> createTiffTagNameMap();
	static QMap<int, QString> createExifTagNameMap();

	TiffParser(QIODevice* device);
	virtual ~TiffParser();

	MetadataDocument* parse();

	MetadataDocument* document();

	quint32 directoryOffset(int index);
	quint32 directoryLength(int index);

protected:
	static QMap<int, QString> _tiffTagNameMap;
	static QMap<int, QString> _exifTagNameMap;

	QDataStream* _dataStream;
	MetadataDocument* _document;
	QList<quint32> _offset;
	QList<quint32> _length;

	void detectByteOrder();

	quint8 readByte();
	quint16 readShort();
	quint32 readLong();
	Rational readRational();
	qint8 readSByte();
	qint16 readSShort();
	qint32 readSLong();
	Rational readSRational();
	float readFloat();
	double readDouble();

	QList<QVariant> readValues(quint16 tag, quint16 type, quint32 count);

	void parseEntry(MetadataNode* parent, quint32 offset);
	void parseExif(MetadataNode* parent, quint32 offset);
	void parseDirectory(MetadataNode* parent, quint32 offset);
	void parseHeader(MetadataNode* parent, quint32 offset=0);

	virtual MetadataQName tagQName(quint16 tag);
	virtual QVariant tagValue(int tag, QVariant value);

	virtual void parseMakerNote(MetadataNode* parent, quint32 offset);
};

class TiffWriter : public ImageWriter {
public:
	static QString name;

	static QMap<QString, Tiff::TagSet> createTiffTagSetMap();
	static QMap<QString, Exif::TagSet> createExifTagSetMap();

	TiffWriter();
	virtual ~TiffWriter();

	virtual bool accepts(QString format);

	virtual bool write(QIODevice* device, Image* image, MetadataResource* resource, BergPhoto::Parameters parameters);

protected:
	static QMap<QString, Tiff::TagSet> _tiffTagSetMap;
	static QMap<QString, Exif::TagSet> _exifTagSetMap;

	void writeHeader(QDataStream* dataStream);
	void writeIdf(QDataStream* dataStream, TiffDirectory* directory, QByteArray data=QByteArray(), quint32 nextIdf=0);
	void writeValues(QDataStream* dataStream, TiffEntry* entry);
	void writeRational(QDataStream* dataStream, Rational value);
	void writeSRational(QDataStream* dataStream, Rational value);

	void addTiffMetadataTags(TiffDirectory* directory, MetadataResource* resource);
	void addExifMetadataTags(TiffDirectory* directory, MetadataResource* resource);

	virtual QVariant tagValue(int tag, QVariant value);
};

class TiffSimpleMetadataTransform : public MetadataTransform {
public:
	virtual ~TiffSimpleMetadataTransform();

	virtual MetadataNode* transform(MetadataNode* source);
};

#endif /* TIFF_H_ */
