/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "ImageFilter.h"

#include <QByteArray>
#include <QFile>
#include <QImageWriter>

#include <math.h>

ImageFilter::~ImageFilter() {
}

BergPhoto::Parameters ImageFilter::parameters() {
	return _parameters;
}

void ImageFilter::setParameters(BergPhoto::Parameters parameters) {
	_parameters = parameters;
}

QString ImageWriter::parameterImageWriter = "imageWriter";
QString ImageWriter::parameterFileFormat = "fileFormat";
QString ImageWriter::parameterCompress = "compress";
QString ImageWriter::parameterQuality = "quality";

bool ImageWriter::write(QString fileName, Image* image, MetadataResource* resource, BergPhoto::Parameters parameter) {
	QFile file(fileName);

	file.open(QIODevice::WriteOnly);

	bool r = write(&file, image, resource, parameter);

	file.close();

	return r;
}

ResizeFilter::ResizeFilter() {
}

ResizeFilter::~ResizeFilter() {
}

QString ResizeFilter::name() {
	return QString("ResizeFilter");
}

bool ResizeFilter::accepts(QString inputFormat, QString outputFormat) {
	return inputFormat == RgbImage48::format() && outputFormat == RgbImage48::format();
}

Image* ResizeFilter::filter(Image* input, QString outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	int sourceWidth = input->width();
	int sourceHeight = input->height();
	int targetWidth = _parameters.value("targetWidth").toInt();
	int targetHeight = _parameters.value("targetHeight").toInt();

	if(_parameters.contains("scale")) {
		double scale = _parameters.value("scale").toDouble();

		targetWidth = (int)((double)sourceWidth*scale);
		targetHeight = (int)((double)sourceHeight*scale);
	}

	RgbImage48* sourceImage = qobject_cast<RgbImage48*>(input);
	RgbImage48* targetImage = new RgbImage48(targetWidth, targetHeight);

	targetImage->setMetadata(sourceImage->metadata());

	for(int y=0; y<targetHeight; y++) {
		quint16* sourceLine = sourceImage->dataLine(y*sourceHeight/targetHeight);
		quint16* targetLine = targetImage->dataLine(y);

		for(int x=0; x<targetWidth; x++) {
			int offset = x*sourceWidth/targetWidth*3;

			targetLine[0] = sourceLine[offset];
			targetLine[1] = sourceLine[offset+1];
			targetLine[2] = sourceLine[offset+2];

			targetLine += 3;
		}
	}

	return targetImage;
}

QString RotateCropFilter::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/filter/rotatecrop/1.0/";
MetadataQName RotateCropFilter::propertyRotateAngle = MetadataQName(RotateCropFilter::namespaceUri, "RotateAngle");
MetadataQName RotateCropFilter::propertyCropLeft = MetadataQName(RotateCropFilter::namespaceUri, "CropLeft");
MetadataQName RotateCropFilter::propertyCropRight = MetadataQName(RotateCropFilter::namespaceUri, "CropRight");
MetadataQName RotateCropFilter::propertyCropTop = MetadataQName(RotateCropFilter::namespaceUri, "CropTop");
MetadataQName RotateCropFilter::propertyCropBottom = MetadataQName(RotateCropFilter::namespaceUri, "CropBottom");

RotateCropFilter::RotateCropFilter() {
}

RotateCropFilter::~RotateCropFilter() {
}

QString RotateCropFilter::name() {
	return QString("RotateCropFilter");
}

bool RotateCropFilter::accepts(QString inputFormat, QString outputFormat) {
	return inputFormat == RgbImage48::format() && outputFormat == RgbImage48::format();
}

Image* RotateCropFilter::filter(Image* input, QString outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	RgbImage48* source = qobject_cast<RgbImage48*>(input);

	double angle = input->metadata()->queryValue(RotateCropFilter::propertyRotateAngle).toDouble() * (M_PI/180.0);
	int cropLeft = input->metadata()->queryValue(RotateCropFilter::propertyCropLeft).toInt();
	int cropRight = input->metadata()->queryValue(RotateCropFilter::propertyCropRight).toInt();
	int cropTop = input->metadata()->queryValue(RotateCropFilter::propertyCropTop).toInt();
	int cropBottom = input->metadata()->queryValue(RotateCropFilter::propertyCropBottom).toInt();

	int interpolate = InterpolateNearest;
	QString interpolateString = _parameters.value("interpolation").toString();
	if(interpolateString == "linear")
		interpolate = InterpolateLinear;
	else if(interpolateString == "cubic")
		interpolate = InterpolateCubic;

	if(_parameters.contains("scale")) {
		double scale = _parameters.value("scale").toDouble();

		cropLeft = (int)((double)cropLeft*scale);
		cropRight = (int)((double)cropRight*scale);
		cropTop = (int)((double)cropTop*scale);
		cropBottom = (int)((double)cropBottom*scale);
	}

	RgbImage48* target = 0;

	if(angle < 0.0 || angle >= M_PI*2.0)
		angle = angle - floor(angle/M_PI*0.5)*M_PI*2.0;

	if(fabs(angle) < 0.0001)
		target = crop(source, cropLeft, cropRight, cropTop, cropBottom);
	else if(fabs(angle - M_PI*0.5) < 0.0001)
		target = rotateLeftCrop(source, cropLeft, cropRight, cropTop, cropBottom);
	else if(fabs(angle - M_PI) < 0.0001)
		target = rotateHalfCrop(source, cropLeft, cropRight, cropTop, cropBottom);
	else if(fabs(angle - M_PI*1.5) < 0.0001)
		target = rotateRightCrop(source, cropLeft, cropRight, cropTop, cropBottom);
	else
		target = rotateCrop(source, angle, interpolate, cropLeft, cropRight, cropTop, cropBottom);

	target->setMetadata(source->metadata());

	return target;
}

void RotateCropFilter::nearestNeighborInterpolation(RgbImage48* source, quint16* targetPoint, double x, double y) {
	quint16* sourcePoint = source->dataOffsetSave((int)(x+0.5), (int)(y+0.5));

	if(sourcePoint != 0) {
		targetPoint[0] = sourcePoint[0];
		targetPoint[1] = sourcePoint[1];
		targetPoint[2] = sourcePoint[2];
	} else {
		targetPoint[0] = 0;
		targetPoint[1] = 0;
		targetPoint[2] = 0;
	}
}

quint16 RotateCropFilter::bilinearInterpolation(quint16 p0, quint16 p1, quint16 p2, quint16 p3, double dx, double dy) {
	return (quint16)qBound(0.0,(1-dy)*(p0+dx*(p1-p0))+dy*(p2+dx*(p3-p2))+0.5,65535.0);
}

void RotateCropFilter::bilinearInterpolation(RgbImage48* source, quint16* targetPoint, double x, double y) {
	int x0 = (int)x;
	int y0 = (int)y;
	quint16* s0 = source->dataOffsetSave(x0, y0);
	int x1 = x0+1;
	int y1 = y0;
	quint16* s1 = source->dataOffsetSave(x1, y1);
	int x2 = x0;
	int y2 = y0+1;
	quint16* s2 = source->dataOffsetSave(x2, y2);
	int x3 = x0+1;
	int y3 = y0+1;
	quint16* s3 = source->dataOffsetSave(x3, y3);

	quint16 black[] = { 0, 0, 0 };

	s0 = s0 != 0 ? s0 : black;
	s1 = s1 != 0 ? s1 : black;
	s2 = s2 != 0 ? s2 : black;
	s3 = s3 != 0 ? s3 : black;

	double dx = x-x0;
	double dy = y-y0;

	targetPoint[0] = bilinearInterpolation(s0[0], s1[0], s2[0], s3[0], dx, dy);
	targetPoint[1] = bilinearInterpolation(s0[1], s1[1], s2[1], s3[1], dx, dy);
	targetPoint[2] = bilinearInterpolation(s0[2], s1[2], s2[2], s3[2], dx, dy);
}

double RotateCropFilter::cubicInterpolation(quint16 p0, quint16 p1, quint16 p2, quint16 p3, double d) {
	return ((((-p0+3*p1-3*p2+p3)*d+(2*p0-5*p1+4*p2-p3))*d+(-p0+p2))*d+(p1+p1))/2.0;
}

quint16 RotateCropFilter::cubicInterpolation(double p0, double p1, double p2, double p3, double d) {
	return (quint16)qBound(0.0,((((-p0+3*p1-3*p2+p3)*d+(2*p0-5*p1+4*p2-p3))*d+(-p0+p2))*d+(p1+p1))/2.0+0.5,65535.0);
}

void RotateCropFilter::bicubicInterpolation(RgbImage48* source, quint16* targetPoint, double x, double y) {
	int xi = (int)x;
	int yi = (int)y;
	double col[3][4];
	double dx = x-xi;
	double dy = y-yi;
	quint16 black[] = { 0, 0, 0 };

	for(int i=-1; i<=2; i++) {
		quint16* s0 = source->dataOffsetSave(xi-1, yi+i);
		quint16* s1 = source->dataOffsetSave(xi, yi+i);
		quint16* s2 = source->dataOffsetSave(xi+1, yi+i);
		quint16* s3 = source->dataOffsetSave(xi+2, yi+i);

		s0 = s0 != 0 ? s0 : black;
		s1 = s1 != 0 ? s1 : black;
		s2 = s2 != 0 ? s2 : black;
		s3 = s3 != 0 ? s3 : black;

		col[0][i+2] = cubicInterpolation(s0[0], s1[0], s2[0], s3[0], dx);
		col[1][i+2] = cubicInterpolation(s0[1], s1[1], s2[1], s3[1], dx);
		col[2][i+2] = cubicInterpolation(s0[2], s1[2], s2[2], s3[2], dx);
	}

	targetPoint[0] = cubicInterpolation(col[0][0], col[0][1], col[0][2], col[0][3], dy);
	targetPoint[1] = cubicInterpolation(col[1][0], col[1][1], col[1][2], col[1][3], dy);
	targetPoint[2] = cubicInterpolation(col[2][0], col[2][1], col[2][2], col[2][3], dy);
}

RgbImage48* RotateCropFilter::crop(RgbImage48* source, int cropLeft, int cropRight, int cropTop, int cropBottom) {
	int targetWidth = source->width()-cropLeft-cropRight;
	int targetHeight = source->height()-cropTop-cropBottom;

	RgbImage48* target = new RgbImage48(targetWidth, targetHeight);

	for(int y=0; y<targetHeight; y++) {
		quint16* sourceLine = source->dataOffset(cropLeft, y+cropTop);
		quint16* targetLine = target->dataLine(y);

		memcpy(targetLine, sourceLine, targetWidth*6);
	}

	return target;
}

RgbImage48* RotateCropFilter::rotateLeftCrop(RgbImage48* source, int cropLeft, int cropRight, int cropTop, int cropBottom) {
	int targetWidth = source->height()-cropLeft-cropRight;
	int targetHeight = source->width()-cropTop-cropBottom;

	RgbImage48* target = new RgbImage48(targetWidth, targetHeight);

	for(int y=0; y<targetHeight; y++) {
		quint16* targetPoint = target->dataLine(targetHeight-y-1);

		for(int x=0; x<targetWidth; x++) {
			quint16* sourcePoint = source->dataOffset(y+cropBottom, x+cropLeft);

			targetPoint[0] = sourcePoint[0];
			targetPoint[1] = sourcePoint[1];
			targetPoint[2] = sourcePoint[2];

			targetPoint += 3;
		}
	}

	return target;
}

RgbImage48* RotateCropFilter::rotateHalfCrop(RgbImage48* source, int cropLeft, int cropRight, int cropTop, int cropBottom) {
	int targetWidth = source->width()-cropLeft-cropRight;
	int targetHeight = source->height()-cropTop-cropBottom;

	RgbImage48* target = new RgbImage48(targetWidth, targetHeight);

	for(int y=0; y<targetHeight; y++) {
		quint16* sourcePoint = source->dataOffset(cropRight, y+cropBottom);
		quint16* targetPoint = target->dataOffset(targetWidth-1, targetHeight-y-1);

		for(int x=0; x<targetWidth; x++) {
			targetPoint[0] = sourcePoint[0];
			targetPoint[1] = sourcePoint[1];
			targetPoint[2] = sourcePoint[2];

			sourcePoint += 3;
			targetPoint -= 3;
		}
	}

	return target;
}

RgbImage48* RotateCropFilter::rotateRightCrop(RgbImage48* source, int cropLeft, int cropRight, int cropTop, int cropBottom) {
	int targetWidth = source->height()-cropLeft-cropRight;
	int targetHeight = source->width()-cropTop-cropBottom;

	RgbImage48* target = new RgbImage48(targetWidth, targetHeight);

	for(int y=0; y<targetHeight; y++) {
		quint16* targetPoint = target->dataOffset(targetWidth-1, y);

		for(int x=0; x<targetWidth; x++) {
			quint16* sourcePoint = source->dataOffset(y+cropTop, x+cropRight);

			targetPoint[0] = sourcePoint[0];
			targetPoint[1] = sourcePoint[1];
			targetPoint[2] = sourcePoint[2];

			targetPoint -= 3;
		}
	}

	return target;
}

RgbImage48* RotateCropFilter::rotateCrop(RgbImage48* source, double angle, int interpolation, int cropLeft, int cropRight, int cropTop, int cropBottom) {
	double mc = cos(angle);
	double ms = sin(angle);

	int targetWidth = (int)(fabs(source->width()*mc) + fabs(source->height()*ms)+0.5);
	int targetWidth2 = targetWidth/2;
	int targetHeight = (int)(fabs(source->height()*mc) + fabs(source->width()*ms)+0.5);
	int targetHeight2 = targetHeight/2;
	int sourceWidth2 = source->width()/2;
	int sourceHeight2 = source->height()/2;

	void (*interpolateFunc)(RgbImage48*, quint16*, double, double);

	switch(interpolation) {
		case InterpolateLinear:
			interpolateFunc = bilinearInterpolation;
			break;

		case InterpolateCubic:
			interpolateFunc = bicubicInterpolation;
			break;

		default:
			interpolateFunc = nearestNeighborInterpolation;
			break;
	}

	RgbImage48* target = new RgbImage48(targetWidth-cropLeft-cropRight, targetHeight-cropTop-cropBottom);

	for(int y=-targetHeight2+cropTop; y<targetHeight2-cropBottom; y++) {
		quint16* targetPoint = target->dataLine(y+targetHeight2-cropTop);

		for(int x=-targetWidth2+cropLeft; x<targetWidth2-cropRight; x++) {
			double dx = x*mc - y*ms + sourceWidth2;
			double dy = x*ms + y*mc + sourceHeight2;

			interpolateFunc(source, targetPoint, dx, dy);

			targetPoint += 3;
		}
	}

	return target;
}

CropFilter::CropFilter() {
}

CropFilter::~CropFilter() {
}

QString CropFilter::name() {
	return QString("CropFilter");
}

bool CropFilter::accepts(QString inputFormat, QString outputFormat) {
	return inputFormat == RgbImage48::format() && outputFormat == RgbImage48::format();
}

Image* CropFilter::filter(Image* input, QString outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	RgbImage48* source = qobject_cast<RgbImage48*>(input);

	int xoffset = 0;
	int yoffset = 0;
	int targetWidth = 0;
	int targetHeight = 0;

	RgbImage48* target = new RgbImage48(targetWidth, targetHeight);

	for(int y=0; y<targetHeight; y++) {
		quint16* targetLine = target->dataLine(y);
		quint16* sourceLine = source->dataOffset(xoffset, y+yoffset);

		memcpy(targetLine, sourceLine, targetWidth*6);
	}

	return target;
}

QString QtImageWriter::name = "QtImageWriter";

bool QtImageWriter::accepts(QString format) {
	return format == RgbImage32::format();
}

bool QtImageWriter::write(QIODevice* device, Image* image, MetadataResource* resource, BergPhoto::Parameters parameters) {
	if(!accepts(image->format()))
		return false;

	QImageWriter imageWriter(device, QByteArray());

	if(parameters.contains(ImageWriter::parameterCompress))
		imageWriter.setCompression(parameters.value(ImageWriter::parameterCompress).toBool() ? 1 : 0);
	else
		imageWriter.setCompression(0);

	if(parameters.contains(ImageWriter::parameterQuality))
		imageWriter.setQuality(parameters.value(ImageWriter::parameterQuality).toInt());
	else
		imageWriter.setQuality(100);

	return imageWriter.write(*qobject_cast<RgbImage32*>(image)->toQImage());
}
