/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "Tiff.h"

#include <QtUtils.h>
#include <RawFile.h>

QString Tiff::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/tiff/1.0/";
MetadataQName Tiff::propertyNewSubfileType = MetadataQName(Tiff::namespaceUri, "NewSubfileType");
MetadataQName Tiff::propertyImageWidth = MetadataQName(Tiff::namespaceUri, "ImageWidth");
MetadataQName Tiff::propertyImageLength = MetadataQName(Tiff::namespaceUri, "ImageLength");
MetadataQName Tiff::propertyBitsPerSample = MetadataQName(Tiff::namespaceUri, "BitsPerSample");
MetadataQName Tiff::propertyCompression = MetadataQName(Tiff::namespaceUri, "Compression");
MetadataQName Tiff::propertyPhotometricInterpretation = MetadataQName(Tiff::namespaceUri, "PhotometricInterpretation");
MetadataQName Tiff::propertyImageDescription = MetadataQName(Tiff::namespaceUri, "ImageDescription");
MetadataQName Tiff::propertyMake = MetadataQName(Tiff::namespaceUri, "Make");
MetadataQName Tiff::propertyModel = MetadataQName(Tiff::namespaceUri, "Model");
MetadataQName Tiff::propertyStripOffsets = MetadataQName(Tiff::namespaceUri, "StripOffsets");
MetadataQName Tiff::propertyOrientation = MetadataQName(Tiff::namespaceUri, "Orientation");
MetadataQName Tiff::propertySamplesPerPixel = MetadataQName(Tiff::namespaceUri, "SamplesPerPixel");
MetadataQName Tiff::propertyRowsPerStrip = MetadataQName(Tiff::namespaceUri, "RowsPerStrip");
MetadataQName Tiff::propertyStripByteCounts = MetadataQName(Tiff::namespaceUri, "StripByteCounts");
MetadataQName Tiff::propertyXResolution = MetadataQName(Tiff::namespaceUri, "XResolution");
MetadataQName Tiff::propertyYResolution = MetadataQName(Tiff::namespaceUri, "YResolution");
MetadataQName Tiff::propertyPlanarConfiguration = MetadataQName(Tiff::namespaceUri, "PlanarConfiguration");
MetadataQName Tiff::propertyResolutionUnit = MetadataQName(Tiff::namespaceUri, "ResolutionUnit");
MetadataQName Tiff::propertyDateTime = MetadataQName(Tiff::namespaceUri, "DateTime");
MetadataQName Tiff::propertyArtist = MetadataQName(Tiff::namespaceUri, "Artist");
MetadataQName Tiff::propertyTileWidth = MetadataQName(Tiff::namespaceUri, "TileWidth");
MetadataQName Tiff::propertyTileLength = MetadataQName(Tiff::namespaceUri, "TileLength");
MetadataQName Tiff::propertyTileOffsets = MetadataQName(Tiff::namespaceUri, "TileOffsets");
MetadataQName Tiff::propertyTileByteCounts = MetadataQName(Tiff::namespaceUri, "TileByteCounts");
MetadataQName Tiff::propertySubIFDs = MetadataQName(Tiff::namespaceUri, "SubIFDs");
MetadataQName Tiff::propertyXMP = MetadataQName(Tiff::namespaceUri, "XMP");
MetadataQName Tiff::propertyCFARepeatPatternDim = MetadataQName(Tiff::namespaceUri, "CFARepeatPatternDim");
MetadataQName Tiff::propertyCFAPattern = MetadataQName(Tiff::namespaceUri, "CFAPattern");
MetadataQName Tiff::propertyCopyright = MetadataQName(Tiff::namespaceUri, "Copyright");
MetadataQName Tiff::propertyIPTCNAA = MetadataQName(Tiff::namespaceUri, "IPTCNAA");
MetadataQName Tiff::propertyExifIFD = MetadataQName(Tiff::namespaceUri, "ExifIFD");
MetadataQName Tiff::propertyICCProfile = MetadataQName(Tiff::namespaceUri, "ICCProfile");
MetadataQName Tiff::propertyTimeZoneOffset = MetadataQName(Tiff::namespaceUri, "TimeZoneOffset");
MetadataQName Tiff::propertyLightSource = MetadataQName(Tiff::namespaceUri, "LightSource");
MetadataQName Tiff::propertyImageNumber = MetadataQName(Tiff::namespaceUri, "ImageNumber");
MetadataQName Tiff::propertySensingMethod = MetadataQName(Tiff::namespaceUri, "SensingMethod");
MetadataQName Tiff::propertyMakerNote = MetadataQName(Tiff::namespaceUri, "MakerNote");

Tiff::TagSet Tiff::tagSetTable[] = {
	{ Tiff::TagNewSubfileType, Tiff::propertyNewSubfileType.localName(), Tiff::TypeLong },
	{ Tiff::TagImageWidth, Tiff::propertyImageWidth.localName(), Tiff::TypeLong },
	{ Tiff::TagImageLength, Tiff::propertyImageLength.localName(), Tiff::TypeLong },
	{ Tiff::TagBitsPerSample, Tiff::propertyBitsPerSample.localName(), Tiff::TypeShort },
	{ Tiff::TagCompression, Tiff::propertyCompression.localName(), Tiff::TypeShort },
	{ Tiff::TagPhotometricInterpretation, Tiff::propertyPhotometricInterpretation.localName(), Tiff::TypeShort },
	{ Tiff::TagImageDescription, Tiff::propertyImageDescription.localName(), Tiff::TypeAscii },
	{ Tiff::TagMake, Tiff::propertyMake.localName(), Tiff::TypeAscii },
	{ Tiff::TagModel, Tiff::propertyModel.localName(), Tiff::TypeAscii },
	{ Tiff::TagStripOffsets, Tiff::propertyStripOffsets.localName(), Tiff::TypeLong },
	{ Tiff::TagOrientation, Tiff::propertyOrientation.localName(), Tiff::TypeShort },
	{ Tiff::TagSamplesPerPixel, Tiff::propertySamplesPerPixel.localName(), Tiff::TypeShort },
	{ Tiff::TagRowsPerStrip, Tiff::propertyRowsPerStrip.localName(), Tiff::TypeLong },
	{ Tiff::TagStripByteCounts, Tiff::propertyStripByteCounts.localName(), Tiff::TypeLong },
	{ Tiff::TagXResolution, Tiff::propertyXResolution.localName(), Tiff::TypeRational },
	{ Tiff::TagYResolution, Tiff::propertyYResolution.localName(), Tiff::TypeRational },
	{ Tiff::TagPlanarConfiguration, Tiff::propertyPlanarConfiguration.localName(), Tiff::TypeShort },
	{ Tiff::TagResolutionUnit, Tiff::propertyResolutionUnit.localName(), Tiff::TypeShort },
	{ Tiff::TagDateTime, Tiff::propertyDateTime.localName(), Tiff::TypeAscii },
	{ Tiff::TagArtist, Tiff::propertyArtist.localName(), Tiff::TypeAscii },
	{ Tiff::TagTileWidth, Tiff::propertyTileWidth.localName(), Tiff::TypeLong },
	{ Tiff::TagTileLength, Tiff::propertyTileLength.localName(), Tiff::TypeLong },
	{ Tiff::TagTileOffsets, Tiff::propertyTileOffsets.localName(), Tiff::TypeLong },
	{ Tiff::TagTileByteCounts, Tiff::propertyTileByteCounts.localName(), Tiff::TypeLong },
	{ Tiff::TagSubIFDs, Tiff::propertySubIFDs.localName(), Tiff::TypeLong },
	{ Tiff::TagXMP, Tiff::propertyXMP.localName(), Tiff::TypeByte },
	{ Tiff::TagCFARepeatPatternDim, Tiff::propertyCFARepeatPatternDim.localName(), Tiff::TypeShort },
	{ Tiff::TagCFAPattern, Tiff::propertyCFAPattern.localName(), Tiff::TypeByte },
	{ Tiff::TagCopyright, Tiff::propertyCopyright.localName(), Tiff::TypeAscii },
	{ Tiff::TagIPTCNAA, Tiff::propertyIPTCNAA.localName(), Tiff::TypeLong },
	{ Tiff::TagExifIFD, Tiff::propertyExifIFD.localName(), Tiff::TypeLong },
	{ Tiff::TagICCProfile, Tiff::propertyICCProfile.localName(), Tiff::TypeUndefined },
	{ Tiff::TagTimeZoneOffset, Tiff::propertyTimeZoneOffset.localName(), Tiff::TypeSShort },
	{ Tiff::TagLightSource, Tiff::propertyLightSource.localName(), Tiff::TypeShort },
	{ Tiff::TagImageNumber, Tiff::propertyImageNumber.localName(), Tiff::TypeLong },
	{ Tiff::TagSensingMethod, Tiff::propertySensingMethod.localName(), Tiff::TypeShort },
	{ Tiff::TagMakerNote, Tiff::propertyMakerNote.localName(), Tiff::TypeLong }
};

QString Exif::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/exif/1.0/";
MetadataQName Exif::propertyExposureTime = MetadataQName(Exif::namespaceUri, "ExposureTime");
MetadataQName Exif::propertyFNumber = MetadataQName(Exif::namespaceUri, "FNumber");
MetadataQName Exif::propertyExposureProgram = MetadataQName(Exif::namespaceUri, "ExposureProgram");
MetadataQName Exif::propertyISOSpeedRatings = MetadataQName(Exif::namespaceUri, "ISOSpeedRatings");
MetadataQName Exif::propertyExifVersion = MetadataQName(Exif::namespaceUri, "ExifVersion");
MetadataQName Exif::propertyDateTimeOriginal = MetadataQName(Exif::namespaceUri, "DateTimeOriginal");
MetadataQName Exif::propertyDateTimeDigitized = MetadataQName(Exif::namespaceUri, "DateTimeDigitized");
MetadataQName Exif::propertyComponentsConfiguration = MetadataQName(Exif::namespaceUri, "ComponentsConfiguration");
MetadataQName Exif::propertyShutterSpeedValue = MetadataQName(Exif::namespaceUri, "ShutterSpeedValue");
MetadataQName Exif::propertyApertureValue = MetadataQName(Exif::namespaceUri, "ApertureValue");
MetadataQName Exif::propertyExposureBiasValue = MetadataQName(Exif::namespaceUri, "ExposureBiasValue");
MetadataQName Exif::propertyMaxApertureValue = MetadataQName(Exif::namespaceUri, "MaxApertureValue");
MetadataQName Exif::propertyMeteringMode = MetadataQName(Exif::namespaceUri, "MeteringMode");
MetadataQName Exif::propertyFlash = MetadataQName(Exif::namespaceUri, "Flash");
MetadataQName Exif::propertyFocalLength = MetadataQName(Exif::namespaceUri, "FocalLength");
MetadataQName Exif::propertyUserComment = MetadataQName(Exif::namespaceUri, "UserComment");
MetadataQName Exif::propertySubsecTime = MetadataQName(Exif::namespaceUri, "SubsecTime");
MetadataQName Exif::propertySubsecTimeOriginal = MetadataQName(Exif::namespaceUri, "SubsecTimeOriginal");
MetadataQName Exif::propertySubsecTimeDigitized = MetadataQName(Exif::namespaceUri, "SubsecTimeDigitized");
MetadataQName Exif::propertyFlashPixVersion = MetadataQName(Exif::namespaceUri, "FlashPixVersion");
MetadataQName Exif::propertyColorSpace = MetadataQName(Exif::namespaceUri, "ColorSpace");
MetadataQName Exif::propertyPixelXDimension = MetadataQName(Exif::namespaceUri, "PixelXDimension");
MetadataQName Exif::propertyPixelYDimension = MetadataQName(Exif::namespaceUri, "PixelYDimension");
MetadataQName Exif::propertyFocalPlaneXResolution = MetadataQName(Exif::namespaceUri, "FocalPlaneXResolution");
MetadataQName Exif::propertyFocalPlaneYResolution = MetadataQName(Exif::namespaceUri, "FocalPlaneYResolution");
MetadataQName Exif::propertyFocalPlaneResolutionUnit = MetadataQName(Exif::namespaceUri, "FocalPlaneResolutionUnit");
MetadataQName Exif::propertyCustomRendered = MetadataQName(Exif::namespaceUri, "CustomRendered");
MetadataQName Exif::propertyExposureMode = MetadataQName(Exif::namespaceUri, "ExposureMode");
MetadataQName Exif::propertyWhiteBalance = MetadataQName(Exif::namespaceUri, "WhiteBalance");
MetadataQName Exif::propertySceneCaptureType = MetadataQName(Exif::namespaceUri, "SceneCaptureType");

Exif::TagSet Exif::tagSetTable[] = {
	{ Exif::TagExposureTime, Exif::propertyExposureTime.localName(), Tiff::TypeRational },
	{ Exif::TagFNumber, Exif::propertyFNumber.localName(), Tiff::TypeRational },
	{ Exif::TagExposureProgram, Exif::propertyExposureProgram.localName(), Tiff::TypeShort },
	{ Exif::TagISOSpeedRatings, Exif::propertyISOSpeedRatings.localName(), Tiff::TypeShort },
	{ Exif::TagExifVersion, Exif::propertyExifVersion.localName(), Tiff::TypeUndefined },
	{ Exif::TagDateTimeOriginal, Exif::propertyDateTimeOriginal.localName(), Tiff::TypeAscii },
	{ Exif::TagDateTimeDigitized, Exif::propertyDateTimeDigitized.localName(), Tiff::TypeAscii },
	{ Exif::TagComponentsConfiguration, Exif::propertyComponentsConfiguration.localName(), Tiff::TypeUndefined },
	{ Exif::TagShutterSpeedValue, Exif::propertyShutterSpeedValue.localName(), Tiff::TypeSRational },
	{ Exif::TagApertureValue, Exif::propertyApertureValue.localName(), Tiff::TypeRational },
	{ Exif::TagExposureBiasValue, Exif::propertyExposureBiasValue.localName(), Tiff::TypeSRational },
	{ Exif::TagMaxApertureValue, Exif::propertyMaxApertureValue.localName(), Tiff::TypeRational },
	{ Exif::TagMeteringMode, Exif::propertyMeteringMode.localName(), Tiff::TypeShort },
	{ Exif::TagFlash, Exif::propertyFlash.localName(), Tiff::TypeShort },
	{ Exif::TagFocalLength, Exif::propertyFocalLength.localName(), Tiff::TypeRational },
	{ Exif::TagUserComment, Exif::propertyUserComment.localName(), Tiff::TypeUndefined },
	{ Exif::TagSubsecTime, Exif::propertySubsecTime.localName(), Tiff::TypeAscii },
	{ Exif::TagSubsecTimeOriginal, Exif::propertySubsecTimeOriginal.localName(), Tiff::TypeAscii },
	{ Exif::TagSubsecTimeDigitized, Exif::propertySubsecTimeDigitized.localName(), Tiff::TypeAscii },
	{ Exif::TagFlashPixVersion, Exif::propertyFlashPixVersion.localName(), Tiff::TypeUndefined },
	{ Exif::TagColorSpace, Exif::propertyColorSpace.localName(), Tiff::TypeShort },
	{ Exif::TagPixelXDimension, Exif::propertyPixelXDimension.localName(), Tiff::TypeLong },
	{ Exif::TagPixelYDimension, Exif::propertyPixelYDimension.localName(), Tiff::TypeLong },
	{ Exif::TagFocalPlaneXResolution, Exif::propertyFocalPlaneXResolution.localName(), Tiff::TypeRational },
	{ Exif::TagFocalPlaneYResolution, Exif::propertyFocalPlaneYResolution.localName(), Tiff::TypeRational },
	{ Exif::TagFocalPlaneResolutionUnit, Exif::propertyFocalPlaneResolutionUnit.localName(), Tiff::TypeShort },
	{ Exif::TagCustomRendered, Exif::propertyCustomRendered.localName(), Tiff::TypeShort },
	{ Exif::TagExposureMode, Exif::propertyExposureMode.localName(), Tiff::TypeShort },
	{ Exif::TagWhiteBalance, Exif::propertyWhiteBalance.localName(), Tiff::TypeShort },
	{ Exif::TagSceneCaptureType, Exif::propertySceneCaptureType.localName(), Tiff::TypeShort }
};

QMap<int, QString> TiffParser::_tiffTagNameMap = TiffParser::createTiffTagNameMap();
QMap<int, QString> TiffParser::_exifTagNameMap = TiffParser::createExifTagNameMap();

int Tiff::typeSize(Type type) {
	switch(type) {
		case TypeByte:
		case TypeAscii:
			return 1;
		case TypeShort:
			return 2;
		case TypeLong:
			return 4;
		case TypeRational:
			return 8;
		case TypeSByte:
		case TypeUndefined:
			return 1;
		case TypeSShort:
			return 2;
		case TypeSLong:
			return 4;
		case TypeSRational:
			return 8;
		case TypeFloat:
			return 4;
		case TypeDouble:
			return 8;
		default:
			return 1;
	}
}

double Tiff::lightSourceTemperature(quint16 value) {
	if(value & 0x8000)
		return (double)(value & 0x7fff);

	// TODO: check this values
	switch(value) {
		case 1:
			return 5600.0;

		case 2:
			return 4000.0;

		case 3:
			return 2800.0;

		case 10:
			return 5200.0;

		case 17:
			return 2856.0;

		case 18:
			return 4874.0;

		case 19:
			return 6774.0;

		case 20:
			return 5503.0;

		case 21:
			return 6504.0;

		case 22:
			return 7504.0;

		default:
			return 0.0;
	}
}

TiffEntry::TiffEntry(quint16 tag, Tiff::Type type, QVariant value) {
	QList<QVariant> values;

	values << value;

	_tag = tag;
	_type = type;
	_values = values;
}

TiffEntry::TiffEntry(quint16 tag, Tiff::Type type, QList<QVariant> values) {
	_tag = tag;
	_type = type;
	_values = values;
}

quint16 TiffEntry::tag() {
	return _tag;
}

Tiff::Type TiffEntry::type() {
	return _type;
}

QList<QVariant> TiffEntry::values() {
	return _values;
}

quint32 TiffEntry::valueCount() {
	switch(_type) {
		case Tiff::TypeAscii:
			return _values.at(0).toString().size() + 1;
		case Tiff::TypeUndefined:
			return _values.at(0).toByteArray().size();
		default:
			return _values.size();
	}
}

quint32 TiffEntry::valueSize() {
	return valueCount() * Tiff::typeSize(_type);
}

void TiffDirectory::addTag(TiffEntry* entry) {
	_entries.insert(entry->tag(), entry);
}

QMap<quint16, TiffEntry*> TiffDirectory::entryMap() {
	return _entries;
}

quint32 TiffDirectory::headerSize() {
	return _entries.size()*12 + 6;
}

quint32 TiffDirectory::valueSize() {
	quint32 size = 0;

	foreach(TiffEntry* entry, _entries) {
		quint32 entryValueSize = entry->valueSize();

		if(entryValueSize > 4)
			size += entryValueSize;
	}

	return size;
}

quint32 TiffDirectory::size() {
	return headerSize() + valueSize();
}

QMap<int, QString> TiffParser::createTiffTagNameMap() {
	QMap<int, QString> tagNameMap;

	for(size_t i=0; i<sizeof(Tiff::tagSetTable)/sizeof(Tiff::TagSet); i++)
		tagNameMap.insert(Tiff::tagSetTable[i].tag, Tiff::tagSetTable[i].property);

	return tagNameMap;
}

QMap<int, QString> TiffParser::createExifTagNameMap() {
	QMap<int, QString> tagNameMap;

	for(size_t i=0; i<sizeof(Exif::tagSetTable)/sizeof(Exif::TagSet); i++)
		tagNameMap.insert(Exif::tagSetTable[i].tag, Exif::tagSetTable[i].property);

	return tagNameMap;
}

TiffParser::TiffParser(QIODevice* device) {
	_dataStream = new QDataStream(device);
	_document = 0;
}

TiffParser::~TiffParser() {
	delete _dataStream;
}

MetadataDocument* TiffParser::parse() {
	_document = new MetadataDocument();

	parseHeader(_document);

	return _document;
}

MetadataDocument* TiffParser::document() {
	return _document;
}

quint32 TiffParser::directoryOffset(int index) {
	return _offset.at(index);
}

quint32 TiffParser::directoryLength(int index) {
	return _length.at(index);
}

void TiffParser::detectByteOrder() {
	quint16 byteOrder;

	*_dataStream >> byteOrder;

	if(byteOrder == 0x4949)
		_dataStream->setByteOrder(QDataStream::LittleEndian);
	else if(byteOrder == 0x4d4d)
		_dataStream->setByteOrder(QDataStream::BigEndian);		
}

quint8 TiffParser::readByte() {
	quint8 value;
	*_dataStream >> value;

	return value;
}

quint16 TiffParser::readShort() {
	quint16 value;
	*_dataStream >> value;

	return value;
}

quint32 TiffParser::readLong() {
	quint32 value;
	*_dataStream >> value;

	return value;
}

Rational TiffParser::readRational() {
	qlonglong numerator = readLong();
	qlonglong denominator = readLong();

	return Rational(numerator, denominator);
}

qint8 TiffParser::readSByte() {
	qint8 value;
	*_dataStream >> value;

	return value;
}

qint16 TiffParser::readSShort() {
	qint16 value;
	*_dataStream >> value;

	return value;
}

qint32 TiffParser::readSLong() {
	qint32 value;
	*_dataStream >> value;

	return value;
}

Rational TiffParser::readSRational() {
	qlonglong numerator = readSLong();
	qlonglong denominator = readSLong();

	return Rational(numerator, denominator);
}

float TiffParser::readFloat() {
	float value;
	*_dataStream >> value;

	return value;
}

double TiffParser::readDouble() {
	double value;
	*_dataStream >> value;

	return value;
}

QList<QVariant> TiffParser::readValues(quint16 tag, quint16 type, quint32 count) {
	QList<QVariant> values;

	if(type == Tiff::TypeAscii) {
		values.append(tagValue(tag, QVariant(QString(_dataStream->device()->read(count)))));
	} else if(type == Tiff::TypeUndefined || type == Tiff::TypeByte || type == Tiff::TypeSByte) {
		values.append(tagValue(tag, QVariant(_dataStream->device()->read(count))));
	} else {
		for(quint32 i=0; i<count; i++) {
			QVariant value;

			switch(type) {
				case Tiff::TypeByte:
					value = QVariant(readByte());
					break;
				case Tiff::TypeShort:
					value = QVariant(readShort());
					break;
				case Tiff::TypeLong:
					value = QVariant(readLong());
					break;
				case Tiff::TypeRational:
					value.setValue(readRational());
					break;
				case Tiff::TypeSByte:
					value = QVariant(readSByte());
					break;
				case Tiff::TypeSShort:
					value = QVariant(readSShort());
					break;
				case Tiff::TypeSLong:
					value = QVariant(readSLong());
					break;
				case Tiff::TypeSRational:
					value.setValue(readSRational());
					break;
				case Tiff::TypeFloat:
					value = QVariant(readFloat());
					break;
				case Tiff::TypeDouble:
					value = QVariant(readDouble());
					break;
				default:
					qDebug("tiff type not supported");
					break;
			}

			if(!value.isNull())
				values.append(tagValue(tag, value));
		}
	}

	return values;
}

void TiffParser::parseEntry(MetadataNode* parent, quint32 offset) {
	_dataStream->device()->seek(offset);

	quint16 tag, type;
	quint32 count;

	*_dataStream >> tag;
	*_dataStream >> type;
	*_dataStream >> count;

	int dataSize = Tiff::typeSize((Tiff::Type)type) * count;

	if(tag == Tiff::TagMakerNote) {
		MetadataComplexProperty* makerNoteProperty = new MetadataComplexProperty(parent, Tiff::propertyMakerNote);
		parseMakerNote(makerNoteProperty, readLong());

		return;
	}

	if(dataSize > 4)
		_dataStream->device()->seek(readLong());

	QList<QVariant> values = readValues(tag, type, count);

	MetadataQName qName = tagQName(tag);

	if(tag == Tiff::TagSubIFDs) {
		foreach(QVariant value, values) {
			MetadataComplexProperty* complexProperty = new MetadataComplexProperty(parent, Tiff::propertySubIFDs);
			parseDirectory(complexProperty, value.toUInt());
		}

		return;
	} else if(tag == Tiff::TagExifIFD) {
		MetadataComplexProperty* exifIfdProperty = new MetadataComplexProperty(parent, Tiff::propertyExifIFD);
		parseExif(exifIfdProperty, values.at(0).toUInt());

		return;
	}

	if(values.size() == 1) {
		new MetadataSimpleProperty(parent, qName, values.at(0));
	} else if(values.size() > 1) {
		MetadataListProperty* listProperty = new MetadataListProperty(parent, qName);

		foreach(QVariant value, values) {
			MetadataListItem* item = new MetadataListItem(listProperty);
			item->setValue(value);
		}
	}

	if(tag == Tiff::TagStripOffsets || tag == Tiff::TagJPEGInterchangeFormat) {
		foreach(QVariant value, values) {
			_offset.append(value.toUInt());
		}
	} else if(tag == Tiff::TagStripByteCounts || tag == Tiff::TagJPEGInterchangeFormatLength) {
		foreach(QVariant value, values) {
			_length.append(value.toUInt());
		}
	}
}

void TiffParser::parseMakerNote(MetadataNode* parent, quint32 offset) {
	Q_UNUSED(parent);
	Q_UNUSED(offset);
}

void TiffParser::parseExif(MetadataNode* parent, quint32 offset) {
	parseDirectory(parent, offset);
}

void TiffParser::parseDirectory(MetadataNode* parent, quint32 offset) {
	_dataStream->device()->seek(offset);
	quint16 directorySize;
	*_dataStream >> directorySize;

	for(int i=0; i<directorySize; i++)
		parseEntry(parent, offset+2+i*12);
}

void TiffParser::parseHeader(MetadataNode* parent, quint32 offset) {
	_dataStream->device()->seek(offset);

	detectByteOrder();

	_dataStream->skipRawData(2);
	quint32 directoryOffset;
	*_dataStream >> directoryOffset;
	int directoryNumber = 0;
 
	while(directoryOffset != 0) {
		_dataStream->device()->seek(directoryOffset);
		quint16 directorySize;
		*_dataStream >> directorySize;

		MetadataResource* resource = new MetadataResource(parent, QString("tiffdir%1").arg(directoryNumber));

		parseDirectory(resource, directoryOffset);

		_dataStream->device()->seek(directoryOffset+directorySize*12+2);
		*_dataStream >> directoryOffset;
		directoryNumber++;
	}
}

MetadataQName TiffParser::tagQName(quint16 tag) {
	if(_tiffTagNameMap.contains(tag))
		return MetadataQName(Tiff::namespaceUri, _tiffTagNameMap.value(tag));
	else if(_exifTagNameMap.contains(tag))
		return MetadataQName(Exif::namespaceUri, _exifTagNameMap.value(tag));
	else
		return MetadataQName(Tiff::namespaceUri, QString("tag%1").arg(QString::number(tag, 16)));
}

QVariant TiffParser::tagValue(int tag, QVariant value) {
	switch(tag) {
		case Tiff::TagCompression:
			switch(value.toInt()) {
				case 6:
					return QVariant("JPEG");
				default:
					return value;
			}

		case Tiff::TagOrientation:
			switch(value.toInt()) {
				case 1:
					return QVariant("horizontal");
				default:
					return value;
			}

		case Tiff::TagResolutionUnit:
			switch(value.toInt()) {
				case 1:
					return QVariant("none");
				case 2:
					return QVariant("inches");
				case 3:
					return QVariant("cm");
				default:
					return value;
			}

		case Exif::TagExposureProgram:
			switch(value.toInt()) {
				case 0:
					return QVariant("not defined");
				case 1:
					return QVariant("manual");
				case 2:
					return QVariant("program ae");
				case 3:
					return QVariant("aperture-priority ae");
				case 4:
					return QVariant("shutter speed priority ae");
				case 5:
					return QVariant("creative (slow speed)");
				case 6:
					return QVariant("action (high speed)");
				case 7:
					return QVariant("portrait");
				case 8:
					return QVariant("landscape");
				default:
					return value;
			}

			case Exif::TagExifVersion:
			case Exif::TagFlashPixVersion:
				return QVariant(QString(value.toByteArray()));

		default:
			return value;
	}
}

QString TiffWriter::name = "TiffWriter";

QMap<QString, Tiff::TagSet> TiffWriter::_tiffTagSetMap = TiffWriter::createTiffTagSetMap();
QMap<QString, Exif::TagSet> TiffWriter::_exifTagSetMap = TiffWriter::createExifTagSetMap();

QMap<QString, Tiff::TagSet> TiffWriter::createTiffTagSetMap() {
	QMap<QString, Tiff::TagSet> tagSetMap;

	for(size_t i=0; i<sizeof(Tiff::tagSetTable)/sizeof(*Tiff::tagSetTable); i++)
		tagSetMap.insert(Tiff::tagSetTable[i].property, Tiff::tagSetTable[i]);

	return tagSetMap;
}

QMap<QString, Exif::TagSet> TiffWriter::createExifTagSetMap() {
	QMap<QString, Exif::TagSet> tagSetMap;

	for(size_t i=0; i<sizeof(Exif::tagSetTable)/sizeof(*Exif::tagSetTable); i++)
		tagSetMap.insert(Exif::tagSetTable[i].property, Exif::tagSetTable[i]);

	return tagSetMap;
}

TiffWriter::TiffWriter() {
}

TiffWriter::~TiffWriter() {
}

bool TiffWriter::accepts(QString format) {
	return format == RgbImage48::format();
}

bool TiffWriter::write(QIODevice* device, Image* image, MetadataResource* resource, BergPhoto::Parameters parameters) {
	RgbImage48* rgbImage48;
	BayerImage16* bayerImage16;

	if((rgbImage48 = qobject_cast<RgbImage48*>(image)) != 0) {
		QDataStream* dataStream = new QDataStream(device);
		dataStream->setByteOrder(QDataStream::LittleEndian);

		QList<QVariant> valuesBitsPerSample;
		valuesBitsPerSample << 16 << 16 << 16;

		QByteArray imageData = QByteArray::fromRawData((const char*)rgbImage48->data(), rgbImage48->width()*rgbImage48->height()*3*2);

		writeHeader(dataStream);

		TiffDirectory directory;
		directory.addTag(new TiffEntry(Tiff::TagImageWidth, Tiff::TypeShort, QVariant(rgbImage48->width())));
		directory.addTag(new TiffEntry(Tiff::TagImageLength, Tiff::TypeShort, QVariant(rgbImage48->height())));
		directory.addTag(new TiffEntry(Tiff::TagBitsPerSample, Tiff::TypeShort, valuesBitsPerSample));
		directory.addTag(new TiffEntry(Tiff::TagCompression, Tiff::TypeShort, QVariant(1)));
		directory.addTag(new TiffEntry(Tiff::TagPhotometricInterpretation, Tiff::TypeShort, QVariant(2)));
		directory.addTag(new TiffEntry(Tiff::TagSamplesPerPixel, Tiff::TypeShort, QVariant(3)));

		if(image->colorProfile() != 0)
			new MetadataSimpleProperty(resource, Tiff::propertyICCProfile, QVariant(image->colorProfile()->toByteArray()));

		addTiffMetadataTags(&directory, resource);

		TiffDirectory exifDirectory;
		addExifMetadataTags(&exifDirectory, resource);

		quint32 exifIdfOffset = (quint32)dataStream->device()->pos() + 4;
		directory.addTag(new TiffEntry(Tiff::TagExifIFD, Tiff::TypeLong, QVariant(exifIdfOffset)));

		quint32 firstIdfOffset = exifIdfOffset + exifDirectory.size();

		*dataStream << firstIdfOffset;

		writeIdf(dataStream, &exifDirectory);
		writeIdf(dataStream, &directory, imageData);

		delete dataStream;

		return true;
	} else if((bayerImage16 = qobject_cast<BayerImage16*>(image)) != 0) {
		QDataStream* dataStream = new QDataStream(device);
		dataStream->setByteOrder(QDataStream::LittleEndian);

		QByteArray imageData = QByteArray::fromRawData((const char*)bayerImage16->data(), bayerImage16->width()*bayerImage16->height()*2);

		writeHeader(dataStream);

		TiffDirectory directory;
		directory.addTag(new TiffEntry(Tiff::TagImageWidth, Tiff::TypeShort, QVariant(bayerImage16->width())));
		directory.addTag(new TiffEntry(Tiff::TagImageLength, Tiff::TypeShort, QVariant(bayerImage16->height())));
		directory.addTag(new TiffEntry(Tiff::TagBitsPerSample, Tiff::TypeShort, QVariant(16)));
		directory.addTag(new TiffEntry(Tiff::TagCompression, Tiff::TypeShort, QVariant(1)));
		directory.addTag(new TiffEntry(Tiff::TagPhotometricInterpretation, Tiff::TypeShort, QVariant(1)));
		directory.addTag(new TiffEntry(Tiff::TagSamplesPerPixel, Tiff::TypeShort, QVariant(1)));

		addTiffMetadataTags(&directory, resource);

		quint32 firstIdfOffset = (quint32)dataStream->device()->pos() + 4;

		*dataStream << firstIdfOffset;

		writeIdf(dataStream, &directory, imageData);

		delete dataStream;

		return true;
	}

	return false;
}

void TiffWriter::writeHeader(QDataStream* dataStream) {
	if(dataStream->byteOrder() == QDataStream::LittleEndian)
		*dataStream << (quint16)Tiff::ByteOrderLittleEndian;
	else
		*dataStream << (quint16)Tiff::ByteOrderBigEndian;

	*dataStream << (quint16)Tiff::MagicNumber;
}

void TiffWriter::writeIdf(QDataStream* dataStream, TiffDirectory* directory, QByteArray data, quint32 nextIdf) {
	if(data.size() != 0) {
		directory->addTag(new TiffEntry(Tiff::TagStripByteCounts, Tiff::TypeLong, QVariant((quint32)data.size())));
		directory->addTag(new TiffEntry(Tiff::TagStripOffsets, Tiff::TypeLong, QVariant(0)));
	}

	quint32 currentOffset = (quint32)dataStream->device()->pos();
	quint32 valuesOffset = currentOffset + directory->headerSize();
	quint32 imageOffset = currentOffset + directory->size();

	QMap<quint16, TiffEntry*> entries = directory->entryMap();
	*dataStream << (quint16)entries.size();

	QList<quint16> sortedTags = entries.keys();
	qSort(sortedTags);

	foreach(quint16 tag, sortedTags) {
		TiffEntry* entry = entries.value(tag);
		quint32 entryValueSize = entry->valueSize();

		*dataStream << tag;
		*dataStream << (quint16)entry->type();
		*dataStream << entry->valueCount();

		if(tag == Tiff::TagStripOffsets) {
			*dataStream << imageOffset;
		} else if(entryValueSize <= 4) {
			writeValues(dataStream, entry);

			for(quint32 i=entryValueSize; i<4; i++)
				*dataStream << (quint8)0;
		} else {
			*dataStream << valuesOffset;
			valuesOffset += entryValueSize;
		}
	}

	*dataStream << nextIdf;

	foreach(quint16 tag, sortedTags) {
		TiffEntry* entry = entries.value(tag);

		if(entry->valueSize() > 4)
			writeValues(dataStream, entry);
	}

	dataStream->device()->write(data);
}

void TiffWriter::writeValues(QDataStream* dataStream, TiffEntry* entry) {
	Tiff::Type type = entry->type();

	if(type == Tiff::TypeAscii) {
		dataStream->device()->write(entry->values().at(0).toString().toAscii());
		*dataStream << (quint8)0;
	} else if(type == Tiff::TypeUndefined) {
		dataStream->device()->write(entry->values().at(0).toByteArray());
	} else {
		foreach(QVariant value, entry->values()) {
			value = tagValue(entry->tag(), value);

			switch(type) {
				case Tiff::TypeByte:
					*dataStream << value.value<quint8>();
					break;
				case Tiff::TypeShort:
					*dataStream << value.value<quint16>();
					break;
				case Tiff::TypeLong:
					*dataStream << value.value<quint32>();
					break;
				case Tiff::TypeRational:
					writeRational(dataStream, value.value<Rational>());
					break;
				case Tiff::TypeSByte:
					*dataStream << value.value<qint8>();
					break;
				case Tiff::TypeSShort:
					*dataStream << value.value<qint16>();
					break;
				case Tiff::TypeSLong:
					*dataStream << value.value<qint32>();
					break;
				case Tiff::TypeSRational:
					writeSRational(dataStream, value.value<Rational>());
					break;
				case Tiff::TypeFloat:
					*dataStream << value.value<float>();
					break;
				case Tiff::TypeDouble:
					*dataStream << value.value<double>();
					break;
				default:
					break;
			}
		}
	}
}

void TiffWriter::writeRational(QDataStream* dataStream, Rational value) {
	*dataStream << (quint32)value.numerator();
	*dataStream << (quint32)value.denominator();
}

void TiffWriter::writeSRational(QDataStream* dataStream, Rational value) {
	*dataStream << (qint32)value.numerator();
	*dataStream << (qint32)value.denominator();
}

void TiffWriter::addTiffMetadataTags(TiffDirectory* directory, MetadataResource* resource) {
	if(resource == 0)
		return;

	foreach(MetadataNode* node, resource->children()) {
		if(!node->isProperty())
			continue;

		MetadataProperty* property = node->toProperty();

		if(property->qName().namespaceUri() != Tiff::namespaceUri)
			continue;

		if(_tiffTagSetMap.contains(property->qName().localName())) {
			Tiff::TagSet tagSet = _tiffTagSetMap.value(property->qName().localName());

			if(!directory->entryMap().contains(tagSet.tag))
				directory->addTag(new TiffEntry(tagSet.tag, tagSet.defaultType, property->values()));
		}
	}
}

void TiffWriter::addExifMetadataTags(TiffDirectory* directory, MetadataResource* resource) {
	if(resource == 0)
		return;

	MetadataProperty* exifIfdProperty = resource->queryProperty(Tiff::propertyExifIFD);

	foreach(MetadataNode* node, exifIfdProperty->children()) {
		if(!node->isProperty())
			continue;

		MetadataProperty* property = node->toProperty();

		if(property->qName().namespaceUri() != Exif::namespaceUri)
			continue;

		if(_exifTagSetMap.contains(property->qName().localName())) {
			Exif::TagSet tagSet = _exifTagSetMap.value(property->qName().localName());

			if(!directory->entryMap().contains(tagSet.tag))
				directory->addTag(new TiffEntry(tagSet.tag, tagSet.defaultType, property->values()));
		}
	}
}

QVariant TiffWriter::tagValue(int tag, QVariant value) {
	QString stringValue = value.toString();

	switch(tag) {
		case Tiff::TagCompression:
			if(stringValue == "JPEG")
				return QVariant(6);
			else
				return value;

		case Tiff::TagOrientation:
			if(stringValue == "horizontal")
				return QVariant(1);
			else
				return value;

		case Tiff::TagResolutionUnit:
			if(stringValue == "none")
				return QVariant(1);
			else if(stringValue == "inches")
				return QVariant(2);
			else if(stringValue == "cm")
				return QVariant(3);

		case Exif::TagExposureProgram:
			if(stringValue == "not defined")
				return QVariant(0);
			else if(stringValue == "manual")
				return QVariant(1);
			else if(stringValue == "program ae")
				return QVariant(2);
			else if(stringValue == "aperture-priority ae")
				return QVariant(3);
			else if(stringValue == "shutter speed priority ae")
				return QVariant(4);
			else if(stringValue == "creative (slow speed)")
				return QVariant(5);
			else if(stringValue == "action (high speed)")
				return QVariant(6);
			else if(stringValue == "portrait")
				return QVariant(7);
			else if(stringValue == "landscape")
				return QVariant(8);
			else
				return value;

		default:
			return value;
	}
}

TiffSimpleMetadataTransform::~TiffSimpleMetadataTransform() {
}

MetadataNode* TiffSimpleMetadataTransform::transform(MetadataNode* source) {
	if(!source->isResource())
		return 0;

	QList<MetadataQName> propertyToKeepTiff;
	propertyToKeepTiff << Tiff::propertyDateTime << Tiff::propertyMake << Tiff::propertyModel << Tiff::propertyOrientation << Tiff::propertyExifIFD;

	MetadataResource* resource = new MetadataResource(0);
	MetadataResource* sourceResource = source->toResource();

	foreach(MetadataNode* sourceChild, sourceResource->children()) {
		if(!sourceChild->isProperty())
			continue;

		MetadataProperty* sourceProperty = sourceChild->toProperty();

		if(sourceProperty->qName().namespaceUri() == Tiff::namespaceUri && !propertyToKeepTiff.contains(sourceProperty->qName()))
			continue;

		sourceProperty->clone(resource);
	}

	return resource;
}
