/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "Widgets.h"

#include <ImageFilter.h>
#include <QtUtils.h>
#include <RawFile.h>
#include <Tiff.h>

#include <QPainter>

ImageWidget::ImageWidget(QWidget* parent) : QWidget(parent) {
	_image = 0;
}

ImageWidget::~ImageWidget() {
}

QImage* ImageWidget::image() {
	return _image;
}

void ImageWidget::setImage(QImage* image) {
	_image = image;

	repaint();
}

void ImageWidget::paintEvent(QPaintEvent* paintEvent) {
	Q_UNUSED(paintEvent);

	if(_image == 0)
		return;

	QPainter painter;
	QRectF source(0, 0, _image->width(), _image->height());

	double widgetWidth = (double)width();
	double widgetHeight = (double)height();
	double imageWidth = (double)_image->width();
	double imageHeight = (double)_image->height();

	int previewWidth = 0;
	int previewHeight = 0;

	if((imageWidth/imageHeight) > (widgetWidth/widgetHeight)) {
		previewWidth = (int)widgetWidth;
		previewHeight = (int)(widgetWidth*imageHeight/imageWidth);
	} else {
		previewWidth = (int)(widgetHeight*imageWidth/imageHeight);
		previewHeight = (int)widgetHeight;
	}

	QRectF target(0, 0, previewWidth, previewHeight);

	painter.begin(this);
	painter.drawImage(target, *_image, source);
	painter.end();
}

void ImageWidget::resizeEvent(QResizeEvent* resizeEvent) {
	emit outputSizeChanged(resizeEvent->size().width(), resizeEvent->size().height());
}

SplineWidget::SplineWidget(QWidget* parent) : QWidget(parent) {
	_pointRadius = 5;
}

SplineWidget::SplineWidget(Spline &spline, QWidget* parent) : QWidget(parent), Spline(spline) {
	_pointRadius = 5;
}

SplineWidget::~SplineWidget() {
}

void SplineWidget::setPoints(QList<QPointF> list, double scale) {
	Spline::setPoints(list, scale);

	repaint();
}

void SplineWidget::mouseMoveEvent(QMouseEvent* mouseEvent) {
	if(_dragPointIndex >= 0) {
		movePoint(QPointF(mouseEvent->pos().x()/(double)width(), 1.0-mouseEvent->pos().y()/(double)height()), _dragPointIndex);
	}

	emit curveChanged(*this);

	repaint();
}

void SplineWidget::mousePressEvent(QMouseEvent* mouseEvent) {
	if(mouseEvent->button() == Qt::LeftButton) {
		_dragPointIndex = -1;

		for(int i=0; i<_p.size(); i++) {
			QPointF p = _p.at(i);

			double x = mouseEvent->pos().x() - p.x()*width();
			double y = height() - mouseEvent->pos().y() - p.y()*height();

			if(x*x+y*y < _pointRadius*_pointRadius) {
				_dragPointIndex = i;
				break;
			}
		}

		if(_dragPointIndex == -1) {
			QPointF p(mouseEvent->pos().x()/(double)width(), 1.0-mouseEvent->pos().y()/(double)height());

			_dragPointIndex = addPoint(p);
		}
	} else if(mouseEvent->button() == Qt::RightButton) {
		for(int i=0; i<_p.size(); i++) {
			QPointF p = _p.at(i);

			double x = mouseEvent->pos().x() - p.x()*width();
			double y = height() - mouseEvent->pos().y() - p.y()*height();

			if(x*x+y*y < _pointRadius*_pointRadius) {
				removePoint(i);
				break;
			}
		}
	} else {
		return;
	}

	emit curveChanged(*this);

	repaint();
}

void SplineWidget::mouseReleaseEvent(QMouseEvent* mouseEvent) {
	Q_UNUSED(mouseEvent);

	_dragPointIndex = -1;
}

void SplineWidget::paintEvent(QPaintEvent* paintEvent) {
	Q_UNUSED(paintEvent);

	double widthD = (double)width();

	QPainter painter;

	double preValue = height() - evaluate(0.0) * height();

	painter.begin(this);

	for(int i=0; i<width(); i++) {
		double value = height() - evaluate((double)i / widthD) * height();

		QPointF start(i, preValue);
		QPointF end(i, value);

		preValue = value;

		painter.drawLine(start, end);
	}

	foreach(QPointF p, _p) {
		QPointF c(p.x()*width(), height()-p.y()*height());

		painter.drawEllipse(QRectF(c.x()-_pointRadius, c.y()-_pointRadius, _pointRadius*2, _pointRadius*2));
	}

	painter.end();
}

DevelopParametersWidget::DevelopParametersWidget(QWidget* parent) : QWidget(parent) {
	_formDevelopParameters.setupUi(this);

	updateComboBoxOutputProfileValues();
}

DevelopParametersWidget::~DevelopParametersWidget() {	
}

QString DevelopParametersWidget::outputProfile() {
	return _formDevelopParameters.comboBoxOutputProfile->currentText();
}

void DevelopParametersWidget::setOutputProfile(QString outputProfile) {
	QtUtils::comboBoxSetCurrentItem(_formDevelopParameters.comboBoxOutputProfile, outputProfile);
}

BergPhoto::Parameters DevelopParametersWidget::parameters() {
	BergPhoto::Parameters parameters;

	parameters.insert(DevelopFilter::parameterOutputProfile, outputProfile());

	return parameters;
}

void DevelopParametersWidget::setParameters(BergPhoto::Parameters parameters) {
	if(parameters.contains(DevelopFilter::parameterOutputProfile))
		setOutputProfile(parameters.value(DevelopFilter::parameterOutputProfile).toString());
}

void DevelopParametersWidget::updateComboBoxOutputProfileValues() {
	QtUtils::comboBoxClearItemList(_formDevelopParameters.comboBoxOutputProfile);

	_formDevelopParameters.comboBoxOutputProfile->addItem(DevelopFilter::profileSRgb);
	_formDevelopParameters.comboBoxOutputProfile->addItem(DevelopFilter::profileAdobeRgb);
	_formDevelopParameters.comboBoxOutputProfile->addItem(DevelopFilter::profileProPhotoLinearRgb);
}

ImageWriterParametersWidget::ImageWriterParametersWidget(QWidget* parent) : QWidget(parent) {
	_formImageWriterParameters.setupUi(this);

	updateComboBoxImageWriter();
	updateComboBoxFileFormat();

	connect(_formImageWriterParameters.comboBoxImageWriter, SIGNAL(currentIndexChanged(QString)), this, SLOT(updateComboBoxFileFormat(QString)));
}

ImageWriterParametersWidget::~ImageWriterParametersWidget() {
}

QString ImageWriterParametersWidget::imageWriter() {
	return _formImageWriterParameters.comboBoxImageWriter->currentText();
}

void ImageWriterParametersWidget::setImageWriter(QString imageWriter) {
	QtUtils::comboBoxSetCurrentItem(_formImageWriterParameters.comboBoxImageWriter, imageWriter);
}

QString ImageWriterParametersWidget::fileFormat() {
	return _formImageWriterParameters.comboBoxFileFormat->currentText();
}

void ImageWriterParametersWidget::setFileFormat(QString fileFormat) {
	QtUtils::comboBoxSetCurrentItem(_formImageWriterParameters.comboBoxFileFormat, fileFormat);
}

bool ImageWriterParametersWidget::compress() {
	return _formImageWriterParameters.checkBoxCompress->isChecked();
}

void ImageWriterParametersWidget::setCompress(bool compress) {
	_formImageWriterParameters.checkBoxCompress->setChecked(compress);
}

int ImageWriterParametersWidget::quality() {
	return _formImageWriterParameters.spinBoxQuality->value();
}

void ImageWriterParametersWidget::setQuality(int quality) {
	_formImageWriterParameters.spinBoxQuality->setValue(quality);
}

BergPhoto::Parameters ImageWriterParametersWidget::parameters() {
	BergPhoto::Parameters parameters;

	parameters.insert(ImageWriter::parameterImageWriter, imageWriter());
	parameters.insert(ImageWriter::parameterFileFormat, fileFormat());
	parameters.insert(ImageWriter::parameterCompress, compress());
	parameters.insert(ImageWriter::parameterQuality, quality());

	return parameters;
}

void ImageWriterParametersWidget::setParameters(BergPhoto::Parameters parameters) {
	if(parameters.contains(ImageWriter::parameterImageWriter))
		setImageWriter(parameters.value(ImageWriter::parameterImageWriter).toString());

	if(parameters.contains(ImageWriter::parameterFileFormat))
		setFileFormat(parameters.value(ImageWriter::parameterFileFormat).toString());

	if(parameters.contains(ImageWriter::parameterCompress))
		setCompress(parameters.value(ImageWriter::parameterCompress).toBool());

	if(parameters.contains(ImageWriter::parameterQuality))
		setQuality(parameters.value(ImageWriter::parameterQuality).toInt());
}

void ImageWriterParametersWidget::updateComboBoxImageWriter() {
	QtUtils::comboBoxClearItemList(_formImageWriterParameters.comboBoxImageWriter);

	_formImageWriterParameters.comboBoxImageWriter->addItem(QtImageWriter::name);
	_formImageWriterParameters.comboBoxImageWriter->addItem(TiffWriter::name);
}

void ImageWriterParametersWidget::updateComboBoxFileFormat() {
	QtUtils::comboBoxClearItemList(_formImageWriterParameters.comboBoxFileFormat);

	if(imageWriter() == QtImageWriter::name) {
		_formImageWriterParameters.comboBoxFileFormat->addItem("bmp");
		_formImageWriterParameters.comboBoxFileFormat->addItem("jpg");
		_formImageWriterParameters.comboBoxFileFormat->addItem("png");
		_formImageWriterParameters.comboBoxFileFormat->addItem("ppm");
		_formImageWriterParameters.comboBoxFileFormat->addItem("tiff");
	} else if(imageWriter() == TiffWriter::name) {
		_formImageWriterParameters.comboBoxFileFormat->addItem("tiff");
	}
}

void ImageWriterParametersWidget::updateComboBoxFileFormat(QString dummy) {
	updateComboBoxFileFormat();
}
