/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "Image.h"

#include <string.h>

ImageFormat::ImageFormat(QString type, int width, int height) {
	_type = type;
	_width = width;
	_height = height;
}

void ImageFormat::setType(QString type) {
	_type = type;
}

QString ImageFormat::type() {
	return _type;
}

void ImageFormat::setWidth(int width) {
	_width = width;
}

int ImageFormat::width() {
	return _width;
}

void ImageFormat::setHeight(int height) {
	_height = height;
}

int ImageFormat::height() {
	return _height;
}

QString Image::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/image/1.0/";
MetadataQName Image::propertyOrientation = MetadataQName(Image::namespaceUri, "Orientation");

Image::Image(QObject* parent) : QObject(parent) {
	_pixelSize = 0;

	_width = -1;
	_height = -1;
	_stripeSize = -1;

	_dataAlloc = 0;
	_data = _dataAlloc;

	_colorProfile = 0;
	_resource = 0;
}

Image::~Image() {
	delete _dataAlloc;
}

void Image::setFormat(ImageFormat imageFormat) {
	_type = imageFormat.type();
	_width = imageFormat.width();
	_height = imageFormat.height();
}

ImageFormat Image::format() {
	return ImageFormat(_type, _width, _height);
}

void Image::setWidth(int width) {
	_width = width;
}

int Image::width() {
	return _width;
}

void Image::setHeight(int height) {
	_height = height;
}

int Image::height() {
	return _height;
}

void Image::setStripeSize(int stripeSize) {
	_stripeSize = stripeSize;
}

int Image::stripeSize() {
	return _stripeSize;
}

void Image::setPixelSize(int pixelSize) {
	_pixelSize = pixelSize;
}

int Image::pixelSize() {
	return _pixelSize;
}

quint8* Image::data() {
	return _data;
}

quint8* Image::dataStripe(int y) {
	return _data+_stripeSize*_pixelSize*y;
}

quint8* Image::dataStripeSave(int y) {
	if(y < 0 || y >= _height)
		return 0;
	else
		return dataStripe(y);
}

quint8* Image::dataOffset(int x, int y) {
	return _data+_stripeSize*_pixelSize*y+_pixelSize*x;
}

quint8* Image::dataOffsetSave(int x, int y) {
	if(y < 0 || y >= _height || x < 0 || x >= _width)
		return 0;
	else
		return dataOffset(x, y);
}

void Image::setType(QString type) {
	_type = type;
}

QString Image::type() {
	return _type;
}

void Image::setColorProfile(CmsProfile* colorProfile) {
	_colorProfile = colorProfile;
}

CmsProfile* Image::colorProfile() {
	return _colorProfile;
}

void Image::setMetadata(MetadataResource* resource) {
	_resource = resource;
}

MetadataResource* Image::metadata() {
	return _resource;
}

RgbImage32::RgbImage32(int width, int height, QObject* parent) : Image(parent) {
	_type = RgbImage32::type();
	_pixelSize = 4;

	_width = width;
	_height = height;
	_stripeSize = _width;

	_dataAlloc = new quint8[_stripeSize*_pixelSize*_height];
	_data = _dataAlloc;
}

RgbImage32::~RgbImage32() {
}

QImage* RgbImage32::toQImage() {
	uchar* qData = new uchar[_width*_height*4];
	uchar* p = qData;

	for(int y=0; y<_height; y++) {
		memcpy(p, dataStripe(y), _width*4);
		p += _width*4;
	}

	return new QImage(qData, _width, _height, QImage::Format_RGB32);
}

RgbImage48::RgbImage48(int width, int height, QObject* parent) : Image(parent) {
	_type = RgbImage48::type();
	_pixelSize = 6;

	_width = width;
	_height = height;
	_stripeSize = _width;

	_dataAlloc = new quint8[_stripeSize*_pixelSize*_height];
	_data = _dataAlloc;
}

RgbImage48::~RgbImage48() {
}
