/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "ImageFilter.h"
#include "ImageFilter_p.h"

#include <QByteArray>
#include <QDebug>
#include <QFile>
#include <QImageWriter>
#include <QSet>
#include <QTime>

#include <math.h>

#include <RawFile.h>

QString ImageFilterSettings::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/image-filter-settings/1.0/";
MetadataQName ImageFilterSettings::propertySettings = MetadataQName(ImageFilterSettings::namespaceUri, "Settings");
MetadataQName ImageFilterSettings::propertyName = MetadataQName(ImageFilterSettings::namespaceUri, "Name");
MetadataQName ImageFilterSettings::propertyDate = MetadataQName(ImageFilterSettings::namespaceUri, "Date");

ImageFilter::~ImageFilter() {
}

BergPhoto::Parameters ImageFilter::parameters() {
	return _parameters;
}

void ImageFilter::setParameters(BergPhoto::Parameters parameters) {
	_parameters = parameters;
}

QString ImageWriter::parameterImageWriter = "imageWriter";
QString ImageWriter::parameterFileFormat = "fileFormat";
QString ImageWriter::parameterCompress = "compress";
QString ImageWriter::parameterQuality = "quality";

bool ImageWriter::write(QString fileName, Image* image, MetadataResource* resource, BergPhoto::Parameters parameter) {
	QFile file(fileName);

	file.open(QIODevice::WriteOnly);

	bool r = write(&file, image, resource, parameter);

	file.close();

	return r;
}

QString QtImageWriter::name = "QtImageWriter";

bool QtImageWriter::accepts(QString format) {
	return format == RgbImage32::type();
}

bool QtImageWriter::write(QIODevice* device, Image* image, MetadataResource* resource, BergPhoto::Parameters parameters) {
	Q_UNUSED(resource);

	if(!accepts(image->type()))
		return false;

	QImageWriter imageWriter(device, QByteArray());

	if(parameters.contains(ImageWriter::parameterCompress))
		imageWriter.setCompression(parameters.value(ImageWriter::parameterCompress).toBool() ? 1 : 0);
	else
		imageWriter.setCompression(0);

	if(parameters.contains(ImageWriter::parameterQuality))
		imageWriter.setQuality(parameters.value(ImageWriter::parameterQuality).toInt());
	else
		imageWriter.setQuality(100);

	return imageWriter.write(*qobject_cast<RgbImage32*>(image)->toQImage());
}

ImageFilterChain::ImageFilterChain(QObject* parent) : QObject(parent) {
	_source = 0;
	_workerThread = new ImageFilterChainWorkerThread(this, this);

	connect(_workerThread, SIGNAL(finished(Image*)), this, SLOT(threadFinished(Image*)));
}

ImageFilterChain::~ImageFilterChain() {
	closeImage();

	foreach(ImageFilter* filter, _filters)
		delete filter;
}

void ImageFilterChain::appendFilter(ImageFilter* filter, ImageFormat outputFormat) {
	_filters.append(filter);
	_outputFormats.insert(filter, outputFormat);
}

ImageFormat ImageFilterChain::prepare() {
	if(_source == 0)
		return ImageFormat();

	MetadataResource* metadata = _source->metadata();
	ImageFormat format = _source->format();

	foreach(ImageFilter* filter, _filters) {
		ImageFormat outputFormat = _outputFormats.value(filter);

		format = filter->prepare(metadata, format, outputFormat);
	}

	return format;
}

Image* ImageFilterChain::filter() {
	if(_source == 0)
		return 0;

	Image* image = _source;
	bool start = false;

	_stillDirty = false;

	foreach(ImageFilter* filter, _filters) {
		Image* preImage = _images.value(filter);

		if(preImage == 0)
			start = true;

		if(!start) {
			if(isDirty(filter))
				start = true;
			else
				image = _images.value(filter);
		}

		if(start) {
			ImageFormat outputFormat = _outputFormats.value(filter);

			QTime time;
			time.start();

			image = filter->filter(image, outputFormat);
#ifdef QT_DEBUG
			qDebug() << "filter" << filter->name() << "time" << time.elapsed();
#endif
			if(preImage != 0)
				delete preImage;

			_images.insert(filter, image);
		}
	}

	if(!_stillDirty) {
		_changedParameters.clear();
		_changedProperties.clear();
		qDebug() << "clear";
	} else {
		qDebug() << "dontClear";
	}

	return image;
}

void ImageFilterChain::filterInBackground() {
	_workerThread->start();
}

bool ImageFilterChain::isRunning() {
	return _workerThread->isRunning();
}

void ImageFilterChain::setImage(Image* source) {
	closeImage();

	_source = source;

	emit imageChanged();
	emit imageChanged(_source);
}

void ImageFilterChain::closeImage() {
	_source = 0;

	clearCache();
}

void ImageFilterChain::clearCache() {
	foreach(Image* image, _images.values())
		delete image;

	_images.clear();
}

void ImageFilterChain::setParameter(QString key, QVariant value) {
	if(_workerThread->isRunning())
		_stillDirty = true;

	if(!value.isNull())
		_parameters.insert(key, value);
	else
		_parameters.remove(key);

	foreach(ImageFilter* filter, _filters)
		filter->setParameters(_parameters);

	_changedParameters.insert(key);

	emit parameterChanged(key, value);
	emit parametersChanged();
}

void ImageFilterChain::setParameters(BergPhoto::Parameters parameters) {
	blockSignals(true);

	foreach(QString key, parameters.keys())
		setParameter(key, parameters.value(key));

	blockSignals(false);

	foreach(QString key, parameters.keys())
		emit parameterChanged(key, parameters.value(key));

	emit parametersChanged();
}

void ImageFilterChain::metadataPropertyChange(MetadataProperty* property) {
	if(_workerThread->isRunning())
		_stillDirty = true;

	_changedProperties.insert(property);

	emit metadataPropertyChanged(property);
	emit propertiesChanged();
}

void ImageFilterChain::threadFinished(Image* image) {
	emit finished(image);
}

bool ImageFilterChain::isDirty(ImageFilter* filter) {
	foreach(QString key, _changedParameters) {
		if(filter->isParamterUsed(key))
			return true;
	}

	foreach(MetadataProperty* property, _changedProperties) {
		if(filter->isPropertyUsed(property))
			return true;
	}

	return false;
}

ImageFilterChainWorkerThread::ImageFilterChainWorkerThread(ImageFilterChain* imageFilterChain, QObject* parent) : QThread(parent) {
	_imageFilterChain = imageFilterChain;
}

void ImageFilterChainWorkerThread::start(QThread::Priority priority) {
	_run = true;

	if(!isRunning())
		QThread::start(priority);
}

void ImageFilterChainWorkerThread::run() {
	while(_run) {
		if(_imageFilterChain.isNull())
			return;

		_run = false;

		qDebug() << "filtering...";
		Image* image = _imageFilterChain->filter();

		if(image != 0) {
			emit finished(image);
			qDebug() << "ImgeFilterChainWorkerThread::finished()";
		}
	}
}
