/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef IMAGEFILTER_H_
#define IMAGEFILTER_H_

#include <QMap>
#include <QPointer>
#include <QSet>
#include <QString>
#include <QVariant>

#include <BergPhoto.h>
#include <Image.h>

class ImageFilterChainWorkerThread;

class ImageFilterSettings {
public:
	static QString namespaceUri;
	static MetadataQName propertySettings;
	static MetadataQName propertyName;
	static MetadataQName propertyDate;
};

class ImageFilter {
public:
	virtual ~ImageFilter();

	virtual QString name() = 0;

	virtual bool accepts(ImageFormat inputFormat, ImageFormat outputFormat) = 0;

	virtual bool isParamterUsed(QString key) = 0;

	virtual bool isPropertyUsed(MetadataProperty* property) = 0;

	virtual ImageFormat prepare(MetadataResource* metadata, ImageFormat inputFormat, ImageFormat outputFormat) = 0;

	virtual Image* filter(Image* input, ImageFormat outputFormat) = 0;

	virtual BergPhoto::Parameters parameters();

	virtual void setParameters(BergPhoto::Parameters parameters);

protected:
	BergPhoto::Parameters _parameters;
};

class ImageWriter {
public:
	static QString parameterImageWriter;
	static QString parameterFileFormat;
	static QString parameterCompress;
	static QString parameterQuality;

	virtual ~ImageWriter() {
	}

	virtual bool accepts(QString format) = 0;

	virtual bool write(QIODevice* device, Image* image, MetadataResource* resource, BergPhoto::Parameters parameters) = 0;

	virtual bool write(QString fileName, Image* image, MetadataResource* resource, BergPhoto::Parameters parameter);
};

class QtImageWriter : public ImageWriter {
public:
	static QString name;

	virtual bool accepts(QString format);

	virtual bool write(QIODevice* device, Image* image, MetadataResource* resource, BergPhoto::Parameters parameters);
};

class ImageFilterChain : public QObject {

	Q_OBJECT

public:
	ImageFilterChain(QObject* parent=0);
	~ImageFilterChain();

	void appendFilter(ImageFilter* filter, ImageFormat outputFormat);

	ImageFormat prepare();

	Image* filter();
	void filterInBackground();

	bool isRunning();

	Image* image() { return _source; }
	void setImage(Image* source);

	void closeImage();


public slots:
	void clearCache();

	void setParameter(QString key, QVariant value);
	void setParameters(BergPhoto::Parameters parameters);

	void metadataPropertyChange(MetadataProperty* property);

private slots:
	void threadFinished(Image* image);

signals:
	void parametersChanged();
	void propertiesChanged();
	void parameterChanged(QString, QVariant);
	void metadataPropertyChanged(MetadataProperty*);
	void imageChanged();
	void imageChanged(Image*);
	void finished(Image*);

protected:
	bool isDirty(ImageFilter* filter);

	Image* _source;
	QList<ImageFilter*> _filters;
	QMap<ImageFilter*, ImageFormat> _outputFormats;
	BergPhoto::Parameters _parameters;
	QSet<QString> _changedParameters;
	QSet<MetadataProperty*> _changedProperties;
	QMap<ImageFilter*, Image*> _images;
	QPointer<ImageFilterChainWorkerThread> _workerThread;
	volatile bool _stillDirty;
};

#endif /*IMAGEFILTER_H_*/
