/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ImagePreviewWidget.h"
#include "ImagePreviewWidget_p.h"

#include <QWheelEvent>

#ifdef USE_OPENGL
#include <QGLWidget>
#endif

#include <RawFile.h>
#include <filters/RotateCropFilter.h>

int ImagePreviewImagePositionEvent::_eventId = QEvent::registerEventType();

ImagePreviewFilterChainBridge::ImagePreviewFilterChainBridge(QObject* parent, ImageFilterChain* imageFilterChain) : QObject(parent) {
	setImageFilterChain(imageFilterChain);
}

void ImagePreviewFilterChainBridge::setImageFilterChain(ImageFilterChain* imageFilterChain) {
	if(_imageFilterChain != 0) {
		disconnect(_imageFilterChain, SIGNAL(imageChanged()), this, SLOT(refresh()));
		disconnect(_imageFilterChain, SIGNAL(metadataPropertyChanged(MetadataProperty*)), this, SLOT(metadataPropertyChange(MetadataProperty*)));
	}

	_imageFilterChain = imageFilterChain;

	if(_imageFilterChain != 0) {
		connect(_imageFilterChain, SIGNAL(imageChanged()), this, SLOT(refresh()));
		connect(_imageFilterChain, SIGNAL(metadataPropertyChanged(MetadataProperty*)), this, SLOT(metadataPropertyChange(MetadataProperty*)));
	}
}

MetadataNode* ImagePreviewFilterChainBridge::settingsCurrent() {
	return MetadataNode::parent(MetadataQuery::property(metadata(), ImageFilterSettings::propertyName, QVariant("current")));
}

void ImagePreviewFilterChainBridge::setPropertyValue(MetadataQName qName, QVariant value, MetadataNode* parent) {
	if(_imageFilterChain == 0)
		return;

	MetadataProperty* property = MetadataQuery::property(parent != 0 ? parent : metadata(), qName);

	if(property == 0 && parent != 0)
		property = new MetadataProperty(parent, qName);

	if(property == 0)
		return;

	property->setValue(value);

	_imageFilterChain->metadataPropertyChange(property);
}

void ImagePreviewFilterChainBridge::parameterChange(QString key, QVariant value) {
	if(key == "scale") {
		emit scaleChanged(value.toDouble());
		emit scaleChanged();
	}
}

void ImagePreviewFilterChainBridge::metadataPropertyChange(MetadataProperty* property) {
	if(property == 0)
		return;

	if(property->parent() == settingsCurrent()) {
		if(property->qName() == RotateCropFilter::propertyCropBottom) {
			emit cropBottomChanged(property->value().toInt());
			emit cropBottomChanged();
		} else if(property->qName() == RotateCropFilter::propertyCropLeft) {
			emit cropLeftChanged(property->value().toInt());
			emit cropLeftChanged();
		} else if(property->qName() == RotateCropFilter::propertyCropRight) {
			emit cropRightChanged(property->value().toInt());
			emit cropRightChanged();
		} else if(property->qName() == RotateCropFilter::propertyCropTop) {
			emit cropTopChanged(property->value().toInt());
			emit cropTopChanged();
		} else if(property->qName() == RotateCropFilter::propertyRotateAngle) {
			emit rotateAngleChanged(property->value().toDouble());
			emit rotateAngleChanged();
		}
	} else {
		if(property->qName() == RawFile::propertyDefaultSize) {
			emit defaultImageSizeChanged(property->value().toSize());
			emit defaultImageSizeChanged();
		} else if(property->qName() == Image::propertyOrientation) {
			emit orientationChanged(property->value().toDouble());
			emit orientationChanged();
		}
	}
}

void ImagePreviewFilterChainBridge::setScale(double scale) {
	if(_imageFilterChain.isNull())
		return;

	_imageFilterChain->setParameter("scale", scale);
}

void ImagePreviewFilterChainBridge::setDefaultImageSize(QSize defaultImageSize) {
	setPropertyValue(RawFile::propertyDefaultSize, defaultImageSize);
}

void ImagePreviewFilterChainBridge::setOrientation(double orientation) {
	setPropertyValue(Image::propertyOrientation, orientation);
}

void ImagePreviewFilterChainBridge::setRotateAngle(double angle) {
	setPropertyValue(RotateCropFilter::propertyRotateAngle, angle, settingsCurrent());
}

void ImagePreviewFilterChainBridge::setCropTop(int cropTop) {
	setPropertyValue(RotateCropFilter::propertyCropTop, cropTop, settingsCurrent());
}

void ImagePreviewFilterChainBridge::setCropLeft(int cropLeft) {
	setPropertyValue(RotateCropFilter::propertyCropLeft, cropLeft, settingsCurrent());
}

void ImagePreviewFilterChainBridge::setCropBottom(int cropBottom) {
	setPropertyValue(RotateCropFilter::propertyCropBottom, cropBottom, settingsCurrent());
}

void ImagePreviewFilterChainBridge::setCropRight(int cropRight) {
	setPropertyValue(RotateCropFilter::propertyCropRight, cropRight, settingsCurrent());
}

void ImagePreviewFilterChainBridge::refresh() {
	metadataPropertyChange(MetadataQuery::property(metadata(), RawFile::propertyDefaultSize));
	metadataPropertyChange(MetadataQuery::property(metadata(), Image::propertyOrientation));
	metadataPropertyChange(MetadataQuery::property(settingsCurrent(), RotateCropFilter::propertyRotateAngle));
	metadataPropertyChange(MetadataQuery::property(settingsCurrent(), RotateCropFilter::propertyCropTop));
	metadataPropertyChange(MetadataQuery::property(settingsCurrent(), RotateCropFilter::propertyCropLeft));
	metadataPropertyChange(MetadataQuery::property(settingsCurrent(), RotateCropFilter::propertyCropBottom));
	metadataPropertyChange(MetadataQuery::property(settingsCurrent(), RotateCropFilter::propertyCropRight));
}

MetadataResource* ImagePreviewFilterChainBridge::metadata() {
	if(_imageFilterChain == 0 || _imageFilterChain->image() == 0)
		return 0;
	else
		return _imageFilterChain->image()->metadata();
}

ImagePreviewWidget::ImagePreviewWidget(QWidget* parent) : QGraphicsView(parent) {
#ifdef USE_OPENGL
	setViewport(new QGLWidget());
	setRenderHint(QPainter::HighQualityAntialiasing, true);
#endif

	setRenderHint(QPainter::SmoothPixmapTransform, true);

	_graphicsScene = new QGraphicsScene(this);

	_graphicsImageItem = new ImagePreviewImageItem();
	_graphicsImageItem->setFlag(QGraphicsItem::ItemIsMovable, true);
	_graphicsImageItem->setEventHandler(this);
	_graphicsImageItem->setZValue(-1.0);
	_graphicsScene->addItem(_graphicsImageItem);

	_cropRectItem = new QGraphicsRectItem();
	_graphicsScene->addItem(_cropRectItem);

	_cropRectTopItem = new QGraphicsRectItem();
	_cropRectTopItem->setPen(Qt::NoPen);
	_graphicsScene->addItem(_cropRectTopItem);

	_cropRectLeftItem = new QGraphicsRectItem();
	_cropRectLeftItem->setPen(Qt::NoPen);
	_graphicsScene->addItem(_cropRectLeftItem);

	_cropRectBottomItem = new QGraphicsRectItem();
	_cropRectBottomItem->setPen(Qt::NoPen);
	_graphicsScene->addItem(_cropRectBottomItem);

	_cropRectRightItem = new QGraphicsRectItem();
	_cropRectRightItem->setPen(Qt::NoPen);
	_graphicsScene->addItem(_cropRectRightItem);

	setScene(_graphicsScene);

	_qImage = 0;

	setBackgroundBrush(Qt::gray);
	setViewCropLines(false);

	connect(&_defaultImageSize, SIGNAL(valueChanged()), this, SLOT(updateImageSize()));
	connect(&_orientation, SIGNAL(valueChanged()), this, SLOT(updateImageSize()));
	connect(&_rotateAngle, SIGNAL(valueChanged()), this, SLOT(updateImageSize()));

	_filterChainBridge = new ImagePreviewFilterChainBridge(this);

	connect(&_filterScale, SIGNAL(valueChanged(double)), _filterChainBridge, SLOT(setScale(double)));

	connect(_filterChainBridge, SIGNAL(defaultImageSizeChanged(QSize)), &_defaultImageSize, SLOT(setValue(QSize)));
	connect(_filterChainBridge, SIGNAL(orientationChanged(double)), &_orientation, SLOT(setValue(double)));
	connect(_filterChainBridge, SIGNAL(rotateAngleChanged(double)), &_rotateAngle, SLOT(setValue(double)));
	connect(_filterChainBridge, SIGNAL(cropTopChanged(int)), &_cropTop, SLOT(setValue(int)));
	connect(_filterChainBridge, SIGNAL(cropRightChanged(int)), &_cropRight, SLOT(setValue(int)));
	connect(_filterChainBridge, SIGNAL(cropBottomChanged(int)), &_cropBottom, SLOT(setValue(int)));
	connect(_filterChainBridge, SIGNAL(cropLeftChanged(int)), &_cropLeft, SLOT(setValue(int)));

	setScale(0.125);
}

ImagePreviewWidget::~ImagePreviewWidget() {
	if(_qImage != 0)
		delete _qImage;
}

void ImagePreviewWidget::setFilterChain(ImageFilterChain* filterChain) {
	if(!_filterChain.isNull()) {
		disconnect(_filterChain, SIGNAL(parametersChanged()), this, SLOT(renderImage()));
		disconnect(_filterChain, SIGNAL(propertiesChanged()), this, SLOT(renderImage()));
		disconnect(_filterChain, SIGNAL(finished(Image*)), this, SLOT(imageRendered(Image*)));
	}

	_filterChain = filterChain;

	if(!_filterChain.isNull()) {
		connect(_filterChain, SIGNAL(parametersChanged()), this, SLOT(renderImage()));
		connect(_filterChain, SIGNAL(propertiesChanged()), this, SLOT(renderImage()));
		connect(_filterChain, SIGNAL(finished(Image*)), this, SLOT(imageRendered(Image*)));

		_filterChainBridge->setImageFilterChain(_filterChain);

		_filterScale.setValue(qBound(0.001, _scale, 1.0));

		_filterChainBridge->refresh();
	}
}

void ImagePreviewWidget::setViewCropLines(bool flag) {
	_viewCropLines = flag;

	_cropRectItem->setVisible(_viewCropLines);

	_cropRectTopItem->setVisible(!_viewCropLines);
	_cropRectLeftItem->setVisible(!_viewCropLines);
	_cropRectBottomItem->setVisible(!_viewCropLines);
	_cropRectRightItem->setVisible(!_viewCropLines);
}

void ImagePreviewWidget::setScale(double scale) {
	if(scale == _scale)
		return;

	_scale = qBound(0.001, scale, 4.0);

	_widgetScale = _scale / _filterScale.value();

	if(_widgetScale > 1.0 || _widgetScale < 0.5) {
		_widgetScale = 1.0;

		_filterScale.setValue(qBound(0.001, _scale, 1.0));
	}

	updateImageItem();

	emit scaleChanged(_scale);
}

void ImagePreviewWidget::setBackgroundBrush(const QBrush& brush) {
	_cropRectTopItem->setBrush(brush);
	_cropRectLeftItem->setBrush(brush);
	_cropRectBottomItem->setBrush(brush);
	_cropRectRightItem->setBrush(brush);

	QGraphicsView::setBackgroundBrush(brush);
}

bool ImagePreviewWidget::event(QEvent* event) {
	if(event->type() == ImagePreviewImagePositionEvent::eventId()) {
		_imagePosition = ((ImagePreviewImagePositionEvent*)event)->pos();

		updateImageItem();

		return true;
	} else {
		return QGraphicsView::event(event);
	}
}

void ImagePreviewWidget::renderImage() {
	_filterChain->filterInBackground();
}

void ImagePreviewWidget::imageRendered(Image* image) {
	if(image == 0)
		return;

	RgbImage32* rgbImage32 = qobject_cast<RgbImage32*>(image);

	if(rgbImage32 == 0)
		return;

	if(_qImage != 0)
		delete _qImage;

	_qImage = rgbImage32->toQImage();

	_graphicsImageItem->setImage(_qImage);
	_graphicsImageItem->update();

	updateImageItem();
}

void ImagePreviewWidget::updateImageItem() {
	if(_graphicsImageItem->image() == 0)
		return;

	_graphicsImageItem->resetTransform();
	_graphicsImageItem->rotate(-_rotateAngle.value() - _orientation.value());
	_graphicsImageItem->scale(_widgetScale, _widgetScale);
	_graphicsImageItem->update();

	QRectF rect(QPointF(), _maxImageSize);
	rect.moveCenter(QPointF());

	QRectF rectCrop = rect.adjusted(_cropLeft.value(), _cropTop.value(), -_cropRight.value(), -_cropBottom.value());

	_cropRectItem->resetTransform();
	_cropRectItem->setRect(rectCrop.adjusted(-1, -1, 0, 0));
	_cropRectItem->translate(_imagePosition.x(), _imagePosition.y());
	_cropRectItem->scale(_scale, _scale);
	_cropRectItem->update();

	_cropRectTopItem->resetTransform();
	_cropRectTopItem->setRect(QRectF(rect.topRight(), rectCrop.topLeft()).adjusted(10, -10, -10, 0));
	_cropRectTopItem->translate(_imagePosition.x(), _imagePosition.y());
	_cropRectTopItem->scale(_scale, _scale);
	_cropRectTopItem->update();

	_cropRectLeftItem->resetTransform();
	_cropRectLeftItem->setRect(QRectF(rect.topLeft(), rectCrop.bottomLeft()).adjusted(-10, -10, 0, 10));
	_cropRectLeftItem->translate(_imagePosition.x(), _imagePosition.y());
	_cropRectLeftItem->scale(_scale, _scale);
	_cropRectLeftItem->update();

	_cropRectBottomItem->resetTransform();
	_cropRectBottomItem->setRect(QRectF(rect.bottomLeft(), rectCrop.bottomRight()).adjusted(-10, 10, 10, 0));
	_cropRectBottomItem->translate(_imagePosition.x(), _imagePosition.y());
	_cropRectBottomItem->scale(_scale, _scale);
	_cropRectBottomItem->update();

	_cropRectRightItem->resetTransform();
	_cropRectRightItem->setRect(QRectF(rect.bottomRight(), rectCrop.topRight()).adjusted(10, 10, 0, -10));
	_cropRectRightItem->translate(_imagePosition.x(), _imagePosition.y());
	_cropRectRightItem->scale(_scale, _scale);
	_cropRectRightItem->update();
}

void ImagePreviewWidget::updateImageSize() {
	double mc = cos((_rotateAngle.value() + _orientation.value()) * (M_PI/180.0));
	double ms = sin((_rotateAngle.value() + _orientation.value()) * (M_PI/180.0));

	int defaultWidth = _defaultImageSize.value().width();
	int defaultHeight = _defaultImageSize.value().height();

	int width = (int)(fabs(defaultWidth*mc) + fabs(defaultHeight*ms) + 0.5);
	int height = (int)(fabs(defaultHeight*mc) + fabs(defaultWidth*ms) + 0.5);

	QSize maxImageSize = QSize(width, height);

	updateImageItem();

	if(maxImageSize != _maxImageSize) {
		_maxImageSize = maxImageSize;

		emit maxImageSizeChanged(_maxImageSize);
	}
}

void ImagePreviewWidget::resizeEvent(QResizeEvent* resizeEvent) {
	QRectF sceneRect = QRectF(QPointF(), resizeEvent->size());
	sceneRect.moveCenter(QPointF());

	setSceneRect(sceneRect);

	QRectF imageRect = QRectF(QPointF(), _maxImageSize * _scale);
	imageRect.moveCenter(QPointF());
	imageRect.translate(_imagePosition.x(), _imagePosition.y());

	if(!sceneRect.intersects(imageRect)) {
		_imagePosition = QPointF(0.0, 0.0);
		_graphicsImageItem->setPos(_imagePosition);
	}
}

void ImagePreviewWidget::wheelEvent(QWheelEvent* wheelEvent) {
	setScale(_scale + (double)wheelEvent->delta() * 0.0001);
}
