/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "Metadata.h"

MetadataNode::MetadataNode(MetadataNode* parent) {
	_parent = parent;

	if(_parent != 0)
		_parent->_children.append(this);
}

MetadataNode::~MetadataNode() {
	foreach(MetadataNode* node, _children) {
		delete node;
	}

	if(_parent != 0)
		_parent->_children.removeAll(this);
}

void MetadataNode::setParent(MetadataNode* parent) {
	if(_parent != 0)
		_parent->_children.removeAll(this);

	_parent = parent;
	_parent->_children.append(this);
}

MetadataNode* MetadataNode::clone(MetadataNode* parent, bool deep) const {
	MetadataNode* clone = new MetadataNode(parent);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

QList<MetadataNode*> MetadataNode::cloneChildren(MetadataNode* parent, bool deep) const {
	QList<MetadataNode*> clones;

	foreach(MetadataNode* child, _children) {
		 clones.append(child->clone(parent, deep));
	}

	return clones;
}

MetadataNode* MetadataDocument::clone(MetadataNode* parent, bool deep) const {
	MetadataDocument* clone = new MetadataDocument(parent);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

MetadataNode* MetadataResource::clone(MetadataNode* parent, bool deep) const {
	MetadataResource* clone = new MetadataResource(parent, _resourceUri);

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

MetadataNode* MetadataProperty::clone(MetadataNode* parent, bool deep) const {
	MetadataProperty* clone = new MetadataProperty(parent, _qName);

	clone->_values = _values;

	if(deep)
		clone->_children = cloneChildren(clone, deep);

	return clone;
}

QList<MetadataResource*> MetadataQuery::resources(MetadataNode* root, QString resourceUri) {
	QList<MetadataResource*> list;

	if(resourceUri == "*") {
		foreach(MetadataNode* node, root->children()) {
			if(node->isResource())
				list.append(node->toResource());
		}

		foreach(MetadataNode* node, root->children()) {
			if(node->isDocument())
				list += resources(node, resourceUri);
		}
	} else {
		foreach(MetadataNode* node, root->children()) {
			if(node->isResource() && node->toResource()->resourceUri() == resourceUri)
				list.append(node->toResource());
		}

		foreach(MetadataNode* node, root->children()) {
			if(node->isDocument() && node->toResource()->resourceUri() == resourceUri)
				list += resources(node, resourceUri);
		}
	}

	return list;
}

MetadataResource* MetadataQuery::resource(MetadataNode* root, QString resourceUri) {
	QList<MetadataResource*> list = resources(root, resourceUri);

	if(list.isEmpty())
		return 0;
	else
		return list.at(0);
}

QList<MetadataProperty*> MetadataQuery::properties(MetadataNode* root, MetadataQName qName, QVariant value) {
	QList<MetadataProperty*> list;

	if(root == 0)
		return list;

	foreach(MetadataNode* node, root->children()) {
		if(node->isProperty()) {
			MetadataProperty* property = node->toProperty();

			if((qName.namespaceUri() == "*" || property->qName().namespaceUri() == qName.namespaceUri()) &&
					(qName.localName() == "*" || property->qName().localName() == qName.localName()) &&
					(value.isNull() || property->values().contains(value)))
				list.append(property);
		}

		list += properties(node, qName, value);
	}

	return list;
}

MetadataProperty* MetadataQuery::property(MetadataNode* root, MetadataQName qName, QVariant value) {
	QList<MetadataProperty*> list = properties(root, qName, value);

	if(list.isEmpty())
		return 0;
	else
		return list.at(0);
}

QList<QVariant> MetadataQuery::values(MetadataNode* root, MetadataQName qName) {
	QList<QVariant> list;

	foreach(MetadataProperty* property, MetadataQuery::properties(root, qName)) {
		list += property->values();
	}

	return list;
}

QVariant MetadataQuery::value(MetadataNode* root, MetadataQName qName) {
	QList<QVariant> list = MetadataQuery::values(root, qName);

	if(list.isEmpty())
		return 0;
	else
		return list.at(0);
}
