/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef METADATA_H_
#define METADATA_H_

#include <QHash>
#include <QList>
#include <QVariant>

#include <math.h>

#include <QtUtils.h>

class Rational {
public:
	Rational() { _numerator=0; _denominator=0; }
	Rational(qlonglong numerator, qlonglong denominator) { _numerator=numerator; _denominator=denominator; }
	Rational(double d, qlonglong denominator) { _denominator=denominator; d<0.0 ? _numerator=(int)(d*_denominator-0.5) : _numerator=(int)(d*_denominator+0.5); }

	qlonglong numerator() const { return _numerator; }
	void setNumerator(qlonglong numerator) { _numerator=numerator; }

	qlonglong denominator() const { return _denominator; }
	void setDenominator(qlonglong denominator) { _denominator=denominator; }

	QString toString() const { return QString("%1/%2").arg(_numerator).arg(_denominator); }
	double toDouble() const { return (double)_numerator/(double)_denominator; }

private:
	qlonglong _numerator;
	qlonglong _denominator;
};

Q_DECLARE_METATYPE(Rational)

class MetadataQName {
public:
	MetadataQName() {}
	MetadataQName(QString namespaceUri, QString localName) { _namespaceUri=namespaceUri; _localName=localName; }

	QString namespaceUri() const { return _namespaceUri; }
	void setNamespaceUri(QString namespaceUri) { _namespaceUri=namespaceUri; }

	QString localName() const { return _localName; }
	void setLocalName(QString localName) { _localName=localName; }

	bool operator !=(const MetadataQName o) const { return _namespaceUri != o._namespaceUri || _localName != o._localName; }
	bool operator <(const MetadataQName o) const { return _namespaceUri == o._namespaceUri ? _localName < o._localName : _namespaceUri < o._namespaceUri; }
	bool operator ==(const MetadataQName o) const { return _namespaceUri == o._namespaceUri && _localName == o._localName; }

private:
	QString _namespaceUri;
	QString _localName;
};

inline uint qHash(const MetadataQName& qName) {
	return qHash(qName.namespaceUri() + qName.localName());
}

Q_DECLARE_METATYPE(MetadataQName)

class MetadataNode;
class MetadataDocument;
class MetadataResource;
class MetadataProperty;

class MetadataNode {
public:
	MetadataNode(MetadataNode* parent=0);
	virtual ~MetadataNode();

	MetadataNode* parent() const { return _parent; }
	void setParent(MetadataNode* parent);

	QList<MetadataNode*> children() const { return _children; }

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true) const;

	virtual bool isDocument() const { return false; }
	virtual bool isResource() const { return false; }
	virtual bool isProperty() const { return false; }

	MetadataDocument* toDocument() { return reinterpret_cast<MetadataDocument*>(this); }
	MetadataResource* toResource() { return reinterpret_cast<MetadataResource*>(this); }
	MetadataProperty* toProperty() { return reinterpret_cast<MetadataProperty*>(this); }

	static MetadataNode* parent(MetadataNode* node) { return node == 0 ? 0 : node->parent(); }

	static MetadataDocument* toDocument(MetadataNode* node) { return node == 0 ? 0 : node->toDocument(); }
	static MetadataResource* toResource(MetadataNode* node) { return node == 0 ? 0 : node->toResource(); }
	static MetadataProperty* toProperty(MetadataNode* node) { return node == 0 ? 0 : node->toProperty(); }

protected:
	QList<MetadataNode*> cloneChildren(MetadataNode* parent, bool deep=true) const;

	MetadataNode* _parent;
	QList<MetadataNode*> _children;
};

class MetadataDocument : public MetadataNode {
public:
	MetadataDocument(MetadataNode* parent=0) : MetadataNode(parent) {}

	virtual bool isDocument() const { return true; }

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true) const;
};

class MetadataResource : public MetadataNode {
public:
	MetadataResource(MetadataNode* parent, QString resourceUri=QString()) : MetadataNode(parent) { _resourceUri=resourceUri; }

	virtual bool isResource() const { return true; }

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true) const;

	QString resourceUri() const { return _resourceUri; }
	void setResourceUri(QString resourceUri) { _resourceUri=resourceUri; }

protected:
	QString _resourceUri;
};

class MetadataProperty : public MetadataNode {
public:
	MetadataProperty(MetadataNode* parent, MetadataQName qName) : MetadataNode(parent) { _qName=qName; }
	MetadataProperty(MetadataNode* parent, MetadataQName qName, QVariant value) : MetadataNode(parent) { _qName=qName; _values.append(value); }

	virtual bool isProperty() const { return true; }

	virtual MetadataNode* clone(MetadataNode* parent=0, bool deep=true) const;

	MetadataQName qName() const { return _qName; }
	void setQName(MetadataQName qName) { _qName=qName; }

	QList<QVariant> values() const { return _values; }
	void setValues(QList<QVariant> values) { _values = values; }
	void appendValues(QList<QVariant> values) { foreach(QVariant value, values) _values.append(value); }

	QVariant value() const { return _values.isEmpty() ? QVariant() : _values.at(0); }
	void setValue(QVariant value) { _values.clear(); _values.append(value); }
	void appendValue(QVariant value) { _values.append(value); }

private:
	MetadataQName _qName;
	QList<QVariant> _values;
};

class MetadataQuery {
public:
	static QList<MetadataResource*> resources(MetadataNode* root, QString resourceUri);
	static MetadataResource* resource(MetadataNode* root, QString resourceUri);

	static QList<MetadataProperty*> properties(MetadataNode* root, MetadataQName qName, QVariant value=QVariant());
	static MetadataProperty* property(MetadataNode* root, MetadataQName qName, QVariant value=QVariant());

	static QList<QVariant> values(MetadataNode* root, MetadataQName qName);
	static QVariant value(MetadataNode* root, MetadataQName qName);

	template <typename T> static QList<T> typedValues(MetadataNode* root, MetadataQName qName) { return QtUtils::toTypedList<T>(values(root, qName)); }
	template <typename T> static T typedValue(MetadataNode* root, MetadataQName qName) { QVariant v=value(root, qName); return v.value<T>(); }
};

class MetadataTransform {
public:
	MetadataTransform() {}
	MetadataTransform(QVariant parameter) { _parameter=parameter; }
	virtual ~MetadataTransform() {}

	QVariant parameter() { return _parameter; };
	void setParameter(QVariant parameter) { _parameter=parameter; }

	virtual MetadataNode* transform(MetadataNode* source) = 0;

protected:
	QVariant _parameter;
};

class MetadataReader {
public:
	virtual MetadataNode* read(MetadataNode* parent=0) = 0;
};

class MetadataWriter {
public:
	virtual void write(MetadataNode* node) = 0;
};

#endif /*METADATA_H_*/
