/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "MetadataExifTool.h"

#include <QDebug>
#include <QProcess>
#include <QStringList>
#include <QTextStream>

#include <formats/DNG.h>
#include <formats/Tiff.h>

QString MetadataExifTool::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/exiftool/1.0/";
QMap<QString, MetadataQName> MetadataExifTool::_keyQNameMap = QMap<QString, MetadataQName>();

MetadataExifTool::MetadataExifTool(QString fileName, QString exifTool) {
	_fileName = fileName;
	_exifTool = exifTool;
}

MetadataExifTool::~MetadataExifTool() {
}

void MetadataExifTool::setFileName(QString fileName) {
	_fileName = fileName;
}

void MetadataExifTool::setExifTool(QString exifTool) {
	_exifTool = exifTool;
}

MetadataNode* MetadataExifTool::read(MetadataNode* parent) {
	QStringList arguments;

	arguments.append("-e");
	arguments.append("-n");
	arguments.append("-S");
	arguments.append(_fileName);

	QProcess process;

	QString exifTool = _exifTool.isEmpty() ? "exiftool" : _exifTool;

	process.start(exifTool, arguments);
	process.waitForFinished();

	QTextStream textStream(&process);

	MetadataNode* root = parent;

	if(parent == 0) {
		root = new MetadataDocument();
		parent = new MetadataResource(root);
	}

	while(!textStream.atEnd()) {
		QString line = textStream.readLine();

		int seperator = line.indexOf(":");
		QString key = line.left(seperator);
		QString value = line.mid(seperator+2);

		qDebug() << key << value;

		MetadataQName qName;

		if(keyQNameMap().contains(key))
			qName = keyQNameMap().value(key);
		else
			qName = MetadataQName(namespaceUri, key);

		QVariant variant = translateValue(key, value);

		if(variant.canConvert(QVariant::List)) {
			MetadataProperty* listProperty = new MetadataProperty(parent, qName);

			foreach(QVariant item, variant.toList())
				listProperty->appendValue(item);
		} else {
			new MetadataProperty(parent, qName, variant);
		}
	}

	return root;
}

MetadataNode* MetadataExifTool::readFile(QString fileName, QString exifTool) {
	MetadataExifTool metadataExifTool(fileName, exifTool);

	return metadataExifTool.read();
}

QVariant MetadataExifTool::translateValue(QString key, QString value) {
	if(key == "CameraCalibration1" || key == "CameraCalibration2" || key == "ColorMatrix1" || key == "ColorMatrix2") {
		QStringList values = value.split(' ');
		QList<QVariant> variantList;
		foreach(QString item, values) {
			QVariant variant;
			variant.setValue<Rational>(Rational(item.toDouble(), 10000));
			variantList.append(variant);
		}
		return QVariant(variantList);
	} else if(key == "ActiveArea" || key == "MaskedAreas" || key == "WB_RGGBLevelsDaylight") {
		QStringList values = value.split(' ');
		QList<QVariant> variantList;
		foreach(QString item, values)
			variantList.append(item.toInt());
		return QVariant(variantList);
	} else if(key == "CalibrationIlluminant1" || key == "CalibrationIlluminant2") {
		return QVariant(Tiff::lightSourceTemperature(value.toUShort()));
	} else {
		return QVariant(value);
	}
}

QMap<QString, MetadataQName> MetadataExifTool::keyQNameMap() {
	if(_keyQNameMap.isEmpty()) {
		_keyQNameMap.insert("Make", Tiff::propertyMake);
		_keyQNameMap.insert("Model", Tiff::propertyModel);
		_keyQNameMap.insert("ColorMatrix1", DNG::propertyColorMatrix1);
		_keyQNameMap.insert("ColorMatrix2", DNG::propertyColorMatrix2);
		_keyQNameMap.insert("ActiveArea", DNG::propertyActiveArea);
		_keyQNameMap.insert("MaskedAreas", DNG::propertyMaskedAreas);
		_keyQNameMap.insert("CameraCalibration1", DNG::propertyCameraCalibration1);
		_keyQNameMap.insert("CameraCalibration2", DNG::propertyCameraCalibration2);
		_keyQNameMap.insert("CalibrationIlluminant1", DNG::propertyCalibrationIlluminant1);
		_keyQNameMap.insert("CalibrationIlluminant2", DNG::propertyCalibrationIlluminant2);

		_keyQNameMap.insert("WB_RGGBLevelsDaylight", MetadataQName(namespaceUri, "WB_RGGBLevelsDaylight"));
	}

	return _keyQNameMap;
}
