/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "MetadataRdf.h"

#include <QBuffer>
#include <QDebug>
#include <QDir>
#include <QPointF>
#include <QSet>
#include <QStringList>

#if QT_VERSION >= 0x040400

#include <QXmlQuery>
#include <QXmlSerializer>

#endif

#include <QtUtils.h>

QString MetadataRdfStringType::name = "string";

MetadataQName MetadataRdfStringType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriXmlSchema, name);
}

void MetadataRdfStringType::write(QVariant value, QXmlStreamWriter* writer) {
	writer->writeCharacters(value.toString());
}

QVariant MetadataRdfStringType::read(QXmlStreamReader* reader) {
	return reader->text().toString();
}

QString MetadataRdfBinaryType::name = "base64Binary";

MetadataQName MetadataRdfBinaryType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriXmlSchema, name);
}

void MetadataRdfBinaryType::write(QVariant value, QXmlStreamWriter* writer) {
	writer->writeCharacters(value.toByteArray().toBase64());
}

QVariant MetadataRdfBinaryType::read(QXmlStreamReader* reader) {
	return QVariant(QByteArray::fromBase64(reader->text().toString().toAscii()));
}

QString MetadataRdfIntegerType::name = "integer";

MetadataQName MetadataRdfIntegerType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriXmlSchema, name);
}

void MetadataRdfIntegerType::write(QVariant value, QXmlStreamWriter* writer) {
	writer->writeCharacters(value.toString());
}

QVariant MetadataRdfIntegerType::read(QXmlStreamReader* reader) {
	QString text = reader->text().toString();

	if(text.toLongLong() == text.toInt())
		return QVariant(text.toInt());
	else
		return QVariant(text.toLongLong());
}

QString MetadataRdfDoubleType::name = "double";

MetadataQName MetadataRdfDoubleType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriXmlSchema, name);
}

void MetadataRdfDoubleType::write(QVariant value, QXmlStreamWriter* writer) {
	writer->writeCharacters(value.toString());
}

QVariant MetadataRdfDoubleType::read(QXmlStreamReader* reader) {
	QString text = reader->text().toString();

	return QVariant(text.toDouble());
}

QString MetadataRdfRationalType::name = "rational";

MetadataQName MetadataRdfRationalType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriBergnetTypes, name);
}

void MetadataRdfRationalType::write(QVariant value, QXmlStreamWriter* writer) {
	writer->writeCharacters(value.value<Rational>().toString());
}

QVariant MetadataRdfRationalType::read(QXmlStreamReader* reader) {
	QStringList strings = reader->text().toString().split('/');

	QVariant value;
	value.setValue(Rational(strings.at(0).toLongLong(), strings.at(1).toLongLong()));

	return value;
}

QString MetadataRdfPointType::name = "point";

MetadataQName MetadataRdfPointType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriBergnetTypes, name);
}

void MetadataRdfPointType::write(QVariant value, QXmlStreamWriter* writer) {
	QPointF point = value.toPointF();

	writer->writeCharacters(QString("%1, %2").arg(point.x()).arg(point.y()));
}

QVariant MetadataRdfPointType::read(QXmlStreamReader* reader) {
	QStringList list = reader->text().toString().split(',');

	return QVariant(QPointF(list.at(0).toDouble(), list.at(1).toDouble()));
}

QString MetadataRdfSize2dType::name = "size2d";

MetadataQName MetadataRdfSize2dType::qName() {
	return MetadataQName(MetadataRdfSchema::namespaceUriBergnetTypes, name);
}

void MetadataRdfSize2dType::write(QVariant value, QXmlStreamWriter* writer) {
	QSize size = value.toSize();

	writer->writeCharacters(QString("%1, %2").arg(size.width()).arg(size.height()));
}

QVariant MetadataRdfSize2dType::read(QXmlStreamReader* reader) {
	QStringList list = reader->text().toString().split(',');

	return QVariant(QSize(list.at(0).toInt(), list.at(1).toInt()));
}

QString MetadataRdfSchema::namespaceUriXmlSchema = "http://www.w3.org/2001/XMLSchema";
QString MetadataRdfSchema::namespaceUriRdf = "http://www.w3.org/1999/02/22-rdf-syntax-ns#";
QString MetadataRdfSchema::namespaceUriBergnetTypes = "http://ns.bergnet.org/bergphoto/rdf/types/1.0/";
QString MetadataRdfSchema::namespaceUriRdfDefinition = "http://ns.bergnet.org/bergphoto/rdf/definition/1.0/";

MetadataRdfSchema* MetadataRdfSchema::_instance = 0;

MetadataRdfSchema* MetadataRdfSchema::instance() {
	if(_instance == 0)
		_instance = new MetadataRdfSchema();

	return _instance;
}

MetadataRdfSchema::MetadataRdfSchema() {
	_stringType = new MetadataRdfStringType();
	insertType(_stringType);

	_binaryType = new MetadataRdfBinaryType();
	insertType(_binaryType);

	_integerType = new MetadataRdfIntegerType();
	insertType(_integerType);

	_doubleType = new MetadataRdfDoubleType();
	insertType(_doubleType);

	_rationalType = new MetadataRdfRationalType();
	insertType(_rationalType);

	_pointType = new MetadataRdfPointType();
	insertType(_pointType);

	_size2dType = new MetadataRdfSize2dType();
	insertType(_size2dType);
}

MetadataRdfSchema::~MetadataRdfSchema() {
	delete _stringType;
	delete _binaryType;
	delete _integerType;
	delete _doubleType;
	delete _rationalType;
	delete _pointType;
	delete _size2dType;
}

MetadataRdfType* MetadataRdfSchema::type(MetadataQName qName) {
	return _types.value(qName);
}

void MetadataRdfSchema::insertType(MetadataRdfType* type) {
	_types.insert(type->qName(), type);
}

MetadataRdfType* MetadataRdfSchema::property(MetadataQName qName) {
	return _properties.value(qName);
}

void MetadataRdfSchema::insertProperty(MetadataQName qName, MetadataRdfType* type) {
	_properties.insert(qName, type);
}

void MetadataRdfSchema::insertProperty(MetadataQName qName, MetadataQName typeQName) {
	_properties.insert(qName, type(typeQName));
}

bool MetadataRdfSchema::containsProperty(MetadataQName qName) {
	return _properties.contains(qName);
}

void MetadataRdfSchema::removeProperty(MetadataQName qName) {
	_properties.remove(qName);
}

void MetadataRdfDefinitionReader::readFolder(QString folder) {
	QDir dir(folder);

	foreach(QString filename, dir.entryList(QStringList("*.xml"))) {
		QFile file(dir.filePath(filename));
		file.open(QIODevice::ReadOnly);
		MetadataRdfDefinitionReader definitionReader(&file);
		definitionReader.read();
		file.close();
	}
}

MetadataRdfDefinitionReader::MetadataRdfDefinitionReader(QIODevice* device) {
	_device = device;
}

bool MetadataRdfDefinitionReader::read() {
	QXmlStreamReader reader(_device);

	MetadataRdfSchema* schema = MetadataRdfSchema::instance();

	QString typeNamespaceUri;
	QString propertyNamespaceUri;
	QList<MetadataQName> propertyDefinitions;

	while(!reader.atEnd()) {
		reader.readNext();

		if(reader.isStartElement() && reader.namespaceUri() == MetadataRdfSchema::namespaceUriRdfDefinition) {
			QString localName = reader.name().toString();

			if(localName == "Types") {
				typeNamespaceUri = reader.attributes().value("namespace").toString();
			} else if(localName == "Type") {
				QString id = reader.attributes().value("id").toString();
				QString typeName = reader.attributes().value("name").toString();

				_typeNamespaceMap.insert(id, typeNamespaceUri);
				_typeLocalNameMap.insert(id, typeName);
			} else if(localName == "Properties") {
				propertyNamespaceUri = reader.attributes().value("namespace").toString();
			} else if(localName == "Property") {
				QString typeId = reader.attributes().value("typeId").toString();
				QString propertyName = reader.attributes().value("name").toString();

				MetadataQName typeQName(_typeNamespaceMap.value(typeId), _typeLocalNameMap.value(typeId));
				MetadataQName qName(propertyNamespaceUri, propertyName);

				schema->insertProperty(qName, typeQName);
				propertyDefinitions.append(qName);
			}
		} else if(reader.isEndElement() && reader.namespaceUri() == MetadataRdfSchema::namespaceUriRdfDefinition) {
			QString localName = reader.name().toString();

			if(localName == "Types")
				typeNamespaceUri = QString();
			else if(localName == "Properties")
				propertyNamespaceUri = QString();
		}
	}

	if(reader.hasError()) {
		foreach(MetadataQName qName, propertyDefinitions) {
			schema->removeProperty(qName);
		}

		qWarning() << QtUtils::fileName(reader.device()) << "error parsing xml document" << reader.errorString();

		return false;
	}

	return true;
}

MetadataNode* MetadataRdfReader::readFile(QString fileName) {
	QFile file(fileName);
	file.open(QIODevice::ReadOnly);
	MetadataRdfReader rdfReader(&file);
	MetadataNode* parent = rdfReader.read();
	file.close();

	return parent;
}

MetadataNode* MetadataRdfReader::readFolder(QString folder, MetadataNode* parent) {
	QDir dir(folder);

	foreach(QString filename, dir.entryList(QStringList("*.xml"))) {
		QFile file(dir.filePath(filename));
		file.open(QIODevice::ReadOnly);
		MetadataRdfReader rdfReader(&file);
		parent = rdfReader.read(parent);
		file.close();
	}

	return parent;
}

MetadataRdfReader::MetadataRdfReader(QIODevice* device) {
	_reader = new QXmlStreamReader(device);
}

MetadataRdfReader::~MetadataRdfReader() {
	delete _reader;
}

MetadataNode* MetadataRdfReader::read(MetadataNode* parent) {
	MetadataRdfSchema* schema = MetadataRdfSchema::instance();
	MetadataNode* root = 0;

	while(!_reader->atEnd()) {
		_reader->readNext();

		if(_reader->isStartElement()) {
			QString namespaceUri = _reader->namespaceUri().toString();
			QString localName = _reader->name().toString();

			if(namespaceUri == MetadataRdfSchema::namespaceUriRdf) {
				if(localName == "RDF") {
					if(parent == 0 || !parent->isDocument())
						root = parent = new MetadataDocument();
				} else if(localName == "Description") {
					QString resourceUri = _reader->attributes().value(MetadataRdfSchema::namespaceUriRdf, "about").toString();
					parent = new MetadataResource(parent, resourceUri);
					if(root == 0) root = parent;
				}
			} else {
				MetadataQName qName(namespaceUri, localName);

				if(schema->containsProperty(qName)) {
					parent = new MetadataProperty(parent, qName);
					if(root == 0) root = parent;
				} else {
					if(parent->isProperty()) {
						MetadataProperty* property = parent->toProperty();
						MetadataRdfType* type = schema->property(property->qName());
						property->setValue(type->read(_reader));
					}
				}
			}
		} else if(_reader->isEndElement()) {
			QString namespaceUri = _reader->namespaceUri().toString();
			QString localName = _reader->name().toString();

			if(namespaceUri == MetadataRdfSchema::namespaceUriRdf) {
				if(localName == "Description")
					parent = parent->parent();
			} else if(schema->containsProperty(MetadataQName(namespaceUri, localName))) {
				parent = parent->parent();
			}
		} else if(_reader->isCharacters() && ! _reader->isWhitespace()) {
			if(parent->isProperty()) {
				MetadataProperty* property = parent->toProperty();
				MetadataRdfType* type = schema->property(property->qName());
				property->appendValue(type->read(_reader));
			}
		}
	}

	if(_reader->hasError()) {
		delete root;

		qWarning() << QtUtils::fileName(_reader->device()) << "error parsing xml document" << _reader->errorString();

		return 0;
	}

	return parent;
}

void MetadataRdfWriter::writeFile(MetadataNode* node, QString filename) {
	QFile file(filename);
	file.open(QIODevice::WriteOnly);
	MetadataRdfWriter writer(&file);
	writer.write(node);
	file.close();
}

MetadataRdfWriter::MetadataRdfWriter(QIODevice* device) {
	_writer = new QXmlStreamWriter(device);

	_writer->setAutoFormatting(true);
	_writer->writeStartDocument();

	_writeUnkownProperties = true;
}

MetadataRdfWriter::~MetadataRdfWriter() {
	_writer->writeEndElement();

	delete _writer;
}

void MetadataRdfWriter::setPrefix(QString namespaceUri, QString prefix) {
	_prefix.insert(namespaceUri, prefix);
}

void MetadataRdfWriter::setWriteUnknownProperties(bool flag) {
	_writeUnkownProperties = flag;
}

void MetadataRdfWriter::write(MetadataNode* node) {
	MetadataRdfSchema* schema = MetadataRdfSchema::instance();

	if(node->isDocument()) {
		_writer->writeNamespace(MetadataRdfSchema::namespaceUriRdf, "rdf");
		_writer->writeStartElement(MetadataRdfSchema::namespaceUriRdf, "RDF");

		write(node->children());

		_writer->writeEndDocument();
	} else if(node->isResource()) {
		_writer->writeStartElement(MetadataRdfSchema::namespaceUriRdf, "Description");
		_writer->writeAttribute(MetadataRdfSchema::namespaceUriRdf, "about", node->toResource()->resourceUri());

		QSet<QString> namespaceUriSet;

		foreach(MetadataNode* child, node->children()) {
			if(child->isProperty()) {
				QString namespaceUri = child->toProperty()->qName().namespaceUri();

				if(!_prefix.contains(namespaceUri))
					namespaceUriSet.insert(namespaceUri);
			}
		}

		foreach(QString namespaceUri, _prefix.keys()) {
			_writer->writeNamespace(namespaceUri, _prefix.value(namespaceUri));
		}

		int i=0;
		foreach(QString namespaceUri, namespaceUriSet) {
			_writer->writeNamespace(namespaceUri, QString("t%1").arg(i));
			i++;
		}

		write(node->children());

		_writer->writeEndElement();
	} else if(node->isProperty()) {
		MetadataProperty* property = node->toProperty();
		MetadataRdfType* rdfType = schema->property(property->qName());

		if(rdfType != 0 || _writeUnkownProperties) {
			_writer->writeStartElement(property->qName().namespaceUri(), property->qName().localName());

			int itemCount = property->values().size();

			if(itemCount == 1) {
				if(rdfType != 0)
					rdfType->write(property->value(), _writer);
				else
					_writer->writeCharacters(property->value().toString());
			} else if(itemCount > 1) {
				_writer->writeStartElement(MetadataRdfSchema::namespaceUriRdf, "Seq");

				foreach(QVariant value, property->values()) {
					_writer->writeStartElement(MetadataRdfSchema::namespaceUriRdf, "li");

					if(rdfType != 0)
						rdfType->write(value, _writer);
					else
						_writer->writeCharacters(value.toString());

					_writer->writeEndElement();
				}

				_writer->writeEndElement();
			}

			write(node->children());

			_writer->writeEndElement();
		} else {
			qDebug() << "property type not defined in schema:" << property->qName().namespaceUri() << property->qName().localName();
		}
	}
}

void MetadataRdfWriter::write(QList<MetadataNode*> nodes) {
	foreach(MetadataNode* node, nodes) {
		write(node);
	}
}

#if QT_VERSION >= 0x040400

MetadataNode* MetadataRdfXQueryTransform::transform(MetadataNode* source) {
	QBuffer sourceDevice;
	sourceDevice.open(QIODevice::ReadWrite);
	MetadataRdfWriter rdfWriter(&sourceDevice);
	rdfWriter.write(source);

	QString queryString = _parameter.toString();

	QXmlQuery query;
	query.bindVariable("root", &sourceDevice);
	query.setQuery(queryString);

	QBuffer targetDevice;
	targetDevice.open(QIODevice::ReadWrite);
	QXmlSerializer serializer(query, &targetDevice);

	sourceDevice.seek(0);

	query.evaluateTo(&serializer);

	sourceDevice.close();

	targetDevice.seek(0);
	qDebug() << QString(targetDevice.readAll());

	targetDevice.seek(0);
	MetadataRdfReader rdfReader(&targetDevice);
	MetadataNode* target = rdfReader.read();

	targetDevice.close();

	return target;
}

#endif
