/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef METADATARDF_H_
#define METADATARDF_H_

#include <QIODevice>
#include <QList>
#include <QMap>
#include <QString>
#include <QVariant>
#include <QXmlStreamReader>
#include <QXmlStreamWriter>

#include <Metadata.h>

class MetadataRdfType {
public:
	virtual ~MetadataRdfType() {
	}

	virtual MetadataQName qName() = 0;

	virtual void write(QVariant value, QXmlStreamWriter* writer) = 0;
	virtual QVariant read(QXmlStreamReader* reader) = 0;
};

class MetadataRdfStringType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfBinaryType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfIntegerType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfDoubleType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfRationalType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfPointType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfSize2dType : public MetadataRdfType {
public:
	QString static name;

	virtual MetadataQName qName();

	virtual void write(QVariant value, QXmlStreamWriter* writer);
	virtual QVariant read(QXmlStreamReader* reader);
};

class MetadataRdfSchema : QObject {

	Q_OBJECT

public:
	static QString namespaceUriXmlSchema;
	static QString namespaceUriBergnetTypes;
	static QString namespaceUriRdf;
	static QString namespaceUriRdfDefinition;

	static MetadataRdfSchema* instance();

	~MetadataRdfSchema();

	MetadataRdfType* type(MetadataQName qName);
	void insertType(MetadataRdfType* type);

	MetadataRdfType* property(MetadataQName qName);
	void insertProperty(MetadataQName qName, MetadataRdfType* type);
	void insertProperty(MetadataQName qName, MetadataQName typeQName);
	bool containsProperty(MetadataQName qName);
	void removeProperty(MetadataQName qName);

private:
	static MetadataRdfSchema* _instance;

	MetadataRdfStringType* _stringType;
	MetadataRdfBinaryType* _binaryType;
	MetadataRdfIntegerType* _integerType;
	MetadataRdfDoubleType* _doubleType;
	MetadataRdfRationalType* _rationalType;
	MetadataRdfPointType* _pointType;
	MetadataRdfSize2dType* _size2dType;
	QMap<MetadataQName, MetadataRdfType*> _types;
	QMap<MetadataQName, MetadataRdfType*> _properties;

	MetadataRdfSchema();
};

class MetadataRdfDefinitionReader {
public:
	static void readFolder(QString folder);

	MetadataRdfDefinitionReader(QIODevice* device);

	bool read();

private:
	QIODevice* _device;
	QMap<QString, QString> _typeNamespaceMap;
	QMap<QString, QString> _typeLocalNameMap;
};

class MetadataRdfReader : public MetadataReader {
public:
	MetadataRdfReader(QIODevice* device);
	virtual ~MetadataRdfReader();

	virtual MetadataNode* read(MetadataNode* parent=0);

	static MetadataNode* readFile(QString fileName);
	static MetadataNode* readFolder(QString folder, MetadataNode* parent=0);

protected:
	QXmlStreamReader* _reader;
};

class MetadataRdfWriter : public MetadataWriter {
public:
	MetadataRdfWriter(QIODevice* device);
	virtual ~MetadataRdfWriter();

	virtual void write(MetadataNode* node);

	void setPrefix(QString namespaceUri, QString prefix);

	void setWriteUnknownProperties(bool flag=true);

	static void writeFile(MetadataNode* node, QString filename);

protected:
	QXmlStreamWriter* _writer;
	QMap<QString, QString> _prefix;
	bool _writeUnkownProperties;

	void write(QList<MetadataNode*> nodes);
};

#if QT_VERSION >= 0x040400

class MetadataRdfXQueryTransform : public MetadataTransform {
public:
	virtual MetadataNode* transform(MetadataNode* source);
};

#endif

#endif /*METADATARDF_H_*/
