/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "RawFile.h"

#include <QDebug>


#include <ColorManagement.h>
#include <MetadataRdf.h>
#include <QtUtils.h>
#include <formats/CanonCr2.h>
#include <formats/DNG.h>

#ifdef USE_LIBRAW
#include <formats/LibRawFile.h>
#endif

// TODO: input profile code

QString RawFile::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/raw/1.0/";
MetadataQName RawFile::propertyAnalogBalance = MetadataQName(RawFile::namespaceUri, "AnalogBalance");
MetadataQName RawFile::propertyBlackLevel = MetadataQName(RawFile::namespaceUri, "BlackLevel");
MetadataQName RawFile::propertyBodyId = MetadataQName(RawFile::namespaceUri, "BodyId");
MetadataQName RawFile::propertyCalibrationIlluminant1 = MetadataQName(RawFile::namespaceUri, "CalibrationIlluminant1");
MetadataQName RawFile::propertyCalibrationIlluminant2 = MetadataQName(RawFile::namespaceUri, "CalibrationIlluminant2");
MetadataQName RawFile::propertyCameraCalibration1 = MetadataQName(RawFile::namespaceUri, "CameraCalibration1");
MetadataQName RawFile::propertyCameraCalibration2 = MetadataQName(RawFile::namespaceUri, "CameraCalibration2");
MetadataQName RawFile::propertyCfaPattern = MetadataQName(RawFile::namespaceUri, "CfaPattern");
MetadataQName RawFile::propertyCfaPatternDim = MetadataQName(RawFile::namespaceUri, "CfaPatternDim");
MetadataQName RawFile::propertyColorMatrix1 = MetadataQName(RawFile::namespaceUri, "ColorMatrix1");
MetadataQName RawFile::propertyColorMatrix2 = MetadataQName(RawFile::namespaceUri, "ColorMatrix2");
MetadataQName RawFile::propertyDefaultSize = MetadataQName(RawFile::namespaceUri, "DefaultSize");
MetadataQName RawFile::propertyMaskedAreas = MetadataQName(RawFile::namespaceUri, "MaskedAreas");
MetadataQName RawFile::propertyRawResourceMarker = MetadataQName(RawFile::namespaceUri, "RawResourceMarker");
MetadataQName RawFile::propertySensorCrop = MetadataQName(RawFile::namespaceUri, "SensorCrop");
MetadataQName RawFile::propertyWhiteLevel = MetadataQName(RawFile::namespaceUri, "WhiteLevel");
MetadataQName RawFile::propertyWhiteBalanceNeutral = MetadataQName(RawFile::namespaceUri, "WhiteBalanceNeutral");
MetadataDocument* RawFile::_cameraDefinitions = 0;

RawFile::~RawFile() {
	if(_metadata != 0)
		delete _metadata;

	if(_image != 0)
		delete _image;
}

MetadataDocument* RawFile::metadata() {
	return _metadata;
}

Image* RawFile::image() {
	return _image;
}

RawFile* RawFile::fromFile(QFile* file) {
	QString fileName = file->fileName();
	RawFile* rawFile = 0;

	if(fileName.endsWith(".cr2", Qt::CaseInsensitive))
		rawFile = new CanonCr2File(file);
	else if(fileName.endsWith(".dng", Qt::CaseInsensitive))
		rawFile = new DNGFile(file);
#ifdef USE_LIBRAW
	else
		rawFile = new LibRawFile(file);
#endif

	return rawFile;
}

bool RawFile::readCameraDefinitions(QString folder) {
	_cameraDefinitions = MetadataNode::toDocument(MetadataRdfReader::readFolder(folder));

	return _cameraDefinitions != 0;
}

MetadataDocument* RawFile::cameraDefinitions() {
	return _cameraDefinitions;
}

RawFile::RawFile() {
	_metadata = 0;
	_image = 0;
}

MetadataResource* RawUtils::findCameraDefinition(QString model) {
	MetadataDocument* cameraDefinitions = RawFile::cameraDefinitions();

	if(cameraDefinitions == 0)
		return 0;

	MetadataProperty* modelProperty = MetadataQuery::property(cameraDefinitions, Tiff::propertyModel, QVariant(model));

	if(modelProperty == 0)
		return 0;

	return modelProperty->parent()->toResource();
}

void RawUtils::appendCameraDefinition(MetadataResource* resource, QString model) {
	MetadataResource* cameraDefinition = findCameraDefinition(model);

	if(cameraDefinition == 0)
		return;

	foreach(MetadataNode* child, cameraDefinition->children()) {
		if(child->isProperty()) {
			MetadataProperty* property = child->toProperty();

			if(Tiff::propertyMake == property->qName())
				continue;

			if(Tiff::propertyModel == property->qName())
				continue;
		}

		child->clone(resource);
	}
}

Matrix3x3 RawUtils::cameraMatrix(MetadataResource* metadataResource, double temperature) {
	return analogBalance(metadataResource) * cameraCalibration(metadataResource, temperature) * colorMatrix(metadataResource, temperature);
}

Colorxy RawUtils::cameraNeutralWhiteBalance(MetadataResource* metadataResource) {
	return cameraNeutralWhiteBalance(metadataResource, ColorXYZ(MetadataQuery::typedValues<double>(metadataResource, RawFile::propertyWhiteBalanceNeutral)));
}

Colorxy RawUtils::cameraNeutralWhiteBalance(MetadataResource* metadataResource, ColorXYZ neutralWhiteBalance) {
	double temperature;
	double tint;
	double d;

	Colorxy lastTest = Colorxy::whitePointD50();

	do {
		TemperatureUtils::fromxy(lastTest, &temperature, &tint);
		Colorxy test = (neutralWhiteBalance * cameraMatrix(metadataResource, temperature).inverse()).toColorxy();

		d = qAbs(lastTest.x()-test.x()) + qAbs(lastTest.y()-test.y());

		lastTest = test;
	} while(d > 0.0000001);

	return lastTest;
}

void RawUtils::appendBlackLevelFromMaskedArea(Image* image) {
	BayerImage16* bayerImage16;

	QList<int> maskedArea = MetadataQuery::typedValues<int>(image->metadata(), RawFile::propertyMaskedAreas);

	if(maskedArea.isEmpty())
		return;

	int left = maskedArea.at(0);
	int top = maskedArea.at(1);
	int width = maskedArea.at(2);
	int height = maskedArea.at(3);

	if((bayerImage16 = qobject_cast<BayerImage16*>(image)) != 0) {
		double blackLevel = 0.0;

		for(int y=top; y<height; y++) {
			quint16* data = (quint16*)bayerImage16->dataStripe(y);

			for(int x=left; x<width; x++)
				blackLevel += (double)data[x];
		}

		new MetadataProperty(image->metadata(), RawFile::propertyBlackLevel, (int)blackLevel / ((width-left)*(height-top)));
	}
}

void RawUtils::appendOrientationFromTiff(MetadataResource* metadataResource) {
	QVariant orientationValue = MetadataQuery::value(metadataResource, Tiff::propertyOrientation);

	if(orientationValue.isNull())
		return;

	switch(orientationValue.toInt()) {
		case 1:
			new MetadataProperty(metadataResource, Image::propertyOrientation, 0.0);
			break;

		case 3:
			new MetadataProperty(metadataResource, Image::propertyOrientation, 180.0);
			break;

		case 6:
			new MetadataProperty(metadataResource, Image::propertyOrientation, -90.0);
			break;

		case 8:
			new MetadataProperty(metadataResource, Image::propertyOrientation, 90.0);
			break;

		default:
			break;
	}
}

MetadataProperty* RawUtils::appendSettingsDefault(MetadataResource* metadataResource) {
	MetadataProperty* settingsDefault = new MetadataProperty(metadataResource, ImageFilterSettings::propertySettings);
	new MetadataProperty(settingsDefault, ImageFilterSettings::propertyName, "default");

	return settingsDefault;
}

MetadataProperty* RawUtils::createCurrentSettings(MetadataResource* metadataResource) {
	MetadataNode* settingsDefault = MetadataQuery::property(metadataResource, ImageFilterSettings::propertyName, QVariant("default"))->parent();

	if(settingsDefault == 0)
		return 0;

	MetadataProperty* settingsCurrent = settingsDefault->clone(metadataResource)->toProperty();
	MetadataQuery::property(settingsCurrent, ImageFilterSettings::propertyName)->toProperty()->setValue("current");

	return settingsCurrent;
}

double RawUtils::illuminant1(MetadataResource* metadataResource) {
	QVariant illuminant1Value = MetadataQuery::value(metadataResource, RawFile::propertyCalibrationIlluminant1);

	if(!illuminant1Value.isNull())
		return illuminant1Value.toDouble();
	else
		return 0.0;
}

double RawUtils::illuminant2(MetadataResource* metadataResource) {
	QVariant illuminant2Value = MetadataQuery::value(metadataResource, RawFile::propertyCalibrationIlluminant2);

	if(!illuminant2Value.isNull())
		return illuminant2Value.toDouble();
	else
		return 0.0;
}

Matrix3x3 RawUtils::colorMatrix(MetadataResource* metadataResource, double temperature) {
	QList<double> colorMatrix1Values = MetadataQuery::typedValues<double>(metadataResource, RawFile::propertyColorMatrix1);
	QList<double> colorMatrix2Values = MetadataQuery::typedValues<double>(metadataResource, RawFile::propertyColorMatrix2);

	double illuminant1Value = illuminant1(metadataResource);
	double illuminant2Value = illuminant2(metadataResource);

	Matrix3x3 colorMatrix1Matrix;
	Matrix3x3 colorMatrix2Matrix;

	if(!colorMatrix1Values.isEmpty())
		colorMatrix1Matrix = Matrix3x3(colorMatrix1Values);

	if(!colorMatrix2Values.isEmpty())
		colorMatrix2Matrix = Matrix3x3(colorMatrix2Values);

	if(illuminant1Value != 0.0 && illuminant2Value != 0.0) {
		double f = ((1.0 / temperature) - (1.0 / illuminant2Value)) / ((1.0 / illuminant1Value) - (1.0 / illuminant2Value));

		return interpolateMatrix(colorMatrix1Matrix, colorMatrix2Matrix, f);
	} else {
		return colorMatrix1Matrix;
	}
}

Matrix3x3 RawUtils::cameraCalibration(MetadataResource* metadataResource, double temperature) {
	QList<double> cameraCalibration1Values = MetadataQuery::typedValues<double>(metadataResource, RawFile::propertyCameraCalibration1);
	QList<double> cameraCalibration2Values = MetadataQuery::typedValues<double>(metadataResource, RawFile::propertyCameraCalibration2);

	double illuminant1Value = illuminant1(metadataResource);
	double illuminant2Value = illuminant2(metadataResource);

	Matrix3x3 cameraCalibration1Matrix;
	Matrix3x3 cameraCalibration2Matrix;

	if(!cameraCalibration1Values.isEmpty())
		cameraCalibration1Matrix = Matrix3x3(cameraCalibration1Values);

	if(!cameraCalibration2Values.isEmpty())
		cameraCalibration2Matrix = Matrix3x3(cameraCalibration2Values);

	if(illuminant1Value != 0.0 && illuminant2Value != 0.0) {
		double f = ((1.0 / temperature) - (1.0 / illuminant2Value)) / ((1.0 / illuminant1Value) - (1.0 / illuminant2Value));

		return interpolateMatrix(cameraCalibration1Matrix, cameraCalibration2Matrix, f);
	} else {
		return cameraCalibration1Matrix;
	}
}

Matrix3x3 RawUtils::analogBalance(MetadataResource* metadataResource) {
	QList<double> analogBalanceValues = MetadataQuery::typedValues<double>(metadataResource, RawFile::propertyAnalogBalance);

	return Matrix3x3(analogBalanceValues);
}

Matrix3x3 RawUtils::interpolateMatrix(Matrix3x3 a, Matrix3x3 b, double f) {
	if(f <= 0.0)
		return a;
	else if(f >= 1.0)
		return b;
	else
		return a*Matrix3x3(f,f,f) + b*Matrix3x3(1.0-f,1.0-f,1.0-f);
}

QString RawExposure::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/raw-exposure/1.0/";
MetadataQName RawExposure::propertyWhiteBalance = MetadataQName(RawExposure::namespaceUri, "WhiteBalance");
MetadataQName RawExposure::propertyTemperature = MetadataQName(RawExposure::namespaceUri, "Temperature");
MetadataQName RawExposure::propertyTint = MetadataQName(RawExposure::namespaceUri, "Tint");
MetadataQName RawExposure::propertyToneCurve = MetadataQName(RawExposure::namespaceUri, "ToneCurve");

BayerImage16::BayerImage16(int width, int height, QObject* parent) : Image(parent) {
	_type = BayerImage16::type();
	_pixelSize = 2;

	_width = width;
	_height = height;
	_stripeSize = _width;

	_dataAlloc = new quint8[_stripeSize*_pixelSize*_height];
	_data = _dataAlloc;
}

BayerImage16::~BayerImage16() {
}

int BayerImage16::precision() {
	return _precision;
}

void BayerImage16::setPrecision(int precision) {
	_precision = precision;
}
