/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef RAWFILE_H_
#define RAWFILE_H_

#include <QFile>
#include <QList>
#include <QString>
#include <QVector>

#include <ColorMath.h>
#include <Image.h>
#include <ImageFilter.h>
#include <Metadata.h>

class RawFile {
public:
	static QString namespaceUri;

	static MetadataQName propertyAnalogBalance;
	static MetadataQName propertyBlackLevel;
	static MetadataQName propertyBodyId;
	static MetadataQName propertyCalibrationIlluminant1;
	static MetadataQName propertyCalibrationIlluminant2;
	static MetadataQName propertyCameraCalibration1;
	static MetadataQName propertyCameraCalibration2;
	static MetadataQName propertyCfaPattern;
	static MetadataQName propertyCfaPatternDim;
	static MetadataQName propertyColorMatrix1;
	static MetadataQName propertyColorMatrix2;
	static MetadataQName propertyDefaultSize;
	static MetadataQName propertyMaskedAreas; // TODO: can be different for sraw!!!
	static MetadataQName propertyRawResourceMarker;
	static MetadataQName propertySensorCrop;
	static MetadataQName propertyWhiteBalanceNeutral;
	static MetadataQName propertyWhiteLevel;

	virtual ~RawFile();

	virtual MetadataDocument* readMetadata() = 0;
	virtual Image* readImage() = 0;
	virtual int thumbnailCount() = 0;
	virtual QImage thumnail(int n) = 0;

	MetadataDocument* metadata();
	Image* image();

	static RawFile* fromFile(QFile* file);

	static bool readCameraDefinitions(QString folder);
	static MetadataDocument* cameraDefinitions();

protected:
	RawFile();

	MetadataDocument* _metadata;
	Image* _image;

private:
	static MetadataDocument* _cameraDefinitions;
};

class RawUtils {
public:
	static MetadataResource* findCameraDefinition(QString model);
	static void appendCameraDefinition(MetadataResource* resource, QString model);

	static Matrix3x3 cameraMatrix(MetadataResource* metadataResource, double temperature);
	static Matrix3x3 cameraCalibration(MetadataResource* metadataResource, double temperature);
	static Matrix3x3 colorMatrix(MetadataResource* metadataResource, double temperature);
	static Matrix3x3 analogBalance(MetadataResource* metadataResource);
	static Colorxy cameraNeutralWhiteBalance(MetadataResource* metadataResource);
	static Colorxy cameraNeutralWhiteBalance(MetadataResource* metadataResource, ColorXYZ neutralWhiteBalance);

	static void appendBlackLevelFromMaskedArea(Image* image);

	static void appendOrientationFromTiff(MetadataResource* metadataResource);

	static MetadataProperty* appendSettingsDefault(MetadataResource* metadataResource);
	static MetadataProperty* createCurrentSettings(MetadataResource* metadataResource);

private:
	static double illuminant1(MetadataResource* metadataResource);
	static double illuminant2(MetadataResource* metadataResource);
	static Matrix3x3 interpolateMatrix(Matrix3x3 a, Matrix3x3 b, double f);
};

class RawExposure {
public:
	static QString namespaceUri;

	static MetadataQName propertyWhiteBalance;
	static MetadataQName propertyTemperature;
	static MetadataQName propertyTint;
	static MetadataQName propertyToneCurve;
};

class BayerImage16 : public Image {

	Q_OBJECT

public:
	BayerImage16(int width, int height, QObject* parent=0);
	virtual ~BayerImage16();

	int precision();
	void setPrecision(int precision);

	static QString type() {
		return "Bayer16";
	}

private:
	int _precision;
};

#endif /*RAWFILE_H_*/
