/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "RawImporter.h"

#include <QApplication>
#include <QColor>
#include <QFileDialog>
#include <QTextStream>
#include <QTime>

#include <math.h>

#include <ColorManagement.h>
#include <Dialogs.h>
#include <MetadataRdf.h>
#include <QtUtils.h>
#include <filters/ResizeFilter.h>
#include <filters/RotateCropFilter.h>
#include <formats/CanonCr2.h>
#include <formats/Tiff.h>

#include <ui_rawdev.h>

RawdevCore::RawdevCore() {
	_rawFile = 0;
	_metadataResource = 0;

	MetadataRdfDefinitionReader::readFolder("./rdfdefinitions/");
	RawFile::readCameraDefinitions("./camdefinitions/");

	_previewFilterChain = new ImageFilterChain(this);

	_previewFilterChain->appendFilter(new DemosaicFilter(), ImageFormat(RgbImage48::type()));
	_previewFilterChain->appendFilter(new ResizeFilter(), ImageFormat(RgbImage48::type()));
	_previewFilterChain->appendFilter(new ExposureFilter(), ImageFormat(RgbImage48::type()));
	_previewFilterChain->appendFilter(new DevelopFilter(), ImageFormat(RgbImage32::type()));

	connect(this, SIGNAL(metadataPropertyChanged(MetadataProperty*)), _previewFilterChain, SLOT(metadataPropertyChange(MetadataProperty*)));
}

RawdevCore::~RawdevCore() {
	if(_rawFile != 0)
		delete _rawFile;
}

bool RawdevCore::fileOpen(QString fileName) {
	fileClose();

	QFile inputFile(fileName);

	if(!inputFile.open(QIODevice::ReadOnly))
		return false;

	if((_rawFile = RawFile::fromFile(&inputFile)) == 0)
		return false;

	Image* bayerImage = _rawFile->readImage();
	inputFile.close();

	_metadataResource = MetadataQuery::resource(_rawFile->readMetadata(), "*");

	_settingsDefault = MetadataQuery::property(_metadataResource, ImageFilterSettings::propertyName, QVariant("default"))->parent();
	_settingsCurrent = MetadataQuery::property(_metadataResource, ImageFilterSettings::propertyName, QVariant("current"))->parent();

	_previewFilterChain->setImage(bayerImage);
	_previewFilterChain->setParameter(DevelopFilter::parameterOutputProfile, DevelopFilter::profileSRgb);

	return true;
}

bool RawdevCore::fileClose() {
	_previewFilterChain->setImage(0);

	if(_rawFile != 0) {
		delete _rawFile;

		_rawFile = 0;
	}

	return true;
}

void RawdevCore::saveImage(QString fileName) {
	if(_rawFile == 0)
		return;

	QString outputProfile = DevelopFilter::profileSRgb;
	if(_developParameters.contains(DevelopFilter::parameterOutputProfile))
		outputProfile = _developParameters.value(DevelopFilter::parameterOutputProfile).toString();

	QString imageWriterName = QtImageWriter::name;
	if(_imageWriterParamters.contains(ImageWriter::parameterImageWriter))
		imageWriterName = _imageWriterParamters.value(ImageWriter::parameterImageWriter).toString();

	ImageFilterChain filterChain;

	filterChain.setImage(_rawFile->image());

	filterChain.appendFilter(new DemosaicFilter(), ImageFormat(RgbImage48::type()));
	filterChain.appendFilter(new RotateCropFilter(), ImageFormat(RgbImage48::type()));
	filterChain.appendFilter(new ExposureFilter(), ImageFormat(RgbImage48::type()));

	if(imageWriterName == TiffWriter::name)
		filterChain.appendFilter(new DevelopFilter(), ImageFormat(RgbImage48::type()));
	else
		filterChain.appendFilter(new DevelopFilter(), ImageFormat(RgbImage32::type()));

	filterChain.setParameter("interpolation", "linear");
	filterChain.setParameters(_developParameters);

	filterChain.clearCache();
	Image* target = filterChain.filter();

	ImageWriter* imageWriter = 0;

	if(imageWriterName == TiffWriter::name)
		imageWriter = new TiffWriter();
	else
		imageWriter = new QtImageWriter();

	MetadataTransform* metadataTransform = new TiffSimpleMetadataTransform();
	MetadataResource* resource = MetadataNode::toResource(metadataTransform->transform(target->metadata()));

	delete metadataTransform;

	imageWriter->write(fileName, target, resource, _imageWriterParamters);
}

void RawdevCore::saveBayerImage(QString fileName) {
	if(_rawFile == 0)
		return;

	ImageWriter* imageWriter = new TiffWriter();

	imageWriter->write(fileName, _rawFile->image(), (MetadataResource*)0, _imageWriterParamters);

	delete imageWriter;
}

void RawdevCore::saveMetadata(QString fileName) {
	if(_rawFile == 0)
		return;

	QFile file(fileName);
	MetadataRdfWriter rdfWriter(&file);
	rdfWriter.setPrefix(Tiff::namespaceUri, "tiff");
	rdfWriter.setPrefix(Exif::namespaceUri, "exif");
	rdfWriter.setPrefix(RawFile::namespaceUri, "raw");
	rdfWriter.setPrefix(ImageFilterSettings::namespaceUri, "if");
	rdfWriter.setPrefix(RotateCropFilter::namespaceUri, "ifrc");
	rdfWriter.setPrefix(RawExposure::namespaceUri, "rawexposure");
	rdfWriter.setPrefix(CanonCr2::namespaceUri, "canoncr2");
	rdfWriter.setWriteUnknownProperties(true);
	file.open(QIODevice::WriteOnly);
	rdfWriter.write(_rawFile->metadata());
	file.close();
}

bool RawdevCore::imageLoaded() {
	return _rawFile != 0;
}

int RawdevCore::sourceWidth() {
	if(_rawFile == 0)
		return -1;

	return _rawFile->image()->width();
}

int RawdevCore::sourceHeight() {
	if(_rawFile == 0)
		return -1;

	return _rawFile->image()->height();
}

double RawdevCore::rotateAngle() {
	return MetadataQuery::typedValue<double>(_settingsCurrent, RotateCropFilter::propertyRotateAngle);
}

void RawdevCore::setRotateAngle(double angle) {
	setSimplePropertySettingsCurrent(RotateCropFilter::propertyRotateAngle, angle);
}

int RawdevCore::cropLeft() {
	return MetadataQuery::typedValue<int>(_settingsCurrent, RotateCropFilter::propertyCropLeft);
}

void RawdevCore::setCropLeft(int cropLeft) {
	setSimplePropertySettingsCurrent(RotateCropFilter::propertyCropLeft, cropLeft);
}

int RawdevCore::cropRight() {
	return MetadataQuery::typedValue<int>(_settingsCurrent, RotateCropFilter::propertyCropRight);
}

void RawdevCore::setCropRight(int cropRight) {
	setSimplePropertySettingsCurrent(RotateCropFilter::propertyCropRight, cropRight);
}

int RawdevCore::cropTop() {
	return MetadataQuery::typedValue<int>(_settingsCurrent, RotateCropFilter::propertyCropTop);
}

void RawdevCore::setCropTop(int cropTop) {
	setSimplePropertySettingsCurrent(RotateCropFilter::propertyCropTop, cropTop);
}

int RawdevCore::cropBottom() {
	return MetadataQuery::typedValue<int>(_settingsCurrent, RotateCropFilter::propertyCropBottom);
}

void RawdevCore::setCropBottom(int cropBottom) {
	setSimplePropertySettingsCurrent(RotateCropFilter::propertyCropBottom, cropBottom);
}

QString RawdevCore::whiteBalance() {
	return MetadataQuery::typedValue<QString>(_settingsCurrent, RawExposure::propertyWhiteBalance);
}

void RawdevCore::setWhiteBalance(QString whiteBalance) {
	if(whiteBalance.isEmpty())
		whiteBalance = "asShot";
	else if(whiteBalance.compare("asShot", Qt::CaseInsensitive) == 0)
		whiteBalance = "asShot";
	else if(whiteBalance.compare("manual", Qt::CaseInsensitive) == 0)
		whiteBalance = "manual";

	setSimplePropertySettingsCurrent(RawExposure::propertyWhiteBalance, whiteBalance);
}

int RawdevCore::whiteBalanceTemperature() {
	return MetadataQuery::typedValue<int>(_settingsCurrent, RawExposure::propertyTemperature);
}

void RawdevCore::setWhiteBalanceTemperature(int temperature) {
	setSimplePropertySettingsCurrent(RawExposure::propertyTemperature, temperature);
}

int RawdevCore::whiteBalanceTint() {
	return MetadataQuery::typedValue<int>(_settingsCurrent, RawExposure::propertyTint);
}

void RawdevCore::setWhiteBalanceTint(int tint) {
	setSimplePropertySettingsCurrent(RawExposure::propertyTint, tint);
}

QList<QPointF> RawdevCore::toneCurve() {
	if(_rawFile == 0)
		return QList<QPointF>();

	MetadataProperty* toneCurveProperty = MetadataQuery::property(_settingsCurrent, RawExposure::propertyToneCurve);

	if(toneCurveProperty == 0)
		return QList<QPointF>();

	return QtUtils::toTypedList<QPointF>(toneCurveProperty->values());
}

void RawdevCore::setToneCurve(Spline curve) {
	if(_metadataResource == 0)
		return;

	MetadataProperty* toneCurveProperty = MetadataQuery::property(_settingsCurrent, RawExposure::propertyToneCurve);

	if(toneCurveProperty == 0)
		toneCurveProperty = new MetadataProperty(_settingsCurrent, RawExposure::propertyToneCurve);

	toneCurveProperty->setValues(QtUtils::toVariantList(curve.points()));

	emit metadataPropertyChanged(toneCurveProperty);
}

BergPhoto::Parameters RawdevCore::developParameters() {
	return _developParameters;
}

void RawdevCore::setDevelopParameters(BergPhoto::Parameters parameters) {
	_developParameters = parameters;
}

BergPhoto::Parameters RawdevCore::imageWriterParameters() {
	return _imageWriterParamters;
}

void RawdevCore::setImageWriterParameters(BergPhoto::Parameters parameters) {
	_imageWriterParamters = parameters;
}

ImageFilterChain* RawdevCore::previewFilterChain() {
	return _previewFilterChain;
}

void RawdevCore::setSimplePropertySettingsCurrent(MetadataQName qName, QVariant value) {
	if(_metadataResource == 0)
		return;

	MetadataProperty* property = MetadataQuery::property(_settingsCurrent, qName);

	if(property == 0)
		property = new MetadataProperty(_settingsCurrent, qName);

	property->setValue(value);

	emit metadataPropertyChanged(property);
}

RawdevGui::RawdevGui(QMainWindow* mainWindow, RawdevCore* rawdevCore) {
	_rawdevCore = rawdevCore;

	_mainWindow = mainWindow;
	_mainWindowRawdev.setupUi(_mainWindow);

	_mainWindowRawdev.widgetSimpleImagePreview->setFilterChain(_rawdevCore->previewFilterChain());

	connect(_mainWindowRawdev.actionFileOpen, SIGNAL(triggered()), this, SLOT(fileOpen()));
	connect(_mainWindowRawdev.actionExit, SIGNAL(triggered()), QCoreApplication::instance(), SLOT(quit()));
	connect(_mainWindowRawdev.actionSaveImage, SIGNAL(triggered()), this, SLOT(saveImage()));
	connect(_mainWindowRawdev.actionSetOutputParameters, SIGNAL(triggered()), this, SLOT(setOutputParameters()));
	connect(_mainWindowRawdev.actionSaveBayerImage, SIGNAL(triggered()), this, SLOT(saveBayerImage()));
	connect(_mainWindowRawdev.actionSaveMetadata, SIGNAL(triggered()), this, SLOT(saveMetadata()));

	connect(_mainWindowRawdev.comboBoxWhiteBalanceMode, SIGNAL(currentIndexChanged(QString)), this, SLOT(setWhiteBalanceMode(QString)));
	connect(_mainWindowRawdev.spinBoxTemperature, SIGNAL(valueChanged(int)), this, SLOT(setWhiteBalanceTemperature(int)));
	connect(_mainWindowRawdev.spinBoxTint, SIGNAL(valueChanged(int)), this, SLOT(setWhiteBalanceTint(int)));

	connect(_mainWindowRawdev.widgetToneCurve, SIGNAL(curveChanged(Spline)), this, SLOT(setToneCurve(Spline)));

	syncGui();
}

RawdevGui::~RawdevGui() {
}

void RawdevGui::fileOpen() {
	QString fileName = QFileDialog::getOpenFileName(_mainWindow, "Open raw file", QString(), "Raw files (*.cr2 *.dng)");

	if(fileName.isEmpty())
		return;

	_rawdevCore->fileOpen(fileName);

	syncGui();
}

void RawdevGui::fileClose() {
	_rawdevCore->fileClose();

	syncGui();
}

void RawdevGui::saveImage() {
	if(!_rawdevCore->imageLoaded())
		return;

	QString fileFormat = "jpg";

	if(_rawdevCore->imageWriterParameters().contains(ImageWriter::parameterFileFormat))
		fileFormat = _rawdevCore->imageWriterParameters().value(ImageWriter::parameterFileFormat).toString();

	QString fileName = QFileDialog::getSaveFileName(_mainWindow, "Save image", QString(), QString("Image file (*.%1)").arg(fileFormat));

	if(fileName.isEmpty())
		return;

	_rawdevCore->saveImage(fileName);
}

void RawdevGui::setOutputParameters() {
	OutputParametersDialog dialog(_mainWindow);

	dialog.setDevelopParameters(_rawdevCore->developParameters());
	dialog.setImageWriterParameters(_rawdevCore->imageWriterParameters());

	if(dialog.exec() == QDialog::Accepted) {
		_rawdevCore->setDevelopParameters(dialog.developParameters());
		_rawdevCore->setImageWriterParameters(dialog.imageWriterParameters());
	}
}

void RawdevGui::saveBayerImage() {
	if(!_rawdevCore->imageLoaded())
		return;

	QString fileName = QFileDialog::getSaveFileName(_mainWindow, "Save bayer image", QString(), "Tiff files (*.tiff)");

	if(fileName.isEmpty())
		return;

	_rawdevCore->saveBayerImage(fileName);
}

void RawdevGui::saveMetadata() {
	if(!_rawdevCore->imageLoaded())
		return;

	QString fileName = QFileDialog::getSaveFileName(_mainWindow, "Save metadata", QString(), "XML file (*.xml)");

	if(fileName.isEmpty())
			return;

	_rawdevCore->saveMetadata(fileName);
}

void RawdevGui::setRotateAngle(double angle) {
	_rawdevCore->setRotateAngle(angle);
}

void RawdevGui::setCropLeft(int cropLeft) {
	_rawdevCore->setCropLeft(cropLeft);
}

void RawdevGui::setCropRight(int cropRight) {
	_rawdevCore->setCropRight(cropRight);
}

void RawdevGui::setCropTop(int cropTop) {
	_rawdevCore->setCropTop(cropTop);
}

void RawdevGui::setCropBottom(int cropBottom) {
	_rawdevCore->setCropBottom(cropBottom);
}

void RawdevGui::setWhiteBalanceMode(QString mode) {
	_rawdevCore->setWhiteBalance(mode);
}

void RawdevGui::setWhiteBalanceTemperature(int temperature) {
	_rawdevCore->setWhiteBalanceTemperature(temperature);
}

void RawdevGui::setWhiteBalanceTint(int tint) {
	_rawdevCore->setWhiteBalanceTint(tint);
}

void RawdevGui::setToneCurve(Spline curve) {
	_rawdevCore->setToneCurve(curve);
}

void RawdevGui::syncGui() {
	if(!_rawdevCore->imageLoaded()) {
		_mainWindowRawdev.pushButtonRealtimePreview->setChecked(false);

		_mainWindowRawdev.comboBoxWhiteBalanceMode->setEnabled(false);
		_mainWindowRawdev.horizontalSliderTemperature->setEnabled(false);
		_mainWindowRawdev.spinBoxTemperature->setEnabled(false);
		_mainWindowRawdev.horizontalSliderTint->setEnabled(false);
		_mainWindowRawdev.spinBoxTint->setEnabled(false);

		_mainWindowRawdev.widgetToneCurve->setEnabled(false);

		_mainWindowRawdev.pushButtonRealtimePreview->setEnabled(false);
		_mainWindowRawdev.pushButtonPreviewImage->setEnabled(false);
		_mainWindowRawdev.pushButtonSaveBayerImage->setEnabled(false);
		_mainWindowRawdev.pushButtonSaveImage->setEnabled(false);
		_mainWindowRawdev.pushButtonSaveMetadata->setEnabled(false);
		_mainWindowRawdev.pushButtonSetOutputParameters->setEnabled(false);
	} else {
		_mainWindowRawdev.comboBoxWhiteBalanceMode->setEnabled(true);
		_mainWindowRawdev.horizontalSliderTemperature->setEnabled(true);
		_mainWindowRawdev.spinBoxTemperature->setEnabled(true);
		_mainWindowRawdev.horizontalSliderTint->setEnabled(true);
		_mainWindowRawdev.spinBoxTint->setEnabled(true);

		_mainWindowRawdev.widgetToneCurve->setEnabled(true);

		_mainWindowRawdev.pushButtonRealtimePreview->setEnabled(true);
		_mainWindowRawdev.pushButtonPreviewImage->setEnabled(true);
		_mainWindowRawdev.pushButtonSaveBayerImage->setEnabled(true);
		_mainWindowRawdev.pushButtonSaveImage->setEnabled(true);
		_mainWindowRawdev.pushButtonSaveMetadata->setEnabled(true);
		_mainWindowRawdev.pushButtonSetOutputParameters->setEnabled(true);

		_mainWindowRawdev.comboBoxWhiteBalanceMode->setCurrentIndex(0);
		_mainWindowRawdev.spinBoxTemperature->setValue(_rawdevCore->whiteBalanceTemperature());
		_mainWindowRawdev.spinBoxTint->setValue(_rawdevCore->whiteBalanceTint());

		_mainWindowRawdev.widgetToneCurve->setPoints(_rawdevCore->toneCurve());
	}
}

#if 0

int main(int argc, char** argv) {
	QApplication application(argc, argv);

	MetadataRdfDefinitionReader::readFolder("./rdfdefinitions/");

	MetadataNode* node = MetadataRdfReader::readFile("test.xml");
	MetadataRdfWriter::writeFile(node, "test.out.xml");

	return 0;
}

#endif

#if 0

#include <DNG.h>
#include <MetadataExifTool.h>

int main(int argc, char** argv) {
	QApplication application(argc, argv);

	MetadataRdfDefinitionReader::readFolder("./rdfdefinitions/");

	//MetadataNode* dngMetadata = MetadataExifTool::readFile("F:\\projects\\bergphoto\\data\\Canon 400D\\IMG_8801.dng");
	MetadataNode* dngMetadata = MetadataExifTool::readFile("F:\\projects\\bergphoto\\data\\Canon EOS 5D Mark II\\IMG_0234.dng");

	MetadataRdfWriter::writeFile(dngMetadata, "dng.rdf.xml");

	/*MetadataDocument* camDefinition = new MetadataDocument();
	MetadataResource* resource = new MetadataResource(camDefinition);

	MetadataProperty* dngMake = MetadataQuery::property(dngMetadata, Tiff::propertyMake);
	dngMake->clone(resource);

	MetadataProperty* dngModel = MetadataQuery::property(dngMetadata, Tiff::propertyModel);
	dngModel->clone(resource);

	int calibrationIlluminant1 = MetadataQuery::typedValue<int>(dngMetadata, DNG::propertyCalibrationIlluminant1);
	new MetadataSimpleProperty(resource, DNG::propertyCalibrationIlluminant1, Tiff::lightSourceTemperature(calibrationIlluminant1));

	int calibrationIlluminant2 = MetadataQuery::typedValue<int>(dngMetadata, DNG::propertyCalibrationIlluminant2);
	new MetadataSimpleProperty(resource, DNG::propertyCalibrationIlluminant2, Tiff::lightSourceTemperature(calibrationIlluminant2));

	MetadataProperty* colorMatrix1 = MetadataQuery::property(dngMetadata, DNG::propertyColorMatrix1);
	colorMatrix1->clone(resource);

	MetadataProperty* colorMatrix2 = MetadataQuery::property(dngMetadata, DNG::propertyColorMatrix2);
	colorMatrix2->clone(resource);

	MetadataProperty* maskedAreas = MetadataQuery::property(dngMetadata, DNG::propertyMaskedAreas);
	maskedAreas->clone(resource);

	QList<Rational> cameraCalibration1 = MetadataQuery::typedValues<Rational>(dngMetadata, DNG::propertyCameraCalibration1);
	QList<int> whiteBalanceDaylight = MetadataQuery::typedValues<int>(dngMetadata, MetadataQName(MetadataExifTool::namespaceUri, "WB_RGGBLevelsDaylight"));

	MetadataListProperty* whiteBalance = new MetadataListProperty(resource, CanonCr2::propertyDefaultWhiteBalanceNeutral5200K);
	new MetadataListItem(whiteBalance, (int)((double)whiteBalanceDaylight[0]*cameraCalibration1[0].toDouble()+0.5));
	new MetadataListItem(whiteBalance, (int)((double)((whiteBalanceDaylight[1]+whiteBalanceDaylight[2])/2)*cameraCalibration1[4].toDouble()+0.5));
	new MetadataListItem(whiteBalance, (int)((double)whiteBalanceDaylight[3]*cameraCalibration1[8].toDouble()+0.5));*/

	QFile file("example/ExifToolDNG2Camdef.xq");
	file.open(QIODevice::ReadOnly);
	QString queryString = QString(file.readAll());
	file.close();
	MetadataRdfXQueryTransform queryTransform;
	queryTransform.setParameter(QVariant(queryString));

	MetadataNode* camDefinition = queryTransform.transform(dngMetadata);

	MetadataRdfWriter::writeFile(camDefinition, "camdef.rdf.xml");

	delete dngMetadata;
	delete camDefinition;

	return 0;
}

#endif

#if 1

int main(int argc, char** argv) {
	QApplication application(argc, argv);

	QTextStream streamOut(stdout);
	QTextStream streamErr(stderr);

	QString inputFilename;
	QString outputFilename;
	QString rdfOutputFilename;
	QString bayerOutputFilename;
	QString whiteBalance;
	QString wbTemperature;
	QString wbTint;
	QString toneCurve;
	QString rotateAngle;
	QString cropLeft;
	QString cropRight;
	QString cropTop;
	QString cropBottom;
	QString developOutputProfile;
	QString imageWriter;
	bool imageWriterCompress = false;
	QString imageWriterQuality;
	bool gui = false;

	for(int i=1; i<argc; i++) {
		QString parameter = QString::fromAscii(argv[i]);

		if(parameter.startsWith("--input="))
			inputFilename = parameter.mid(8);

		if(parameter.startsWith("--output="))
			outputFilename = parameter.mid(9);

		if(parameter.startsWith("--rdfoutput="))
			rdfOutputFilename = parameter.mid(12);

		if(parameter.startsWith("--dumpbayer="))
			bayerOutputFilename = parameter.mid(12);

		if(parameter.startsWith("--wb="))
			whiteBalance = parameter.mid(5);

		if(parameter.startsWith("--wb-temperature="))
			wbTemperature = parameter.mid(17);

		if(parameter.startsWith("--wb-tint="))
			wbTint = parameter.mid(10);

		if(parameter.startsWith("--tonecurve="))
			toneCurve = parameter.mid(12);

		if(parameter.startsWith("--rotate-angle="))
			rotateAngle = parameter.mid(15);

		if(parameter.startsWith("--crop-left="))
			cropLeft = parameter.mid(12);

		if(parameter.startsWith("--crop-right="))
			cropRight = parameter.mid(13);

		if(parameter.startsWith("--crop-top="))
			cropTop = parameter.mid(11);

		if(parameter.startsWith("--crop-bottom="))
			cropBottom = parameter.mid(14);

		if(parameter.startsWith("--develop-outputprofile="))
			developOutputProfile = parameter.mid(24);

		if(parameter.startsWith("--imagewriter="))
			imageWriter = parameter.mid(14);

		if(parameter.startsWith("--imagewriter-compress="))
			imageWriterCompress = parameter.mid(23).compare("true", Qt::CaseInsensitive) == 0;

		if(parameter.startsWith("--imagewriter-quality="))
			imageWriterQuality = parameter.mid(22);

		if(parameter.startsWith("--gui="))
			gui = parameter.mid(6).compare("true", Qt::CaseInsensitive) == 0;
	}

	if(!gui && inputFilename.isEmpty()) {
		streamErr << "no input file defined\n";
		return -1;
	}

	RawdevCore rawdevCore;

	if(!inputFilename.isEmpty())
		rawdevCore.fileOpen(inputFilename);

	if(!whiteBalance.isEmpty())
		rawdevCore.setWhiteBalance(whiteBalance);

	if(!wbTemperature.isEmpty())
		rawdevCore.setWhiteBalanceTemperature(wbTemperature.toInt());

	if(!wbTint.isEmpty())
		rawdevCore.setWhiteBalanceTint(wbTint.toInt());

	if(!toneCurve.isEmpty())
		rawdevCore.setToneCurve(Spline(toneCurve));

	if(!rotateAngle.isEmpty())
		rawdevCore.setRotateAngle(rotateAngle.toDouble());

	if(!cropLeft.isEmpty())
		rawdevCore.setCropLeft(cropLeft.toInt());

	if(!cropRight.isEmpty())
		rawdevCore.setCropRight(cropRight.toInt());

	if(!cropTop.isEmpty())
		rawdevCore.setCropTop(cropTop.toInt());

	if(!cropBottom.isEmpty())
		rawdevCore.setCropBottom(cropBottom.toInt());

	if(!developOutputProfile.isEmpty()) {
		BergPhoto::Parameters parameters = rawdevCore.developParameters();
		parameters.insert(DevelopFilter::parameterOutputProfile, developOutputProfile);
		rawdevCore.setDevelopParameters(parameters);
	}

	if(!imageWriter.isEmpty()) {
		BergPhoto::Parameters parameters = rawdevCore.imageWriterParameters();
		parameters.insert(ImageWriter::parameterImageWriter, imageWriter);
		rawdevCore.setImageWriterParameters(parameters);
	}

	if(imageWriterCompress) {
		BergPhoto::Parameters parameters = rawdevCore.imageWriterParameters();
		parameters.insert(ImageWriter::parameterCompress, imageWriterCompress);
		rawdevCore.setImageWriterParameters(parameters);
	}

	if(!imageWriterQuality.isEmpty()) {
		BergPhoto::Parameters parameters = rawdevCore.imageWriterParameters();
		parameters.insert(ImageWriter::parameterQuality, imageWriterQuality);
		rawdevCore.setImageWriterParameters(parameters);
	}

	if(!rdfOutputFilename.isEmpty())
		rawdevCore.saveMetadata(rdfOutputFilename);

	if(!bayerOutputFilename.isEmpty())
		rawdevCore.saveBayerImage(bayerOutputFilename);

	if(!outputFilename.isEmpty())
		rawdevCore.saveImage(outputFilename);

	if(gui) {
		QMainWindow mainWindow;
		RawdevGui rawdev(&mainWindow, &rawdevCore);
		mainWindow.show();
		return application.exec();
	}

	return 0;
}

#endif
