/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef RAWIMPORTER_H_
#define RAWIMPORTER_H_

#include <QList>
#include <QMainWindow>
#include <QPointer>
#include <QPointF>
#include <QThread>

#include <BergPhoto.h>
#include <ColorMath.h>
#include <Image.h>
#include <ImageFilter.h>
#include <Metadata.h>
#include <RawFile.h>
#include <filters/DemosaicFilter.h>
#include <filters/DevelopFilter.h>
#include <filters/ExposureFilter.h>

#include <ui_rawdev.h>


class RawdevCore : public QObject {

	Q_OBJECT

public:
	RawdevCore();
	~RawdevCore();

	bool fileOpen(QString fileName);
	bool fileClose();
	void saveImage(QString fileName);
	void saveBayerImage(QString fileName);
	void saveMetadata(QString fileName);

	bool imageLoaded();

	int sourceWidth();
	int sourceHeight();

	double rotateAngle();
	void setRotateAngle(double angle);

	int cropLeft();
	void setCropLeft(int cropLeft);

	int cropRight();
	void setCropRight(int cropRight);

	int cropTop();
	void setCropTop(int cropTop);

	int cropBottom();
	void setCropBottom(int cropBottom);

	QString whiteBalance();
	void setWhiteBalance(QString whiteBalance);

	int whiteBalanceTemperature();
	void setWhiteBalanceTemperature(int temperature);

	int whiteBalanceTint();
	void setWhiteBalanceTint(int tint);

	QList<QPointF> toneCurve();
	void setToneCurve(Spline curve);

	BergPhoto::Parameters developParameters();
	void setDevelopParameters(BergPhoto::Parameters parameters);

	BergPhoto::Parameters imageWriterParameters();
	void setImageWriterParameters(BergPhoto::Parameters parameters);

	ImageFilterChain* previewFilterChain();

signals:
	void metadataPropertyChanged(MetadataProperty* property);

private:
	DemosaicFilter _demosaicFilter;
	ExposureFilter _exposureFilter;
	DevelopFilter _developFilter;
	BergPhoto::Parameters _developParameters;
	BergPhoto::Parameters _imageWriterParamters;
	RawFile* _rawFile;
	MetadataResource* _metadataResource;
	MetadataNode* _settingsDefault;
	MetadataNode* _settingsCurrent;
	ImageFilterChain* _previewFilterChain;

	void setSimplePropertySettingsCurrent(MetadataQName qName, QVariant value);
};

class RawdevGui : public QObject {

	Q_OBJECT

public:
	RawdevGui(QMainWindow* mainWindow, RawdevCore* rawdevCore);
	~RawdevGui();

public slots:
	void fileOpen();
	void fileClose();
	void saveImage();
	void setOutputParameters();
	void saveBayerImage();
	void saveMetadata();
	void setRotateAngle(double angle);
	void setCropLeft(int cropLeft);
	void setCropRight(int cropRight);
	void setCropTop(int cropTop);
	void setCropBottom(int cropBottom);
	void setWhiteBalanceMode(QString mode);
	void setWhiteBalanceTemperature(int temperature);
	void setWhiteBalanceTint(int tint);
	void setToneCurve(Spline curve);

private:
	QMainWindow* _mainWindow;
	Ui::MainWindowRawdev _mainWindowRawdev;

	RawdevCore* _rawdevCore;

	void syncGui();
};

#endif /*RAWIMPORTER_H_*/
