/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "DemosaicFilter.h"

#include <QDebug>

#include <RawFile.h>

DemosaicFilter::DemosaicFilter() {
}

DemosaicFilter::~DemosaicFilter() {
}

QString DemosaicFilter::name() {
	return QString("DemosaicFilter");
}

bool DemosaicFilter::accepts(ImageFormat inputFormat, ImageFormat outputFormat) {
	return inputFormat.type() == BayerImage16::type() && outputFormat.type() == RgbImage48::type();
}

bool DemosaicFilter::isParamterUsed(QString key) {
	Q_UNUSED(key);

	return false;
}

bool DemosaicFilter::isPropertyUsed(MetadataProperty* property) {
	if(MetadataQuery::property(MetadataNode::parent(property), ImageFilterSettings::propertyName, "current") != 0) {
		QSet<MetadataQName> qNames;

		qNames << RawFile::propertyCfaPatternDim << RawFile::propertyCfaPattern << RawFile::propertySensorCrop;

		if(qNames.contains(property->qName()))
			return true;
	}

	return false;
}

ImageFormat DemosaicFilter::prepare(MetadataResource* metadata, ImageFormat inputFormat, ImageFormat outputFormat) {
	if(accepts(inputFormat, outputFormat)) {
		int leftBorder = 0;
		int topBorder = 0;
		int rightBorder = inputFormat.width()-1;
		int bottomBorder = inputFormat.height()-1;

		QList<int> sensorCrop = MetadataQuery::typedValues<int>(metadata, RawFile::propertySensorCrop);

		if(!sensorCrop.isEmpty()) {
			leftBorder = sensorCrop.at(0);
			topBorder = sensorCrop.at(1);
			rightBorder = sensorCrop.at(2)+leftBorder-1;
			bottomBorder = sensorCrop.at(3)+topBorder-1;
		}

		int targetWidth = rightBorder - leftBorder + 1;
		int targetHeight = bottomBorder - topBorder + 1;

		return ImageFormat(outputFormat.type(), targetWidth, targetHeight);
	} else {
		return ImageFormat();
	}
}

Image* DemosaicFilter::filter(Image* input, ImageFormat outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	BayerImage16* bayerImage = qobject_cast<BayerImage16*>(input);

	MetadataResource* metadataResource = input->metadata();

	int bayerWidth = bayerImage->width();
	int bayerHeight = bayerImage->height();
	int bayerStripeSize = bayerImage->stripeSize();
	int leftBorder = 0;
	int topBorder = 0;
	int rightBorder = bayerWidth-1;
	int bottomBorder = bayerHeight-1;

	QList<int> cfaPattern = MetadataQuery::typedValues<int>(metadataResource, RawFile::propertyCfaPattern);
	QList<int> cfaPatternDim = MetadataQuery::typedValues<int>(metadataResource, RawFile::propertyCfaPatternDim);
	QList<int> sensorCrop = MetadataQuery::typedValues<int>(metadataResource, RawFile::propertySensorCrop);

	if(!sensorCrop.isEmpty()) {
		leftBorder = sensorCrop.at(0);
		topBorder = sensorCrop.at(1);
		rightBorder = sensorCrop.at(2)+leftBorder-1;
		bottomBorder = sensorCrop.at(3)+topBorder-1;
	}

	int width = rightBorder - leftBorder + 1;
	int height = bottomBorder - topBorder + 1;

	RgbImage48* rgbImage = new RgbImage48(width, height);

	rgbImage->setMetadata(bayerImage->metadata());

	quint16* bayerData = (quint16*)bayerImage->data();

	int bayerMask[3][4];

	QList<int> patternGBRG;
	patternGBRG << Image::ColorGreen << Image::ColorBlue << Image::ColorRed << Image::ColorGreen;

	QList<int> patternGRBG;
	patternGRBG << Image::ColorGreen << Image::ColorRed << Image::ColorBlue << Image::ColorGreen;

	QList<int> patternRGGB;
	patternRGGB << Image::ColorRed << Image::ColorGreen << Image::ColorGreen << Image::ColorBlue;

#define FILLMASKROW(r,a,b,c,d) r[0]=a; r[1]=b; r[2]=c; r[3]=d;

	if(cfaPattern == patternRGGB) {
		FILLMASKROW(bayerMask[0], 0, 1, 2, 3);
		FILLMASKROW(bayerMask[1], 4, 0, 0, 4);
		FILLMASKROW(bayerMask[2], 3, 2, 1, 0);
	} else if(cfaPattern == patternGRBG) {
		FILLMASKROW(bayerMask[0], 1, 0, 3, 2);
		FILLMASKROW(bayerMask[1], 0, 4, 4, 0);
		FILLMASKROW(bayerMask[2], 2, 3, 0, 1);
	} else if(cfaPattern == patternGBRG) {
		FILLMASKROW(bayerMask[0], 2, 3, 0, 1);
		FILLMASKROW(bayerMask[1], 0, 4, 4, 0);
		FILLMASKROW(bayerMask[2], 1, 0, 3, 2);
	} else {
		qDebug() << "unkown bayer pattern";
		for(int i=0; i<cfaPattern.size(); i++)
			qDebug() << cfaPattern.at(i);

		FILLMASKROW(bayerMask[0], 0, 0, 0, 0);
		FILLMASKROW(bayerMask[1], 0, 0, 0, 0);
		FILLMASKROW(bayerMask[2], 0, 0, 0, 0);
	}

#undef FILLMASKROW

	int border = 2;

	for(int y=topBorder; y<topBorder+border; y++) {
		quint16* data = (quint16*)rgbImage->dataStripe(y-topBorder);

		for(int x=leftBorder; x<=rightBorder; x++) {
			data[0] = bayerPixelSave(bayerData, bayerMask[0], x, y, bayerStripeSize, bayerWidth, bayerHeight);
			data[1] = bayerPixelSave(bayerData, bayerMask[1], x, y, bayerStripeSize, bayerWidth, bayerHeight);
			data[2] = bayerPixelSave(bayerData, bayerMask[2], x, y, bayerStripeSize, bayerWidth, bayerHeight);

			data += 3;
		}
	}

	for(int y=bottomBorder-border; y<=bottomBorder; y++) {
		quint16* data = (quint16*)rgbImage->dataStripe(y-topBorder);

		for(int x=leftBorder; x<=rightBorder; x++) {
			data[0] = bayerPixelSave(bayerData, bayerMask[0], x, y, bayerStripeSize, bayerWidth, bayerHeight);
			data[1] = bayerPixelSave(bayerData, bayerMask[1], x, y, bayerStripeSize, bayerWidth, bayerHeight);
			data[2] = bayerPixelSave(bayerData, bayerMask[2], x, y, bayerStripeSize, bayerWidth, bayerHeight);

			data += 3;
		}
	}

	for(int y=topBorder+border; y<bottomBorder-border; y++) {
		quint16* data = (quint16*)rgbImage->dataStripe(y-topBorder);

		for(int x=leftBorder; x<leftBorder+border; x++) {
			data[0] = bayerPixelSave(bayerData, bayerMask[0], x, y, bayerStripeSize, bayerWidth, bayerHeight);
			data[1] = bayerPixelSave(bayerData, bayerMask[1], x, y, bayerStripeSize, bayerWidth, bayerHeight);
			data[2] = bayerPixelSave(bayerData, bayerMask[2], x, y, bayerStripeSize, bayerWidth, bayerHeight);

			data += 3;
		}
	}

	for(int y=topBorder+border; y<bottomBorder-border; y++) {
		quint16* data = (quint16*)rgbImage->dataOffset(rightBorder-border-leftBorder, y-topBorder);

		for(int x=rightBorder-border; x<=rightBorder; x++) {
			data[0] = bayerPixelSave(bayerData, bayerMask[0], x, y, bayerStripeSize, bayerWidth, bayerHeight);
			data[1] = bayerPixelSave(bayerData, bayerMask[1], x, y, bayerStripeSize, bayerWidth, bayerHeight);
			data[2] = bayerPixelSave(bayerData, bayerMask[2], x, y, bayerStripeSize, bayerWidth, bayerHeight);

			data += 3;
		}
	}

	for(int y=topBorder+border; y<bottomBorder-border; y++) {
		quint16* data = (quint16*)rgbImage->dataOffset(border, y-topBorder);

		for(int x=leftBorder+border; x<rightBorder-border; x++) {
			data[0] = bayerPixel(bayerData, bayerMask[0], x, y, bayerStripeSize);
			data[1] = bayerPixel(bayerData, bayerMask[1], x, y, bayerStripeSize);
			data[2] = bayerPixel(bayerData, bayerMask[2], x, y, bayerStripeSize);

			data += 3;
		}
	}

	return rgbImage;
}
