/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef DEMOSAICFILTER_H_
#define DEMOSAICFILTER_H_

#include <ImageFilter.h>

class DemosaicFilter : public ImageFilter {
public:
	DemosaicFilter();

	virtual ~DemosaicFilter();

	virtual QString name();

	virtual bool accepts(ImageFormat inputFormat, ImageFormat outputFormat);

	virtual bool isParamterUsed(QString key);

	virtual bool isPropertyUsed(MetadataProperty* property);

	virtual ImageFormat prepare(MetadataResource* metadata, ImageFormat inputFormat, ImageFormat outputFormat);

	virtual Image* filter(Image* input, ImageFormat outputFormat);

private:
	static inline quint16 bayerPixel(quint16* d, int* bm, int x, int y, int s) {
		quint16* p = d+x+y*s;

		switch(bm[(x&1) | (y<<1&2)]) {
			case 0:
				return p[0];
			case 1:
				return ((int)p[-1]+(int)p[1]+1)/2;
			case 2:
				return ((int)p[-s]+(int)p[s]+1)/2;
			case 3:
				return ((int)p[-s-1]+(int)p[-s+1]+(int)p[s-1]+(int)p[s+1]+2)/4;
			case 4:
				return ((int)p[-s]+(int)p[-1]+(int)p[1]+(int)p[s]+2)/4;
			default:
				return 0;
		}
	}

	static inline void getPixelSave(quint16* d, int* v, int* c, int x, int y, int s, int w, int h) {
		if(x >= w || x < 0 || y >= h || y < 0)
			return;

		(*c)++;
		*v += *(d+x+y*s);
	}

	static inline quint16 bayerPixelSave(quint16* d, int* bm, int x, int y, int s, int w, int h) {
		int c = 0, v = 0;

		switch(bm[(x&1) | (y<<1&2)]) {
			case 0:
				getPixelSave(d, &v, &c, x, y, s, w, h);
				break;
			case 1:
				getPixelSave(d, &v, &c, x-1, y, s, w, h);
				getPixelSave(d, &v, &c, x+1, y, s, w, h);
				break;
			case 2:
				getPixelSave(d, &v, &c, x, y-1, s, w, h);
				getPixelSave(d, &v, &c, x, y+1, s, w, h);
				break;
			case 3:
				getPixelSave(d, &v, &c, x-1, y-1, s, w, h);
				getPixelSave(d, &v, &c, x+1, y-1, s, w, h);
				getPixelSave(d, &v, &c, x-1, y+1, s, w, h);
				getPixelSave(d, &v, &c, x+1, y+1, s, w, h);
				break;
			case 4:
				getPixelSave(d, &v, &c, x, y-1, s, w, h);
				getPixelSave(d, &v, &c, x-1, y, s, w, h);
				getPixelSave(d, &v, &c, x+1, y, s, w, h);
				getPixelSave(d, &v, &c, x, y+1, s, w, h);
				break;
			default:
				return 0;
		}

		if(c == 1)
			return v;
		else if(c == 2)
			return (v+1)/2;
		else if(c == 3)
			return (v+1)/3;
		else if(c == 4)
			return (v+2)/4;
		else
			return 0;
	}
};

#endif /* DEMOSAICFILTER_H_ */
