/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "DevelopFilter.h"

QString DevelopFilter::parameterOutputProfile = "outputProfile";
QString DevelopFilter::profileSRgb = "sRgb";
QString DevelopFilter::profileProPhotoRgbLinear = "proPhotoLinearRgb";
QString DevelopFilter::profileAdobeRgb = "adobeRgb";

DevelopFilter::DevelopFilter() {
}

DevelopFilter::~DevelopFilter() {
}

QString DevelopFilter::name() {
	return QString("DevelopFilter");
}

bool DevelopFilter::accepts(ImageFormat inputFormat, ImageFormat outputFormat) {
	return inputFormat.type() == RgbImage48::type() && (outputFormat.type() == RgbImage32::type() || outputFormat.type() == RgbImage48::type());
}

bool DevelopFilter::isParamterUsed(QString key) {
	if(key == parameterOutputProfile)
		return true;

	return false;
}

bool DevelopFilter::isPropertyUsed(MetadataProperty* property) {
	Q_UNUSED(property);

	return false;
}

ImageFormat DevelopFilter::prepare(MetadataResource* metadata, ImageFormat inputFormat, ImageFormat outputFormat) {
	Q_UNUSED(metadata);

	if(accepts(inputFormat, outputFormat))
		return ImageFormat(outputFormat.type(), inputFormat.width(), inputFormat.height());
	else
		return ImageFormat();
}

Image* DevelopFilter::filter(Image* input, ImageFormat outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	int width = input->width();
	int height = input->height();
	CmsProfile* inputProfile = input->colorProfile();

	QString outputProfileName = profileSRgb;

	if(_parameters.contains(parameterOutputProfile))
		outputProfileName = _parameters.value(parameterOutputProfile).toString();

	CmsProfile* outputProfile = 0;

	if(outputProfileName == profileSRgb)
		outputProfile = CmsProfile::sRgbProfile();
	else if(outputProfileName == profileAdobeRgb)
		outputProfile = CmsProfile::adobeRgbProfile();
	else if(outputProfileName == profileProPhotoRgbLinear)
		outputProfile = CmsProfile::proPhotoRgbLinearProfile();
	else
		return 0;

	if(outputFormat.type() == RgbImage32::type()) {
		RgbImage32* target = new RgbImage32(width, height);
		target->setMetadata(input->metadata());
		target->setColorProfile(outputProfile);

		CmsTransform* transform = new CmsTransform(inputProfile, CmsTransform::Rgb16, outputProfile, CmsTransform::Bgra8);

		for(int y=0; y<height; y++) {
			void* sourceStripe = (void*)input->dataStripe(y);
			void* targetStripe = (void*)target->dataStripe(y);

			memset(targetStripe, 0xff, width*4);

			transform->transform(sourceStripe, targetStripe, width);
		}

		delete transform;

		return target;
	} else if(outputFormat.type() == RgbImage48::type()) {
		RgbImage48* target = new RgbImage48(width, height);
		target->setMetadata(input->metadata());
		target->setColorProfile(outputProfile);

		CmsTransform* transform = new CmsTransform(inputProfile, CmsTransform::Rgb16, outputProfile, CmsTransform::Rgb16);

		for(int y=0; y<height; y++) {
			void* sourceStripe = (void*)input->dataStripe(y);
			void* targetStripe = (void*)target->dataStripe(y);

			transform->transform(sourceStripe, targetStripe, width);
		}

		delete transform;

		return target;
	} else {
		return 0;
	}
}
