/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "ExposureFilter.h"

#include <QDebug>

#include <RawFile.h>

ExposureFilter::ExposureFilter() {
}

ExposureFilter::~ExposureFilter() {
}

QString ExposureFilter::name() {
	return QString("ExposureFilter");
}

bool ExposureFilter::accepts(ImageFormat inputFormat, ImageFormat outputFormat) {
	return inputFormat.type() == RgbImage48::type() && outputFormat.type() == RgbImage48::type();
}

bool ExposureFilter::isParamterUsed(QString key) {
	Q_UNUSED(key);

	return false;
}

bool ExposureFilter::isPropertyUsed(MetadataProperty* property) {
	if(MetadataQuery::property(MetadataNode::parent(property), ImageFilterSettings::propertyName, "current") != 0) {
		QSet<MetadataQName> qNames;

		qNames << RawExposure::propertyWhiteBalance << RawExposure::propertyTemperature << RawExposure::propertyTint << RawExposure::propertyToneCurve;

		if(qNames.contains(property->qName()))
			return true;
	}

	return false;
}

ImageFormat ExposureFilter::prepare(MetadataResource* metadata, ImageFormat inputFormat, ImageFormat outputFormat) {
	Q_UNUSED(metadata);

	if(accepts(inputFormat, outputFormat))
		return ImageFormat(outputFormat.type(), inputFormat.width(), inputFormat.height());
	else
		return ImageFormat();
}

Image* ExposureFilter::filter(Image* input, ImageFormat outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	RgbImage48* rgbInput = qobject_cast<RgbImage48*>(input);

	int width = rgbInput->width();
	int height = rgbInput->height();

	RgbImage48* rgbImageOutput = new RgbImage48(width, height);

	rgbImageOutput->setColorProfile(CmsProfile::proPhotoRgbLinearProfile());
	rgbImageOutput->setMetadata(input->metadata());

	MetadataResource* metadata = input->metadata();
	Matrix3x3 matrix;
	double blackLevel = 0.0;
	double whiteLevel = 65535.0;
	//QString cameraProfile;
	MetadataNode* settingsCurrent = MetadataNode::parent(MetadataQuery::property(metadata, ImageFilterSettings::propertyName, QVariant("current")));
	QString whiteBalance = MetadataQuery::typedValue<QString>(settingsCurrent, RawExposure::propertyWhiteBalance);
	double temperature = MetadataQuery::typedValue<double>(settingsCurrent, RawExposure::propertyTemperature);
	double tint = MetadataQuery::typedValue<double>(settingsCurrent, RawExposure::propertyTint);
	Spline toneCurveSpline = Spline(MetadataQuery::typedValues<QPointF>(settingsCurrent, RawExposure::propertyToneCurve));

	/*if(_parameters.contains("cameraProfile"))
		cameraProfile = _parameters.value("cameraProfile").toString();*/

	blackLevel = MetadataQuery::typedValue<double>(metadata, RawFile::propertyBlackLevel);
	whiteLevel = MetadataQuery::typedValue<double>(metadata, RawFile::propertyWhiteLevel);

	double divisor = 1.0 / (whiteLevel - blackLevel);

	/*CmsProfile* inProfile = 0;
	CmsProfile* outProfile = 0;
	CmsTransform* transform = 0;

	if(!cameraProfile.isEmpty()) {
		inProfile = new CmsProfile(cameraProfile);
		outProfile = CmsProfile::proPhotoLinearProfile();
		transform = new CmsTransform(inProfile, CmsTransform::Rgb16, outProfile, CmsTransform::Rgb16);
	} else {
	}*/

	ColorXYZ whitePointXYZ;

	if(whiteBalance == "manual")
		whitePointXYZ = TemperatureUtils::toxy(temperature, tint).toColorXYZ();
	else
		whitePointXYZ = RawUtils::cameraNeutralWhiteBalance(metadata).toColorXYZ();

	TemperatureUtils::fromxy(whitePointXYZ.toColorxy(), &temperature, &tint);

	Matrix3x3 cameraMatrix = RawUtils::cameraMatrix(metadata, temperature);
	Matrix3x3 whitePointMatrix = whitePointXYZ.toMatrix3x3();
	Matrix3x3 whitePointShift = CIEUtils::whitePointXYZ2WhitePointXYZ(Colorxy::whitePointD50().toColorXYZ(), whitePointXYZ);
	Matrix3x3 toTargetColorSpace = CIEUtils::fromXYZ2ProPhotoRgb();

	matrix = toTargetColorSpace * (cameraMatrix * whitePointShift).inverse();

	/*QVector<double> cameraForwardVector;
	cameraForwardVector << 0.8716 << -0.0692 << 0.1618 << 0.3408 << 0.8077 << -0.1486 << -0.0013 << -0.6583 << 1.4847;
	Matrix3x3 cameraForwardMatrix(cameraForwardVector.data());

	ColorXYZ cameraNeutral = ColorXYZ(MetadataQuery::typedValues<double>(metadata, RawFile::propertyWhiteBalanceNeutral));
	Matrix3x3 dMatrix = (RawUtils::cameraCalibration(metadata, temperature).inverse() * Matrix3x3(cameraNeutral.x(), cameraNeutral.y(), cameraNeutral.z())).inverse();

	matrix = cameraForwardMatrix * dMatrix * RawUtils::cameraCalibration(metadata, temperature).inverse();
	matrix = toTargetColorSpace * matrix;*/

	qDebug() << "white level:" << whiteLevel;
	qDebug() << "black level:" << blackLevel;
	qDebug() << "divisor:" << divisor;

	qDebug("camera matrix:");
	cameraMatrix.debug();
	qDebug("white point matrix:");
	whitePointMatrix.debug();
	qDebug("white point shift:");
	whitePointShift.debug();
	qDebug("to target color space:");
	toTargetColorSpace.debug();
	qDebug("final matrix:");
	matrix.debug();

	//quint16 buffer[width*3];

	double curve[65536];
	double d = 1.0 / 65535.0;

	for(int i=0; i<65536; i++) {
		double v = (double)i * d;

		v = toneCurveSpline.evaluate(qBound(0.0, v, 1.0));

		curve[i] = qBound(0.0, v, 1.0);
	}

	for(int y=0; y<height; y++) {
		quint16* dataInput = (quint16*)rgbInput->dataStripe(y);
		quint16* dataOutput = (quint16*)rgbImageOutput->dataStripe(y);

		for(int x=0; x<width; x++) {
			double p[3];

			p[0] = (double)dataInput[0];
			p[1] = (double)dataInput[1];
			p[2] = (double)dataInput[2];

			p[0] -= blackLevel;
			p[1] -= blackLevel;
			p[2] -= blackLevel;

			p[0] *= divisor;
			p[1] *= divisor;
			p[2] *= divisor;

			matrix.multiplyVector(p);

			dataOutput[0] = (quint16)(curve[(int)qBound(0.0, p[0]*65535.0, 65535.0)] * 65535.0);
			dataOutput[1] = (quint16)(curve[(int)qBound(0.0, p[1]*65535.0, 65535.0)] * 65535.0);
			dataOutput[2] = (quint16)(curve[(int)qBound(0.0, p[2]*65535.0, 65535.0)] * 65535.0);

			/*if(transform == 0) {*/
				/*dataOutput[0] = (quint16)qBound(0.0, p[0] * 65535.0, 65535.0);
				dataOutput[1] = (quint16)qBound(0.0, p[1] * 65535.0, 65535.0);
				dataOutput[2] = (quint16)qBound(0.0, p[2] * 65535.0, 65535.0);*/
			/*} else {
				buffer[x*3+0] = (quint16)qBound(0.0, p[0] * 65535.0, 65535.0);
				buffer[x*3+1] = (quint16)qBound(0.0, p[1] * 65535.0, 65535.0);
				buffer[x*3+2] = (quint16)qBound(0.0, p[2] * 65535.0, 65535.0);
			}*/

			dataInput += 3;
			dataOutput += 3;
		}

		/*if(transform != 0)
			transform->transform((void*)buffer, (void*)dataOutput, width);*/
	}

	/*if(transform != 0) {
		delete inProfile;
		delete outProfile;
		delete transform;
	}*/

	return rgbImageOutput;
}
