/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "RotateCropFilter.h"

QString RotateCropFilter::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/filter/rotate-crop/1.0/";
MetadataQName RotateCropFilter::propertyRotateAngle = MetadataQName(RotateCropFilter::namespaceUri, "RotateAngle");
MetadataQName RotateCropFilter::propertyCropLeft = MetadataQName(RotateCropFilter::namespaceUri, "CropLeft");
MetadataQName RotateCropFilter::propertyCropRight = MetadataQName(RotateCropFilter::namespaceUri, "CropRight");
MetadataQName RotateCropFilter::propertyCropTop = MetadataQName(RotateCropFilter::namespaceUri, "CropTop");
MetadataQName RotateCropFilter::propertyCropBottom = MetadataQName(RotateCropFilter::namespaceUri, "CropBottom");

RotateCropFilter::RotateCropFilter() {
}

RotateCropFilter::~RotateCropFilter() {
}

QString RotateCropFilter::name() {
	return QString("RotateCropFilter");
}

bool RotateCropFilter::accepts(ImageFormat inputFormat, ImageFormat outputFormat) {
	return inputFormat.type() == RgbImage48::type() && outputFormat.type() == RgbImage48::type();
}

bool RotateCropFilter::isParamterUsed(QString key) {
	if(key == "scale" || key == "interpolation")
		return true;

	return false;
}

bool RotateCropFilter::isPropertyUsed(MetadataProperty* property) {
	if(MetadataQuery::property(MetadataNode::parent(property), ImageFilterSettings::propertyName, "current") != 0) {
		QSet<MetadataQName> qNames;

		qNames << propertyCropBottom << propertyCropLeft << propertyCropRight << propertyCropTop << propertyCropTop << propertyRotateAngle << Image::propertyOrientation;

		if(qNames.contains(property->qName()))
			return true;
	}

	return false;
}

ImageFormat RotateCropFilter::prepare(MetadataResource* metadata, ImageFormat inputFormat, ImageFormat outputFormat) {
	if(accepts(inputFormat, outputFormat)) {
		MetadataNode* settingsCurrent = MetadataQuery::property(metadata, ImageFilterSettings::propertyName, QVariant("current"))->parent();

		double orientation = MetadataQuery::typedValue<double>(settingsCurrent, Image::propertyOrientation) * (M_PI/180.0);
		double angle = MetadataQuery::typedValue<double>(settingsCurrent, RotateCropFilter::propertyRotateAngle) * (M_PI/180.0) + orientation;
		int cropLeft = MetadataQuery::typedValue<int>(settingsCurrent, RotateCropFilter::propertyCropLeft);
		int cropRight = MetadataQuery::typedValue<int>(settingsCurrent, RotateCropFilter::propertyCropRight);
		int cropTop = MetadataQuery::typedValue<int>(settingsCurrent, RotateCropFilter::propertyCropTop);
		int cropBottom = MetadataQuery::typedValue<int>(settingsCurrent, RotateCropFilter::propertyCropBottom);
		int targetWidth = -1;
		int targetHeight = -1;

		if(_parameters.contains("scale")) {
			double scale = _parameters.value("scale").toDouble();

			cropLeft = (int)(cropLeft*scale);
			cropRight = (int)(cropRight*scale);
			cropTop = (int)(cropTop*scale);
			cropBottom = (int)(cropBottom*scale);
		}

		if(angle < 0.0 || angle >= M_PI*2.0)
			angle = angle - floor(angle/M_PI*0.5)*M_PI*2.0;

		if(fabs(angle) < 0.0001) {
			targetWidth = inputFormat.width()-cropLeft-cropRight;
			targetHeight = inputFormat.height()-cropTop-cropBottom;
		} else if(fabs(angle - M_PI*0.5) < 0.0001) {
			targetWidth = inputFormat.height()-cropLeft-cropRight;
			targetHeight = inputFormat.width()-cropTop-cropBottom;
		} else if(fabs(angle - M_PI) < 0.0001) {
			targetWidth = inputFormat.width()-cropLeft-cropRight;
			targetHeight = inputFormat.height()-cropTop-cropBottom;
		} else if(fabs(angle - M_PI*1.5) < 0.0001) {
			targetWidth = inputFormat.height()-cropLeft-cropRight;
			targetHeight = inputFormat.width()-cropTop-cropBottom;
		} else {
			double mc = cos(angle);
			double ms = sin(angle);

			targetWidth = (int)(fabs(inputFormat.width()*mc) + fabs(inputFormat.height()*ms)+0.5);
			targetHeight = (int)(fabs(inputFormat.height()*mc) + fabs(inputFormat.width()*ms)+0.5);
		}

		return ImageFormat(outputFormat.type(), targetWidth, targetHeight);
	} else {
		return ImageFormat();
	}
}

Image* RotateCropFilter::filter(Image* input, ImageFormat outputFormat) {
	if(!accepts(input->type(), outputFormat))
		return 0;

	RgbImage48* source = qobject_cast<RgbImage48*>(input);

	MetadataNode* settingsCurrent = MetadataQuery::property(input->metadata(), ImageFilterSettings::propertyName, QVariant("current"))->parent();

	double orientation = MetadataQuery::typedValue<double>(settingsCurrent, Image::propertyOrientation) * (M_PI/180.0);
	double angle = MetadataQuery::typedValue<double>(settingsCurrent, RotateCropFilter::propertyRotateAngle) * (M_PI/180.0) + orientation;
	int cropLeft = MetadataQuery::typedValue<int>(settingsCurrent, RotateCropFilter::propertyCropLeft);
	int cropRight = MetadataQuery::typedValue<int>(settingsCurrent, RotateCropFilter::propertyCropRight);
	int cropTop = MetadataQuery::typedValue<int>(settingsCurrent, RotateCropFilter::propertyCropTop);
	int cropBottom = MetadataQuery::typedValue<int>(settingsCurrent, RotateCropFilter::propertyCropBottom);

	int interpolate = InterpolateNearest;
	QString interpolateString = _parameters.value("interpolation").toString();
	if(interpolateString == "linear")
		interpolate = InterpolateLinear;
	else if(interpolateString == "cubic")
		interpolate = InterpolateCubic;

	if(_parameters.contains("scale")) {
		double scale = _parameters.value("scale").toDouble();

		cropLeft = (int)(cropLeft*scale);
		cropRight = (int)(cropRight*scale);
		cropTop = (int)(cropTop*scale);
		cropBottom = (int)(cropBottom*scale);
	}

	RgbImage48* target = 0;

	if(angle < 0.0 || angle >= M_PI*2.0)
		angle = angle - floor(angle/M_PI*0.5)*M_PI*2.0;

	if(fabs(angle) < 0.0001)
		target = crop(source, cropLeft, cropRight, cropTop, cropBottom);
	else if(fabs(angle - M_PI*0.5) < 0.0001)
		target = rotate90Crop(source, cropLeft, cropRight, cropTop, cropBottom);
	else if(fabs(angle - M_PI) < 0.0001)
		target = rotate180Crop(source, cropLeft, cropRight, cropTop, cropBottom);
	else if(fabs(angle - M_PI*1.5) < 0.0001)
		target = rotate270Crop(source, cropLeft, cropRight, cropTop, cropBottom);
	else
		target = rotateCrop(source, angle, interpolate, cropLeft, cropRight, cropTop, cropBottom);

	target->setMetadata(source->metadata());

	return target;
}

void RotateCropFilter::nearestNeighborInterpolation(RgbImage48* source, quint16* targetPixel, double x, double y) {
	quint16* sourcePixel = (quint16*)source->dataOffsetSave((int)(x+0.5), (int)(y+0.5));

	if(sourcePixel != 0) {
		targetPixel[0] = sourcePixel[0];
		targetPixel[1] = sourcePixel[1];
		targetPixel[2] = sourcePixel[2];
	} else {
		targetPixel[0] = 0;
		targetPixel[1] = 0;
		targetPixel[2] = 0;
	}
}

quint16 RotateCropFilter::bilinearInterpolation(quint16 p0, quint16 p1, quint16 p2, quint16 p3, double dx, double dy) {
	return (quint16)qBound(0.0,(1-dy)*(p0+dx*(p1-p0))+dy*(p2+dx*(p3-p2))+0.5,65535.0);
}

void RotateCropFilter::bilinearInterpolation(RgbImage48* source, quint16* targetPixel, double x, double y) {
	int x0 = (int)x;
	int y0 = (int)y;
	quint16* s0 = (quint16*)source->dataOffsetSave(x0, y0);
	int x1 = x0+1;
	int y1 = y0;
	quint16* s1 = (quint16*)source->dataOffsetSave(x1, y1);
	int x2 = x0;
	int y2 = y0+1;
	quint16* s2 = (quint16*)source->dataOffsetSave(x2, y2);
	int x3 = x0+1;
	int y3 = y0+1;
	quint16* s3 = (quint16*)source->dataOffsetSave(x3, y3);

	quint16 black[] = { 0, 0, 0 };

	s0 = s0 != 0 ? s0 : black;
	s1 = s1 != 0 ? s1 : black;
	s2 = s2 != 0 ? s2 : black;
	s3 = s3 != 0 ? s3 : black;

	double dx = x-x0;
	double dy = y-y0;

	targetPixel[0] = bilinearInterpolation(s0[0], s1[0], s2[0], s3[0], dx, dy);
	targetPixel[1] = bilinearInterpolation(s0[1], s1[1], s2[1], s3[1], dx, dy);
	targetPixel[2] = bilinearInterpolation(s0[2], s1[2], s2[2], s3[2], dx, dy);
}

double RotateCropFilter::cubicInterpolation(quint16 p0, quint16 p1, quint16 p2, quint16 p3, double d) {
	return ((((-p0+3*p1-3*p2+p3)*d+(2*p0-5*p1+4*p2-p3))*d+(-p0+p2))*d+(p1+p1))/2.0;
}

quint16 RotateCropFilter::cubicInterpolation(double p0, double p1, double p2, double p3, double d) {
	return (quint16)qBound(0.0,((((-p0+3*p1-3*p2+p3)*d+(2*p0-5*p1+4*p2-p3))*d+(-p0+p2))*d+(p1+p1))/2.0+0.5,65535.0);
}

void RotateCropFilter::bicubicInterpolation(RgbImage48* source, quint16* targetPixel, double x, double y) {
	int xi = (int)x;
	int yi = (int)y;
	double col[3][4];
	double dx = x-xi;
	double dy = y-yi;
	quint16 black[] = { 0, 0, 0 };

	for(int i=-1; i<=2; i++) {
		quint16* s0 = (quint16*)source->dataOffsetSave(xi-1, yi+i);
		quint16* s1 = (quint16*)source->dataOffsetSave(xi, yi+i);
		quint16* s2 = (quint16*)source->dataOffsetSave(xi+1, yi+i);
		quint16* s3 = (quint16*)source->dataOffsetSave(xi+2, yi+i);

		s0 = s0 != 0 ? s0 : black;
		s1 = s1 != 0 ? s1 : black;
		s2 = s2 != 0 ? s2 : black;
		s3 = s3 != 0 ? s3 : black;

		col[0][i+1] = cubicInterpolation(s0[0], s1[0], s2[0], s3[0], dx);
		col[1][i+1] = cubicInterpolation(s0[1], s1[1], s2[1], s3[1], dx);
		col[2][i+1] = cubicInterpolation(s0[2], s1[2], s2[2], s3[2], dx);
	}

	targetPixel[0] = cubicInterpolation(col[0][0], col[0][1], col[0][2], col[0][3], dy);
	targetPixel[1] = cubicInterpolation(col[1][0], col[1][1], col[1][2], col[1][3], dy);
	targetPixel[2] = cubicInterpolation(col[2][0], col[2][1], col[2][2], col[2][3], dy);
}

RgbImage48* RotateCropFilter::crop(RgbImage48* source, int cropLeft, int cropRight, int cropTop, int cropBottom) {
	int targetWidth = source->width()-cropLeft-cropRight;
	int targetHeight = source->height()-cropTop-cropBottom;

	RgbImage48* target = new RgbImage48(targetWidth, targetHeight);

	for(int y=0; y<targetHeight; y++) {
		quint16* sourcePixel = (quint16*)source->dataOffset(cropLeft, y+cropTop);
		quint16* targetPixel = (quint16*)target->dataStripe(y);

		memcpy(targetPixel, sourcePixel, targetWidth*6);
	}

	return target;
}

RgbImage48* RotateCropFilter::rotate90Crop(RgbImage48* source, int cropLeft, int cropRight, int cropTop, int cropBottom) {
	int targetWidth = source->height()-cropLeft-cropRight;
	int targetHeight = source->width()-cropTop-cropBottom;

	RgbImage48* target = new RgbImage48(targetWidth, targetHeight);

	for(int y=0; y<targetHeight; y++) {
		quint16* targetPixel = (quint16*)target->dataStripe(targetHeight-y-1);

		for(int x=0; x<targetWidth; x++) {
			quint16* sourcePixel = (quint16*)source->dataOffset(y+cropBottom, x+cropLeft);

			targetPixel[0] = sourcePixel[0];
			targetPixel[1] = sourcePixel[1];
			targetPixel[2] = sourcePixel[2];

			targetPixel += 3;
		}
	}

	return target;
}

RgbImage48* RotateCropFilter::rotate180Crop(RgbImage48* source, int cropLeft, int cropRight, int cropTop, int cropBottom) {
	int targetWidth = source->width()-cropLeft-cropRight;
	int targetHeight = source->height()-cropTop-cropBottom;

	RgbImage48* target = new RgbImage48(targetWidth, targetHeight);

	for(int y=0; y<targetHeight; y++) {
		quint16* sourcePixel = (quint16*)source->dataOffset(cropRight, y+cropBottom);
		quint16* targetPixel = (quint16*)target->dataOffset(targetWidth-1, targetHeight-y-1);

		for(int x=0; x<targetWidth; x++) {
			targetPixel[0] = sourcePixel[0];
			targetPixel[1] = sourcePixel[1];
			targetPixel[2] = sourcePixel[2];

			sourcePixel += 3;
			targetPixel -= 3;
		}
	}

	return target;
}

RgbImage48* RotateCropFilter::rotate270Crop(RgbImage48* source, int cropLeft, int cropRight, int cropTop, int cropBottom) {
	int targetWidth = source->height()-cropLeft-cropRight;
	int targetHeight = source->width()-cropTop-cropBottom;

	RgbImage48* target = new RgbImage48(targetWidth, targetHeight);

	for(int y=0; y<targetHeight; y++) {
		quint16* targetPixel = (quint16*)target->dataOffset(targetWidth-1, y);

		for(int x=0; x<targetWidth; x++) {
			quint16* sourcePixel = (quint16*)source->dataOffset(y+cropTop, x+cropRight);

			targetPixel[0] = sourcePixel[0];
			targetPixel[1] = sourcePixel[1];
			targetPixel[2] = sourcePixel[2];

			targetPixel -= 3;
		}
	}

	return target;
}

RgbImage48* RotateCropFilter::rotateCrop(RgbImage48* source, double angle, int interpolation, int cropLeft, int cropRight, int cropTop, int cropBottom) {
	double mc = cos(angle);
	double ms = sin(angle);

	int targetWidth = (int)(fabs(source->width()*mc) + fabs(source->height()*ms)+0.5);
	int targetWidth2 = targetWidth/2;
	int targetHeight = (int)(fabs(source->height()*mc) + fabs(source->width()*ms)+0.5);
	int targetHeight2 = targetHeight/2;
	int sourceWidth2 = source->width()/2;
	int sourceHeight2 = source->height()/2;

	void (*interpolateFunc)(RgbImage48*, quint16*, double, double);

	switch(interpolation) {
		case InterpolateLinear:
			interpolateFunc = bilinearInterpolation;
			break;

		case InterpolateCubic:
			interpolateFunc = bicubicInterpolation;
			break;

		default:
			interpolateFunc = nearestNeighborInterpolation;
			break;
	}

	RgbImage48* target = new RgbImage48(targetWidth-cropLeft-cropRight, targetHeight-cropTop-cropBottom);

	for(int y=-targetHeight2+cropTop; y<targetHeight2-cropBottom; y++) {
		quint16* targetPixel = (quint16*)target->dataStripe(y+targetHeight2-cropTop);

		for(int x=-targetWidth2+cropLeft; x<targetWidth2-cropRight; x++) {
			double dx = x*mc - y*ms + sourceWidth2;
			double dy = x*ms + y*mc + sourceHeight2;

			interpolateFunc(source, targetPixel, dx, dy);

			targetPixel += 3;
		}
	}

	return target;
}
