/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "CanonCr2.h"

#include <QBuffer>
#include <QTextStream>

#include <MetadataRdf.h>
#include <QtUtils.h>

QString CanonCr2::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/makernote/canon/1.0/";
TiffMetadataQName CanonCr2::propertyShotInfo = TiffMetadataQName(CanonCr2::namespaceUri, "ShotInfo", 0x4, Tiff::TypeShort);
TiffMetadataQName CanonCr2::propertyBodyInfo = TiffMetadataQName(CanonCr2::namespaceUri, "BodyInfo", 0xc, Tiff::TypeShort);
TiffMetadataQName CanonCr2::propertySensorInfo = TiffMetadataQName(CanonCr2::namespaceUri, "SensorInfo", 0xe0, Tiff::TypeShort);
TiffMetadataQName CanonCr2::propertyColorBalance = TiffMetadataQName(CanonCr2::namespaceUri, "ColorBalance", 0x4001, Tiff::TypeShort);
TiffMetadataQName CanonCr2::propertySensorPlanes = TiffMetadataQName(CanonCr2::namespaceUri, "SensorPlanes", 0xc640, Tiff::TypeShort);
MetadataQName CanonCr2::propertyDefaultWhiteBalanceNeutral5200K = MetadataQName(CanonCr2::namespaceUri, "DefaultWhiteBalanceNeutral5200K");
QList<TiffMetadataQName> CanonCr2::_tiffMetadataQNames = QList<TiffMetadataQName>();

QList<TiffMetadataQName> CanonCr2::tiffMetadataQNames() {
	if(_tiffMetadataQNames.isEmpty()) {
		_tiffMetadataQNames.append(propertyShotInfo);
		_tiffMetadataQNames.append(propertyBodyInfo);
		_tiffMetadataQNames.append(propertySensorInfo);
		_tiffMetadataQNames.append(propertyColorBalance);
		_tiffMetadataQNames.append(propertySensorPlanes);
	}

	return _tiffMetadataQNames;
}

CanonCr2Image::CanonCr2Image(int width, int height) : BayerImage16(width, height) {
	_type = type();
}

CanonCr2Image::~CanonCr2Image() {
}

QMap<quint16, MetadataQName> CanonCr2Reader::_makerNodeTagMetadataQNameMap = QMap<quint16, MetadataQName>();

CanonCr2Reader::CanonCr2Reader(QIODevice* device) : TiffReader(device) {
}

MetadataNode* CanonCr2Reader::read(MetadataNode* parent) {
	Q_UNUSED(parent)

	_document = new MetadataDocument();

	parseHeader(_document);

	_dataStream->device()->seek(12);
	*_dataStream >> _rawIfdOffset;

	parseDirectories(_document, 4);

	return _document;
}

void CanonCr2Reader::parseDirectory(MetadataNode* parent, quint32 offset) {
	if(offset == _rawIfdOffset)
		new MetadataProperty(parent, RawFile::propertyRawResourceMarker);

	TiffReader::parseDirectory(parent, offset);
}

MetadataQName CanonCr2Reader::tagQName(quint16 tag) {
	if(makerNodeTagMetadataQNameMap().contains(tag))
		return makerNodeTagMetadataQNameMap().value(tag);
	else
		return TiffReader::tagQName(tag);
}


void CanonCr2Reader::parseMakerNote(MetadataNode* parent, quint32 offset) {
	parseDirectory(parent, offset);
}

QMap<quint16, MetadataQName> CanonCr2Reader::makerNodeTagMetadataQNameMap() {
	if(_makerNodeTagMetadataQNameMap.isEmpty()) {
		foreach(TiffMetadataQName qName, CanonCr2::tiffMetadataQNames()) {
			_makerNodeTagMetadataQNameMap.insert(qName.tag(), qName);
		}
	}

	return _makerNodeTagMetadataQNameMap;
}

CanonCr2File::CanonCr2File(QIODevice* device) : RawFile() {
	_device = device;
	_canonCr2Reader = new CanonCr2Reader(_device);
	_rawOffset = 0;
	_rawLength = 0;
}

CanonCr2File::~CanonCr2File() {
	delete _canonCr2Reader;
}

MetadataDocument* CanonCr2File::readMetadata() {
	if(_metadata == 0) {
		CanonCr2MetadataCleanupTransform transform;

		MetadataNode* metadata = _canonCr2Reader->read();
#ifdef QT_DEBUG
		MetadataRdfWriter::writeFile(metadata, "f:\\test.xml");
#endif
		QList<MetadataResource*> resources = MetadataQuery::resources(metadata, "*");

		_tnbOffset = MetadataQuery::typedValue<quint32>(resources.at(0), Tiff::propertyStripOffsets);
		_tnbLength = MetadataQuery::typedValue<quint32>(resources.at(0), Tiff::propertyStripByteCounts);

		_tnsOffset = MetadataQuery::typedValue<quint32>(resources.at(1), Tiff::propertyJPEGInterchangeFormat);
		_tnsLength = MetadataQuery::typedValue<quint32>(resources.at(1), Tiff::propertyJPEGInterchangeFormatLength);

		MetadataNode* rawMarker = MetadataQuery::property(metadata, RawFile::propertyRawResourceMarker);

		if(rawMarker == 0) {
			qDebug("raw marker not found");
			return 0;
		}

		MetadataNode* rawResource = rawMarker->parent();

		_rawOffset = MetadataQuery::typedValue<quint32>(rawResource, Tiff::propertyStripOffsets);
		_rawLength = MetadataQuery::typedValue<quint32>(rawResource, Tiff::propertyStripByteCounts);

		_metadata = MetadataNode::toDocument(transform.transform(metadata));
	}

	return _metadata;
}

Image* CanonCr2File::readImage() {
	if(_image == 0) {
		readMetadata();

		MetadataResource* resource = MetadataQuery::resource(_metadata, "*");

		QList<int> sensorInfo = MetadataQuery::typedValues<int>(resource, CanonCr2::propertySensorInfo);

		CanonCr2Image canonCr2Image(sensorInfo.at(1), sensorInfo.at(2));

		TiffImageReader tiffImageReader(_device);
		tiffImageReader.readLosslessJpeg(_rawOffset, _rawLength, &canonCr2Image);

		canonCr2Image.setMetadata(resource);

		Cr2BayerConverter cr2BayerConverter;
		_image = cr2BayerConverter.filter(&canonCr2Image, ImageFormat(BayerImage16::type()));

		int whiteLevel = (1 << canonCr2Image.precision()) -1;
		new MetadataProperty(resource, RawFile::propertyWhiteLevel, whiteLevel);

		RawUtils::appendBlackLevelFromMaskedArea(_image);
	}

	return _image;
}

int CanonCr2File::thumbnailCount() {
	return 2;
}

QImage CanonCr2File::thumnail(int n) {
	TiffImageReader tiffImageReader(_device);

	if(n == 0)
		return tiffImageReader.readQImage(_tnsOffset, _tnsLength);
	else if(n == 1)
		return tiffImageReader.readQImage(_tnbOffset, _tnbLength);

	return QImage();
}

QString Cr2BayerConverter::name() {
	return QString("Cr2BayerFilter");
}

bool Cr2BayerConverter::accepts(ImageFormat inputFormat, ImageFormat outputFormat) {
	return inputFormat.type() == CanonCr2Image::type() && outputFormat.type() == BayerImage16::type();
}

bool Cr2BayerConverter::isParamterUsed(QString key) {
	Q_UNUSED(key);

	return false;
}

bool Cr2BayerConverter::isPropertyUsed(MetadataProperty* property) {
	Q_UNUSED(property);

	return false;
}

ImageFormat Cr2BayerConverter::prepare(MetadataResource* metadata, ImageFormat inputFormat, ImageFormat outputFormat) {
	Q_UNUSED(metadata);

	if(accepts(inputFormat, outputFormat))
		return ImageFormat(outputFormat.type(), inputFormat.width(), inputFormat.height());
	else
		return ImageFormat();
}

Image* Cr2BayerConverter::filter(Image* input, ImageFormat outputFormat) {
	if(!accepts(input->format(), outputFormat))
		return 0;

	CanonCr2Image* canonCr2Image = qobject_cast<CanonCr2Image*>(input);

	int width = canonCr2Image->width();
	int height = canonCr2Image->height();

	MetadataResource* metadataResource = canonCr2Image->metadata();
	QList<int> sensorPlanes = MetadataQuery::typedValues<int>(metadataResource, CanonCr2::propertySensorPlanes);

	int strideR = sensorPlanes.at(0);
	int strideA = sensorPlanes.at(1);
	int strideB = sensorPlanes.at(2);

	BayerImage16* output = new BayerImage16(width, height);
	output->setMetadata(canonCr2Image->metadata());
	output->setPrecision(canonCr2Image->precision());

	quint16* inputData = (quint16*)canonCr2Image->data();

#if 0
	quint16* outputData = output->data();

	memcpy(outputData, inputData, width*height*2);
#else
	quint16* outputData0 = 0;
	quint16* outputData1 = 0;

	for(int r=0; r<strideR; r++) {
		outputData0 = (quint16*)output->data() + r*strideA;
		outputData1 = outputData0 + width;

		for(int i=0; i<height/2; i++) {
			memcpy(outputData0, inputData, strideA*2);
			inputData += strideA;
			outputData0 += width*2;

			memcpy(outputData1, inputData, strideA*2);
			inputData += strideA;
			outputData1 += width*2;
		}
	}

	outputData0 = (quint16*)output->data() + strideR*strideA;
	outputData1 = outputData0 + width;

	for(int i=0; i<height/2; i++) {
		memcpy(outputData0, inputData, strideB*2);
		inputData += strideB;
		outputData0 += width*2;

		memcpy(outputData1, inputData, strideB*2);
		inputData += strideB;
		outputData1 += width*2;
	}
#endif
	return output;
}

CanonCr2MetadataCleanupTransform::~CanonCr2MetadataCleanupTransform() {
}

MetadataNode* CanonCr2MetadataCleanupTransform::transform(MetadataNode* source) {
	MetadataResource* resource = source->children().at(0)->toResource();
	resource->setResourceUri(QString());

	// reassign sensor planes property
	MetadataProperty* makerNoteProperty = MetadataQuery::property(source, Exif::propertyMakerNote);
	MetadataProperty* sensorPlanesProperty = MetadataQuery::property(source, CanonCr2::propertySensorPlanes);

	if(makerNoteProperty == 0 || sensorPlanesProperty == 0) {
		qDebug("makernote or sensorplanes property missing");
		return 0;
	}

	sensorPlanesProperty->setParent(makerNoteProperty);

	// remove other ifds
	foreach(MetadataNode* node, source->children()) {
		if(node != resource)
			delete node;
	}

	// remove needless properties
	delete MetadataQuery::property(resource, Tiff::propertyBitsPerSample);
	delete MetadataQuery::property(resource, Tiff::propertyCompression);

	QString make = MetadataQuery::typedValue<QString>(resource, Tiff::propertyMake);
	QString model = MetadataQuery::typedValue<QString>(resource, Tiff::propertyModel);

	RawUtils::appendCameraDefinition(resource, model);

	// body id
	new MetadataProperty(resource, RawFile::propertyBodyId, MetadataQuery::value(resource, CanonCr2::propertyBodyInfo));

	// cfa pattern dim
	MetadataProperty* cfaPatternDim = new MetadataProperty(resource, RawFile::propertyCfaPatternDim);
	cfaPatternDim->appendValue(2);
	cfaPatternDim->appendValue(2);

	// cfa pattern
	int cfaShift = 0;

	QList<quint32> sensorInfo = MetadataQuery::typedValues<quint32>(resource, CanonCr2::propertySensorInfo);

	if(!sensorInfo.isEmpty())
		cfaShift = (sensorInfo.at(5) | (sensorInfo.at(6)<<1)) & 3;

	QList<int> defaultCfaPattern;
	defaultCfaPattern << Image::ColorGreen << Image::ColorBlue << Image::ColorRed << Image::ColorGreen;

	MetadataProperty* cfaPattern = new MetadataProperty(resource, RawFile::propertyCfaPattern);

	for(int i=0; i<4; i++)
		cfaPattern->appendValue((int)defaultCfaPattern.at((i+cfaShift)&3));

	// sensor crop
	QList<int> cropValues = MetadataQuery::typedValues<int>(resource, CanonCr2::propertySensorInfo);
	MetadataProperty* cropProperty = new MetadataProperty(resource, RawFile::propertySensorCrop);
	cropProperty->appendValue(cropValues.at(5));
	cropProperty->appendValue(cropValues.at(6));
	cropProperty->appendValue(cropValues.at(7)-cropValues.at(5)+1);
	cropProperty->appendValue(cropValues.at(8)-cropValues.at(6)+1);

	// shot infos
	//QList<QVariant> shotInfoValues = MetadataQuery::values(resource, CanonCr2::propertyShotInfo);

	// white balance
	QList<double> colorBalanceValues = MetadataQuery::typedValues<double>(resource, CanonCr2::propertyColorBalance);
	int whiteBalanceOffset = -1;
	double whiteBalanceNeutral[3] = { 1.0, 1.0, 1.0 };

	QList<double> defaultWhiteBalanceNeutral5200KValues = MetadataQuery::typedValues<double>(resource, CanonCr2::propertyDefaultWhiteBalanceNeutral5200K);
	int whiteBalanceNeutral5200KOffset = -1;
	double cameraCalibration[3] = { 1.0, 1.0, 1.0 };

	switch(colorBalanceValues.size()) {
		case 582:
			whiteBalanceOffset = 25;
			whiteBalanceNeutral5200KOffset = 35;
			break;

		case 653:
			whiteBalanceOffset = 24;
			whiteBalanceNeutral5200KOffset = 39;
			break;

		case 674:
		case 692:
		case 702:
		case 1227:
			whiteBalanceOffset = 63;
			whiteBalanceNeutral5200KOffset = 87;
			break;

		case 796:
			whiteBalanceOffset = 63;
			whiteBalanceNeutral5200KOffset = 78;
			break;

		case 1250:
			whiteBalanceOffset = 63;
			whiteBalanceNeutral5200KOffset = 83;
			break;
	}

	if(whiteBalanceOffset > 0) {
		double green = (colorBalanceValues.at(whiteBalanceOffset+1) + colorBalanceValues.at(whiteBalanceOffset+2)) / 2.0;

		whiteBalanceNeutral[0] = green / colorBalanceValues.at(whiteBalanceOffset);
		whiteBalanceNeutral[2] = green / colorBalanceValues.at(whiteBalanceOffset+3);
	}

	MetadataProperty* whiteBalanceNeutralProperty = new MetadataProperty(resource, RawFile::propertyWhiteBalanceNeutral);
	whiteBalanceNeutralProperty->appendValue(whiteBalanceNeutral[0]);
	whiteBalanceNeutralProperty->appendValue(whiteBalanceNeutral[1]);
	whiteBalanceNeutralProperty->appendValue(whiteBalanceNeutral[2]);

	if(!defaultWhiteBalanceNeutral5200KValues.isEmpty()) {
		cameraCalibration[0] = defaultWhiteBalanceNeutral5200KValues.at(0) / colorBalanceValues.at(whiteBalanceNeutral5200KOffset);
		cameraCalibration[1] = defaultWhiteBalanceNeutral5200KValues.at(1) / (colorBalanceValues.at(whiteBalanceNeutral5200KOffset+1) + colorBalanceValues.at(whiteBalanceNeutral5200KOffset+2)) * 2;
		cameraCalibration[2] = defaultWhiteBalanceNeutral5200KValues.at(2) / colorBalanceValues.at(whiteBalanceNeutral5200KOffset+3);
	}

	MetadataProperty* cameraCalibration1Property = new MetadataProperty(resource, RawFile::propertyCameraCalibration1);
	cameraCalibration1Property->appendValue(cameraCalibration[0]);
	cameraCalibration1Property->appendValue(cameraCalibration[1]);
	cameraCalibration1Property->appendValue(cameraCalibration[2]);

	MetadataProperty* cameraCalibration2Property = new MetadataProperty(resource, RawFile::propertyCameraCalibration2);
	cameraCalibration2Property->appendValue(cameraCalibration[0]);
	cameraCalibration2Property->appendValue(cameraCalibration[1]);
	cameraCalibration2Property->appendValue(cameraCalibration[2]);

	// default image size
	int defaultWidth = MetadataQuery::typedValue<int>(resource, Exif::propertyPixelXDimension);
	int defaultHeight = MetadataQuery::typedValue<int>(resource, Exif::propertyPixelYDimension);
	new MetadataProperty(resource, RawFile::propertyDefaultSize, QSize(defaultWidth, defaultHeight));

	// image filter settings default
	MetadataProperty* settingsDefault = RawUtils::appendSettingsDefault(resource);

	new MetadataProperty(settingsDefault, RawExposure::propertyWhiteBalance, "asShot");

	double temperature, tint;
	TemperatureUtils::fromxy(RawUtils::cameraNeutralWhiteBalance(resource), &temperature, &tint);

	new MetadataProperty(settingsDefault, RawExposure::propertyTemperature, (int)temperature);
	new MetadataProperty(settingsDefault, RawExposure::propertyTint, (int)tint);

	RawUtils::appendOrientationFromTiff(resource);

	// image filter settings current
	RawUtils::createCurrentSettings(resource);

	return source;
}
