/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "DNG.h"

#include <QBuffer>

#include <MetadataRdf.h>
#include <QtUtils.h>

QString DNG::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/dng/1.0/";
TiffMetadataQName DNG::propertyDNGVersion = TiffMetadataQName(DNG::namespaceUri, "DNGVersion", 0xc612, Tiff::TypeByte);
TiffMetadataQName DNG::propertyDNGBackwardVersion = TiffMetadataQName(DNG::namespaceUri, "DNGBackwardVersion", 0xc613, Tiff::TypeByte);
TiffMetadataQName DNG::propertyUniqueCameraModel = TiffMetadataQName(DNG::namespaceUri, "UniqueCameraModel", 0xc614, Tiff::TypeAscii);
TiffMetadataQName DNG::propertyLocalizedCameraModel = TiffMetadataQName(DNG::namespaceUri, "LocalizedCameraModel", 0xc615, Tiff::TypeAscii);
TiffMetadataQName DNG::propertyCFAPlaneColor = TiffMetadataQName(DNG::namespaceUri, "CFAPlaneColor", 0xc616, Tiff::TypeByte);
TiffMetadataQName DNG::propertyCFALayout = TiffMetadataQName(DNG::namespaceUri, "CFALayout", 0xc617, Tiff::TypeShort);
TiffMetadataQName DNG::propertyBlackLevelRepeatDim = TiffMetadataQName(DNG::namespaceUri, "BlackLevelRepeatDim", 0xc619, Tiff::TypeShort);
TiffMetadataQName DNG::propertyBlackLevel = TiffMetadataQName(DNG::namespaceUri, "BlackLevel", 0xc61a, Tiff::TypeRational);
TiffMetadataQName DNG::propertyBlackLevelDeltaH = TiffMetadataQName(DNG::namespaceUri, "BlackLevelDeltaH", 0xc61b, Tiff::TypeSRational);
TiffMetadataQName DNG::propertyBlackLevelDeltaV = TiffMetadataQName(DNG::namespaceUri, "BlackLevelDeltaV", 0xc61c, Tiff::TypeSRational);
TiffMetadataQName DNG::propertyWhiteLevel = TiffMetadataQName(DNG::namespaceUri, "WhiteLevel", 0xc61d, Tiff::TypeLong);
TiffMetadataQName DNG::propertyDefaultScale = TiffMetadataQName(DNG::namespaceUri, "DefaultScale", 0xc61e, Tiff::TypeRational);
TiffMetadataQName DNG::propertyDefaultCropOrigin = TiffMetadataQName(DNG::namespaceUri, "DefaultCropOrigin", 0xc61f, Tiff::TypeRational);
TiffMetadataQName DNG::propertyDefaultCropSize = TiffMetadataQName(DNG::namespaceUri, "DefaultCropSize", 0xc620, Tiff::TypeRational);
TiffMetadataQName DNG::propertyColorMatrix1 = TiffMetadataQName(DNG::namespaceUri, "ColorMatrix1", 0xc621, Tiff::TypeSRational);
TiffMetadataQName DNG::propertyColorMatrix2 = TiffMetadataQName(DNG::namespaceUri, "ColorMatrix2", 0xc622, Tiff::TypeSRational);
TiffMetadataQName DNG::propertyCameraCalibration1 = TiffMetadataQName(DNG::namespaceUri, "CameraCalibration1", 0xc623, Tiff::TypeSRational);
TiffMetadataQName DNG::propertyCameraCalibration2 = TiffMetadataQName(DNG::namespaceUri, "CameraCalibration2", 0xc624, Tiff::TypeSRational);
TiffMetadataQName DNG::propertyAnalogBalance = TiffMetadataQName(DNG::namespaceUri, "AnalogBalance", 0xc627, Tiff::TypeRational);
TiffMetadataQName DNG::propertyAsShotNeutral = TiffMetadataQName(DNG::namespaceUri, "AsShotNeutral", 0xc628, Tiff::TypeRational);
TiffMetadataQName DNG::propertyBaselineExposure = TiffMetadataQName(DNG::namespaceUri, "BaselineExposure", 0xc62a, Tiff::TypeSRational);
TiffMetadataQName DNG::propertyBaselineNoise = TiffMetadataQName(DNG::namespaceUri, "BaselineNoise", 0xc62b, Tiff::TypeRational);
TiffMetadataQName DNG::propertyBaselineSharpness = TiffMetadataQName(DNG::namespaceUri, "BaselineSharpness", 0xc62c, Tiff::TypeRational);
TiffMetadataQName DNG::propertyBayerGreenSplit = TiffMetadataQName(DNG::namespaceUri, "BayerGreenSplit", 0xc62d, Tiff::TypeLong);
TiffMetadataQName DNG::propertyLinearResponseLimit = TiffMetadataQName(DNG::namespaceUri, "LinearResponseLimit", 0xc62e, Tiff::TypeRational);
TiffMetadataQName DNG::propertyCameraSerialNumber = TiffMetadataQName(DNG::namespaceUri, "CameraSerialNumber", 0xc62f, Tiff::TypeAscii);
TiffMetadataQName DNG::propertyLensInfo = TiffMetadataQName(DNG::namespaceUri, "LensInfo", 0xc630, Tiff::TypeRational);
TiffMetadataQName DNG::propertyAntiAliasStrength = TiffMetadataQName(DNG::namespaceUri, "AntiAliasStrength", 0xc632, Tiff::TypeRational);
TiffMetadataQName DNG::propertyShadowScale = TiffMetadataQName(DNG::namespaceUri, "ShadowScale", 0xc633, Tiff::TypeRational);
TiffMetadataQName DNG::propertyDNGPrivateData = TiffMetadataQName(DNG::namespaceUri, "DNGPrivateData", 0xc634, Tiff::TypeByte);
TiffMetadataQName DNG::propertyCalibrationIlluminant1 = TiffMetadataQName(DNG::namespaceUri, "CalibrationIlluminant1", 0xc65a, Tiff::TypeShort);
TiffMetadataQName DNG::propertyCalibrationIlluminant2 = TiffMetadataQName(DNG::namespaceUri, "CalibrationIlluminant2", 0xc65b, Tiff::TypeShort);
TiffMetadataQName DNG::propertyBestQualityScale = TiffMetadataQName(DNG::namespaceUri, "BestQualityScale", 0xc65c, Tiff::TypeRational);
TiffMetadataQName DNG::propertyRawDataUniqueID = TiffMetadataQName(DNG::namespaceUri, "RawDataUniqueID", 0xc65d, Tiff::TypeByte);
TiffMetadataQName DNG::propertyOriginalRawFileName = TiffMetadataQName(DNG::namespaceUri, "OriginalRawFileName", 0xc68b, Tiff::TypeByte);
TiffMetadataQName DNG::propertyOriginalRawFileData = TiffMetadataQName(DNG::namespaceUri, "OriginalRawFileData", 0xc68c, Tiff::TypeUndefined);
TiffMetadataQName DNG::propertyActiveArea = TiffMetadataQName(DNG::namespaceUri, "ActiveArea", 0xc68d, Tiff::TypeLong);
TiffMetadataQName DNG::propertyMaskedAreas = TiffMetadataQName(DNG::namespaceUri, "MaskedAreas", 0xc68e, Tiff::TypeLong);
QList<TiffMetadataQName> DNG::_tiffMetadataQNames = QList<TiffMetadataQName>();

QList<TiffMetadataQName> DNG::tiffMetadataQNames() {
	if(_tiffMetadataQNames.isEmpty()) {
		_tiffMetadataQNames.append(propertyDNGVersion);
		_tiffMetadataQNames.append(propertyDNGBackwardVersion);
		_tiffMetadataQNames.append(propertyUniqueCameraModel);
		_tiffMetadataQNames.append(propertyLocalizedCameraModel);
		_tiffMetadataQNames.append(propertyCFAPlaneColor);
		_tiffMetadataQNames.append(propertyCFALayout);
		_tiffMetadataQNames.append(propertyBlackLevelRepeatDim);
		_tiffMetadataQNames.append(propertyBlackLevel);
		_tiffMetadataQNames.append(propertyBlackLevelDeltaH);
		_tiffMetadataQNames.append(propertyBlackLevelDeltaV);
		_tiffMetadataQNames.append(propertyWhiteLevel);
		_tiffMetadataQNames.append(propertyDefaultScale);
		_tiffMetadataQNames.append(propertyDefaultCropOrigin);
		_tiffMetadataQNames.append(propertyDefaultCropSize);
		_tiffMetadataQNames.append(propertyColorMatrix1);
		_tiffMetadataQNames.append(propertyColorMatrix2);
		_tiffMetadataQNames.append(propertyCameraCalibration1);
		_tiffMetadataQNames.append(propertyCameraCalibration2);
		_tiffMetadataQNames.append(propertyAnalogBalance);
		_tiffMetadataQNames.append(propertyAsShotNeutral);
		_tiffMetadataQNames.append(propertyBaselineExposure);
		_tiffMetadataQNames.append(propertyBaselineNoise);
		_tiffMetadataQNames.append(propertyBaselineSharpness);
		_tiffMetadataQNames.append(propertyBayerGreenSplit);
		_tiffMetadataQNames.append(propertyLinearResponseLimit);
		_tiffMetadataQNames.append(propertyCameraSerialNumber);
		_tiffMetadataQNames.append(propertyLensInfo);
		_tiffMetadataQNames.append(propertyAntiAliasStrength);
		_tiffMetadataQNames.append(propertyShadowScale);
		_tiffMetadataQNames.append(propertyDNGPrivateData);
		_tiffMetadataQNames.append(propertyCalibrationIlluminant1);
		_tiffMetadataQNames.append(propertyCalibrationIlluminant2);
		_tiffMetadataQNames.append(propertyBestQualityScale);
		_tiffMetadataQNames.append(propertyRawDataUniqueID);
		_tiffMetadataQNames.append(propertyOriginalRawFileName);
		_tiffMetadataQNames.append(propertyOriginalRawFileData);
		_tiffMetadataQNames.append(propertyActiveArea);
		_tiffMetadataQNames.append(propertyMaskedAreas);
	}

	return _tiffMetadataQNames;
}

QMap<quint16, MetadataQName> DNGReader::_dngTagMetadataQNameMap = QMap<quint16, MetadataQName>();

QMap<quint16, MetadataQName> DNGReader::dngTagMetadataQNameMap() {
	if(_dngTagMetadataQNameMap.isEmpty()) {
		foreach(TiffMetadataQName qName, DNG::tiffMetadataQNames()) {
			_dngTagMetadataQNameMap.insert(qName.tag(), qName);
		}
	}

	return _dngTagMetadataQNameMap;
}

DNGReader::DNGReader(QIODevice* device) : TiffReader(device) {
}

MetadataQName DNGReader::tagQName(quint16 tag) {
	if(dngTagMetadataQNameMap().contains(tag))
		return dngTagMetadataQNameMap().value(tag);
	else
		return TiffReader::tagQName(tag);
}

QVariant DNGReader::tagValue(quint16 tag, QVariant value) {
	QByteArray d(value.toByteArray());

	switch(tag) {
		case 0xc612:
		case 0xc613:
			return QVariant(QString("%1.%2.%3.%4").arg((int)d.at(0)).arg((int)d.at(1)).arg((int)d.at(2)).arg((int)d.at(3)));

		default:
			return TiffReader::tagValue(tag, value);
	}
}

DNGFile::DNGFile(QIODevice* device) : RawFile() {
	_device = device;
	_dngReader = new DNGReader(_device);
}

DNGFile::~DNGFile() {
	delete _dngReader;
}

MetadataDocument* DNGFile::readMetadata() {
	if(_metadata == 0) {
		DNGMetadataCleanupTransform transform;

		MetadataNode* metadata = _dngReader->read();
#ifdef QT_DEBUG
		MetadataRdfWriter::writeFile(metadata, "f:\\test.xml");
#endif
		QMap<int, int> tnSizeMap;
		QList<MetadataProperty*> newSubfileTypes = MetadataQuery::properties(metadata, Tiff::propertyNewSubfileType);
		int count = 0;
		for(int i=0; i<newSubfileTypes.size(); i++) {
			MetadataProperty* newSubfileType = newSubfileTypes.at(i);

			if(newSubfileType->values().at(0).toInt() != 0) {
				MetadataNode* parent = newSubfileType->parent();
				tnSizeMap.insert(MetadataQuery::typedValue<int>(parent, Tiff::propertyImageWidth), count++);
				_tnOffsets.append(MetadataQuery::typedValue<quint32>(parent, Tiff::propertyStripOffsets));
				_tnLengths.append(MetadataQuery::typedValue<quint32>(parent, Tiff::propertyStripByteCounts));
			}
		}

		QList<int> tnSizes = tnSizeMap.keys();
		qSort(tnSizes);

		foreach(int tnSize, tnSizes) {
			_tnIndex.append(tnSizeMap.value(tnSize));
		}

		_metadata = MetadataNode::toDocument(transform.transform(metadata));
	}

	return _metadata;
}

Image* DNGFile::readImage() {
	if(_image == 0) {
		readMetadata();

		MetadataResource* resource = MetadataQuery::resource(_metadata, "*");

		TiffImageReader tiffImageReader(_device);
		_image = tiffImageReader.read(resource, 0, ImageFormat(BayerImage16::type()));

		_image->setMetadata(resource);
	}

	return _image;
}

int DNGFile::thumbnailCount() {
	return _tnIndex.size();
}

QImage DNGFile::thumnail(int n) {
	if(_tnIndex.size() <= n)
		return QImage();

	TiffImageReader tiffImageReader(_device);

	return tiffImageReader.readQImage(_tnOffsets.at(_tnIndex.at(n)), _tnLengths.at(_tnIndex.at(n)));
}

DNGMetadataCleanupTransform::~DNGMetadataCleanupTransform() {
}

MetadataNode* DNGMetadataCleanupTransform::transform(MetadataNode* source) {
	MetadataResource* resource = MetadataQuery::resource(source, "*");
	resource->setResourceUri(QString());

	// remove other subifds
	foreach(MetadataProperty* newSubfileType, MetadataQuery::properties(resource, Tiff::propertyNewSubfileType)) {
		if(newSubfileType->values().at(0).toInt() != 0 && newSubfileType->parent() != resource)
			delete newSubfileType->parent();
	}

	// reassign properties from raw sub ifd
	MetadataNode* subIfd = MetadataNode::parent(MetadataQuery::property(resource, Tiff::propertyNewSubfileType, 0));

	if(subIfd == 0) {
		qDebug("now high quality subfile type found");
		return 0;
	}

	foreach(MetadataNode* node, subIfd->children()) {
		MetadataProperty* property = node->toProperty();

		// remove other properties of the same type
		foreach(MetadataProperty* oldProperty, MetadataQuery::properties(resource, property->qName())) {
			if(oldProperty != property)
				delete oldProperty;
		}

		property->setParent(resource);
	}

	delete subIfd;

	// body id
	new MetadataProperty(resource, RawFile::propertyBodyId, MetadataQuery::value(resource, DNG::propertyCameraSerialNumber));

	// sensor crop
	QList<quint32> activeArea = MetadataQuery::typedValues<quint32>(resource, DNG::propertyActiveArea);
	QList<Rational> cropOrigin = MetadataQuery::typedValues<Rational>(resource, DNG::propertyDefaultCropOrigin);
	QList<Rational> cropSize = MetadataQuery::typedValues<Rational>(resource, DNG::propertyDefaultCropSize);
	quint32 imageWidth = MetadataQuery::typedValue<quint32>(resource, Tiff::propertyImageWidth);
	quint32 imageLength = MetadataQuery::typedValue<quint32>(resource, Tiff::propertyImageLength);

	int cropX = 0, cropY = 0, cropWidth = imageWidth, cropHeight = imageLength;

	if(!activeArea.empty()) {
		cropX = activeArea.at(1);
		cropY = activeArea.at(0);
		cropWidth = activeArea.at(3);
		cropHeight = activeArea.at(2);
	}

	if(!cropOrigin.empty()) {
		cropX += (int)cropOrigin.at(0).toDouble();
		cropY += (int)cropOrigin.at(1).toDouble();
	}

	if(!cropSize.empty()) {
		cropWidth = (int)cropSize.at(0).toDouble();
		cropHeight = (int)cropSize.at(1).toDouble();
	}

	MetadataProperty* cropProperty = new MetadataProperty(resource, RawFile::propertySensorCrop);
	cropProperty->appendValue(cropX);
	cropProperty->appendValue(cropY);
	cropProperty->appendValue(cropWidth);
	cropProperty->appendValue(cropHeight);

	// cfa pattern dim
	QList<quint32> cfaPatternDimDNG = MetadataQuery::typedValues<quint32>(resource, Tiff::propertyCFARepeatPatternDim);
	if(!cfaPatternDimDNG.isEmpty()) {
		MetadataProperty* cfaPatternDim = new MetadataProperty(resource, RawFile::propertyCfaPatternDim);

		cfaPatternDim->appendValue(cfaPatternDimDNG.at(0));
		cfaPatternDim->appendValue(cfaPatternDimDNG.at(1));
	}

	// cfa pattern
	QByteArray cfaPatternDNG = MetadataQuery::typedValue<QByteArray>(resource, Tiff::propertyCFAPattern);
	if(!cfaPatternDNG.isEmpty()) {
		int cfaShift = 0;

		if(!activeArea.isEmpty())
			cfaShift = (activeArea.at(1)%cfaPatternDimDNG.at(0)) | ((activeArea.at(0)%cfaPatternDimDNG.at(1))*cfaPatternDimDNG.at(0));

		MetadataProperty* cfaPattern = new MetadataProperty(resource, RawFile::propertyCfaPattern);

		for(int i=0; i<cfaPatternDNG.size(); i++)
			cfaPattern->appendValue((int)cfaPatternDNG.at((i+cfaShift)%cfaPatternDNG.size()));
	}

	// white level
	new MetadataProperty(resource, RawFile::propertyWhiteLevel, MetadataQuery::value(resource, DNG::propertyWhiteLevel));

	// black level
	// TODO: black level row and column based
	double blackLevel = 0.0;

	QList<Rational> blackLevels = MetadataQuery::typedValues<Rational>(resource, DNG::propertyBlackLevel);
	if(!blackLevels.isEmpty())
		blackLevel = blackLevels.at(0).toDouble();

	QList<Rational> blackLevelDeltaH = MetadataQuery::typedValues<Rational>(resource, DNG::propertyBlackLevelDeltaH);
	double blackLevelDeltaHSum = 0.0;
	if(!blackLevelDeltaH.isEmpty()) {
		foreach(Rational item, blackLevelDeltaH) {
			blackLevelDeltaHSum += item.toDouble();
		}
		blackLevelDeltaHSum /= (double)blackLevelDeltaH.size();
	}

	QList<Rational> blackLevelDeltaV = MetadataQuery::typedValues<Rational>(resource, DNG::propertyBlackLevelDeltaV);
	double blackLevelDeltaVSum = 0.0;
	if(!blackLevelDeltaV.isEmpty()) {
		foreach(Rational item, blackLevelDeltaV) {
			blackLevelDeltaVSum += item.toDouble();
		}
		blackLevelDeltaVSum /= (double)blackLevelDeltaV.size();
	}

	new MetadataProperty(resource, RawFile::propertyBlackLevel, blackLevel + blackLevelDeltaHSum + blackLevelDeltaVSum);

	// white balance neutral
	MetadataProperty* whiteBalanceNeutralProperty = new MetadataProperty(resource, RawFile::propertyWhiteBalanceNeutral);
	foreach(Rational value, MetadataQuery::typedValues<Rational>(resource, DNG::propertyAsShotNeutral)) {
		whiteBalanceNeutralProperty->appendValue(value.toDouble());
	}

	// calibration illuminant
	QVariant calibrationIlluminant1Value = MetadataQuery::value(resource, DNG::propertyCalibrationIlluminant1);
	if(!calibrationIlluminant1Value.isNull())
		new MetadataProperty(resource, RawFile::propertyCalibrationIlluminant1, Tiff::lightSourceTemperature(calibrationIlluminant1Value.value<quint16>()));

	QVariant calibrationIlluminant2Value = MetadataQuery::value(resource, DNG::propertyCalibrationIlluminant2);
	if(!calibrationIlluminant2Value.isNull())
		new MetadataProperty(resource, RawFile::propertyCalibrationIlluminant2, Tiff::lightSourceTemperature(calibrationIlluminant2Value.value<quint16>()));

	// color matrix
	MetadataProperty* colorMatrix1Property = new MetadataProperty(resource, RawFile::propertyColorMatrix1);
	foreach(Rational value, MetadataQuery::typedValues<Rational>(resource, DNG::propertyColorMatrix1)) {
		colorMatrix1Property->appendValue(value.toDouble());
	}

	MetadataProperty* colorMatrix2Property = new MetadataProperty(resource, RawFile::propertyColorMatrix2);
	foreach(Rational value, MetadataQuery::typedValues<Rational>(resource, DNG::propertyColorMatrix2)) {
		colorMatrix2Property->appendValue(value.toDouble());
	}

	// camera calibration
	MetadataProperty* cameraCalibration1Property = new MetadataProperty(resource, RawFile::propertyCameraCalibration1);
	foreach(Rational value, MetadataQuery::typedValues<Rational>(resource, DNG::propertyCameraCalibration1)) {
		cameraCalibration1Property->appendValue(value.toDouble());
	}

	MetadataProperty* cameraCalibration2Property = new MetadataProperty(resource, RawFile::propertyCameraCalibration2);
	foreach(Rational value, MetadataQuery::typedValues<Rational>(resource, DNG::propertyCameraCalibration2)) {
		cameraCalibration2Property->appendValue(value.toDouble());
	}

	// default image size
	new MetadataProperty(resource, RawFile::propertyDefaultSize, QSize(cropWidth, cropHeight));

	// image filter settings default
	MetadataProperty* settingsDefault = RawUtils::appendSettingsDefault(resource);

	new MetadataProperty(settingsDefault, RawExposure::propertyWhiteBalance, "asShot");

	double temperature, tint;
	TemperatureUtils::fromxy(RawUtils::cameraNeutralWhiteBalance(resource), &temperature, &tint);

	new MetadataProperty(settingsDefault, RawExposure::propertyTemperature, (int)temperature);
	new MetadataProperty(settingsDefault, RawExposure::propertyTint, (int)tint);

	RawUtils::appendOrientationFromTiff(resource);

	// image filter settings current
	RawUtils::createCurrentSettings(resource);

	return source;
}
