/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "LibRawFile.h"

#include <QDebug>

#include <libraw.h>

#include <formats/Tiff.h>


LibRawFile::LibRawFile(QIODevice* device) : RawFile() {
	QString fileName = QtUtils::fileName(device);

	if(fileName.isEmpty())
		return;

	_libRaw.imgdata.params.filtering_mode = LIBRAW_FILTERING_NONE;

	if(_libRaw.open_file(fileName.toAscii().data()) != LIBRAW_SUCCESS)
		return;
}

LibRawFile::~LibRawFile() {
	_libRaw.recycle();
}

MetadataDocument* LibRawFile::readMetadata() {
	if(_metadata != 0)
		return _metadata;

	MetadataDocument* document = new MetadataDocument();

	MetadataResource* resource = new MetadataResource(document);

	new MetadataProperty(resource, Tiff::propertyMake, _libRaw.imgdata.idata.make);

	new MetadataProperty(resource, Tiff::propertyModel, _libRaw.imgdata.idata.model);

	// cfa pattern dim
	int cfaPatternWidth = 2;
	int cfaPatternHeight = 8;

	quint16* tempFilters16 = (quint16*)&_libRaw.imgdata.idata.filters;

	if(tempFilters16[0] == tempFilters16[1]) {
		cfaPatternHeight = 4;

		quint8* tempFilters8 = (quint8*)&_libRaw.imgdata.idata.filters;

		if(tempFilters8[0] == tempFilters8[1])
			cfaPatternHeight = 2;
	}

	int cfaPatternSize = cfaPatternWidth * cfaPatternHeight;

	MetadataProperty* cfaPatternDim = new MetadataProperty(resource, RawFile::propertyCfaPatternDim);
	cfaPatternDim->appendValue(cfaPatternWidth);
	cfaPatternDim->appendValue(cfaPatternHeight);

	qDebug() << "cfaPatternDim" << cfaPatternWidth << cfaPatternHeight;

	// cfa pattern
	int cfaPatternShift = (_libRaw.imgdata.sizes.left_margin*2 + _libRaw.imgdata.sizes.top_margin*cfaPatternWidth) % cfaPatternSize;
	MetadataProperty* cfaPattern = new MetadataProperty(resource, RawFile::propertyCfaPattern);

	for(int i=0; i<cfaPatternSize; i++) {
		int index = (i + cfaPatternShift) % cfaPatternSize;
		int color = (_libRaw.imgdata.idata.filters>>(index<<1))&3;
		qDebug() << index << color;
		switch(color) {
			case 0:
				cfaPattern->appendValue(Image::ColorRed);
				break;

			case 1:
				cfaPattern->appendValue(Image::ColorGreen);
				break;

			case 2:
				cfaPattern->appendValue(Image::ColorBlue);
				break;

			case 3:
				cfaPattern->appendValue(Image::ColorGreen);
				break;
		}
	}

	// sensor crop
	MetadataProperty* sensorCrop = new MetadataProperty(resource, RawFile::propertySensorCrop);
	sensorCrop->appendValue(_libRaw.imgdata.sizes.left_margin);
	sensorCrop->appendValue(_libRaw.imgdata.sizes.top_margin);
	sensorCrop->appendValue(_libRaw.imgdata.sizes.width);
	sensorCrop->appendValue(_libRaw.imgdata.sizes.height);

	// color matrix 1
	MetadataProperty* colorMatrix1 = new MetadataProperty(resource, RawFile::propertyColorMatrix1);
	for(int y=0; y<3; y++) {
		for(int x=0; x<3; x++)
			colorMatrix1->appendValue(_libRaw.imgdata.color.cam_xyz[y][x]);
	}

	// white balance neutral
	double wbnGreenTemp = (_libRaw.imgdata.color.cam_mul[1] + _libRaw.imgdata.color.cam_mul[3]) / 2.0;
	double wbnRed = wbnGreenTemp / _libRaw.imgdata.color.cam_mul[0];
	double wbnBlue = wbnGreenTemp / _libRaw.imgdata.color.cam_mul[2];

	MetadataProperty* whiteBalanceNeutralProperty = new MetadataProperty(resource, RawFile::propertyWhiteBalanceNeutral);
	whiteBalanceNeutralProperty->appendValue(wbnRed);
	whiteBalanceNeutralProperty->appendValue(1.0);
	whiteBalanceNeutralProperty->appendValue(wbnBlue);

	// default image size
	new MetadataProperty(resource, RawFile::propertyDefaultSize, QSize(_libRaw.imgdata.sizes.width, _libRaw.imgdata.sizes.height));

	qDebug() << "size:" << _libRaw.imgdata.sizes.width << _libRaw.imgdata.sizes.height;

	// white level
	new MetadataProperty(resource, RawFile::propertyWhiteLevel, _libRaw.imgdata.color.maximum);

	// black level
	new MetadataProperty(resource, RawFile::propertyBlackLevel, _libRaw.imgdata.color.black);

	// image filter settings default
	MetadataProperty* settingsDefault = RawUtils::appendSettingsDefault(resource);

	new MetadataProperty(settingsDefault, RawExposure::propertyWhiteBalance, QVariant("asShot"));

	double temperature, tint;
	TemperatureUtils::fromxy(RawUtils::cameraNeutralWhiteBalance(resource), &temperature, &tint);

	new MetadataProperty(settingsDefault, RawExposure::propertyTemperature, (int)temperature);
	new MetadataProperty(settingsDefault, RawExposure::propertyTint, (int)tint);

	RawUtils::appendOrientationFromTiff(resource);

	// image filter settings current
	RawUtils::createCurrentSettings(resource);

	_metadata = document;

	return _metadata;
}

Image* LibRawFile::readImage() {
	if(_image == 0) {
		readMetadata();

		if(_libRaw.unpack() != LIBRAW_SUCCESS)
			return 0;

		BayerImage16* bayerImage = new BayerImage16(_libRaw.imgdata.sizes.raw_width, _libRaw.imgdata.sizes.raw_height);

		for(int i=0, y=0; y<_libRaw.imgdata.sizes.height; y++)
			for(int x=0; x<_libRaw.imgdata.sizes.width; x++, i++)
				*(reinterpret_cast<quint16*>(bayerImage->dataOffset(_libRaw.imgdata.sizes.left_margin+x, _libRaw.imgdata.sizes.top_margin+y))) = _libRaw.imgdata.image[i][0];

		bayerImage->setMetadata(MetadataQuery::resource(_metadata, "*"));

		_image = bayerImage;
	}

	return _image;
}

int LibRawFile::thumbnailCount() {
	return 0;
}

QImage LibRawFile::thumnail(int n) {
	return QImage();
}
