/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "Tiff.h"

#include <QBuffer>

#include <QtUtils.h>
#include <RawFile.h>
#include <formats/LosslessJpegDecoder.h>

QString Tiff::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/tiff/1.0/";
TiffMetadataQName Tiff::propertyNewSubfileType = TiffMetadataQName(Tiff::namespaceUri, "NewSubfileType", 0xfe, Tiff::TypeLong);
TiffMetadataQName Tiff::propertySubfileType = TiffMetadataQName(Tiff::namespaceUri, "SubfileType", 0xff, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyImageWidth = TiffMetadataQName(Tiff::namespaceUri, "ImageWidth", 0x100, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyImageLength = TiffMetadataQName(Tiff::namespaceUri, "ImageLength", 0x101, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyBitsPerSample = TiffMetadataQName(Tiff::namespaceUri, "BitsPerSample", 0x102, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyCompression = TiffMetadataQName(Tiff::namespaceUri, "Compression", 0x103, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyPhotometricInterpretation = TiffMetadataQName(Tiff::namespaceUri, "PhotometricInterpretation", 0x106, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyThreshholding = TiffMetadataQName(Tiff::namespaceUri, "Threshholding", 0x107, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyCellWidth = TiffMetadataQName(Tiff::namespaceUri, "CellWidth", 0x108, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyCellLength = TiffMetadataQName(Tiff::namespaceUri, "CellLength", 0x109, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyFillOrder  = TiffMetadataQName(Tiff::namespaceUri, "FillOrder", 0x10a, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyDocumentName = TiffMetadataQName(Tiff::namespaceUri, "DocumentName", 0x10d, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyImageDescription = TiffMetadataQName(Tiff::namespaceUri, "ImageDescription", 0x10e, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyMake = TiffMetadataQName(Tiff::namespaceUri, "Make", 0x10f, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyModel = TiffMetadataQName(Tiff::namespaceUri, "Model", 0x110, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyStripOffsets = TiffMetadataQName(Tiff::namespaceUri, "StripOffsets", 0x111, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyOrientation = TiffMetadataQName(Tiff::namespaceUri, "Orientation", 0x112, Tiff::TypeShort);
TiffMetadataQName Tiff::propertySamplesPerPixel = TiffMetadataQName(Tiff::namespaceUri, "SamplesPerPixel", 0x115, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyRowsPerStrip = TiffMetadataQName(Tiff::namespaceUri, "RowsPerStrip", 0x116, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyStripByteCounts = TiffMetadataQName(Tiff::namespaceUri, "StripByteCounts", 0x117, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyMinSampleValue = TiffMetadataQName(Tiff::namespaceUri, "MinSampleValue", 0x118, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyMaxSampleValue = TiffMetadataQName(Tiff::namespaceUri, "MaxSampleValue", 0x119, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyXResolution = TiffMetadataQName(Tiff::namespaceUri, "XResolution", 0x11a, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyYResolution = TiffMetadataQName(Tiff::namespaceUri, "YResolution", 0x11b, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyPlanarConfiguration = TiffMetadataQName(Tiff::namespaceUri, "PlanarConfiguration", 0x11c, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyPageName = TiffMetadataQName(Tiff::namespaceUri, "PageName", 0x11d, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyXPosition = TiffMetadataQName(Tiff::namespaceUri, "XPosition", 0x11e, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyYPosition = TiffMetadataQName(Tiff::namespaceUri, "YPosition", 0x11f, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyFreeOffsets = TiffMetadataQName(Tiff::namespaceUri, "FreeOffsets", 0x120, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyFreeByteCounts = TiffMetadataQName(Tiff::namespaceUri, "FreeByteCounts", 0x121, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyGrayResponseUnit  = TiffMetadataQName(Tiff::namespaceUri, "GrayResponseUnit", 0x122, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyGrayResponseCurve = TiffMetadataQName(Tiff::namespaceUri, "GrayResponseCurve", 0x123, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyT4Options = TiffMetadataQName(Tiff::namespaceUri, "T4Options", 0x124, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyT6Options = TiffMetadataQName(Tiff::namespaceUri, "T6Options", 0x125, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyResolutionUnit = TiffMetadataQName(Tiff::namespaceUri, "ResolutionUnit", 0x128, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyPageNumber = TiffMetadataQName(Tiff::namespaceUri, "PageNumber", 0x129, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyTransferFunction = TiffMetadataQName(Tiff::namespaceUri, "TransferFunction", 0x12d, Tiff::TypeShort);
TiffMetadataQName Tiff::propertySoftware = TiffMetadataQName(Tiff::namespaceUri, "Software", 0x131, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyDateTime = TiffMetadataQName(Tiff::namespaceUri, "DateTime", 0x132, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyArtist = TiffMetadataQName(Tiff::namespaceUri, "Artist", 0x13b, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyHostComputer = TiffMetadataQName(Tiff::namespaceUri, "HostComputer", 0x13c, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyPredictor = TiffMetadataQName(Tiff::namespaceUri, "Predictor", 0x13d, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyWhitePoint = TiffMetadataQName(Tiff::namespaceUri, "WhitePoint", 0x13e, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyPrimaryChromaticities = TiffMetadataQName(Tiff::namespaceUri, "PrimaryChromaticities", 0x13f, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyColorMap = TiffMetadataQName(Tiff::namespaceUri, "ColorMap", 0x140, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyHalftoneHints = TiffMetadataQName(Tiff::namespaceUri, "HalftoneHints", 0x141, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyTileWidth = TiffMetadataQName(Tiff::namespaceUri, "TileWidth", 0x142, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyTileLength = TiffMetadataQName(Tiff::namespaceUri, "TileLength", 0x143, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyTileOffsets = TiffMetadataQName(Tiff::namespaceUri, "TileOffsets", 0x144, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyTileByteCounts = TiffMetadataQName(Tiff::namespaceUri, "TileByteCounts", 0x145, Tiff::TypeLong);
TiffMetadataQName Tiff::propertySubIFDs = TiffMetadataQName(Tiff::namespaceUri, "SubIFDs", 0x14a, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyInkSet = TiffMetadataQName(Tiff::namespaceUri, "InkSet", 0x14c, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyInkNames = TiffMetadataQName(Tiff::namespaceUri, "InkNames", 0x14d, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyNumberOfInks = TiffMetadataQName(Tiff::namespaceUri, "NumberOfInks", 0x14e, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyDotRange = TiffMetadataQName(Tiff::namespaceUri, "DotRange", 0x150, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyTargetPrinter = TiffMetadataQName(Tiff::namespaceUri, "TargetPrinter", 0x151, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyExtraSamples = TiffMetadataQName(Tiff::namespaceUri, "ExtraSamples", 0x152, Tiff::TypeByte);
TiffMetadataQName Tiff::propertySampleFormat = TiffMetadataQName(Tiff::namespaceUri, "SampleFormat", 0x153, Tiff::TypeShort);
TiffMetadataQName Tiff::propertySMinSampleValue = TiffMetadataQName(Tiff::namespaceUri, "SMinSampleValue", 0x154, Tiff::TypeShort);
TiffMetadataQName Tiff::propertySMaxSampleValue = TiffMetadataQName(Tiff::namespaceUri, "SMaxSampleValue", 0x155, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyTransferRange = TiffMetadataQName(Tiff::namespaceUri, "TransferRange", 0x156, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyJPEGTables = TiffMetadataQName(Tiff::namespaceUri, "JPEGTables", 0x15b, Tiff::TypeUndefined);
TiffMetadataQName Tiff::propertyJPEGProc = TiffMetadataQName(Tiff::namespaceUri, "JPEGProc", 0x200, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyJPEGInterchangeFormat = TiffMetadataQName(Tiff::namespaceUri, "JPEGInterchangeFormat", 0x201, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyJPEGInterchangeFormatLength = TiffMetadataQName(Tiff::namespaceUri, "JPEGInterchangeFormatLength", 0x202, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyJPEGRestartInterval = TiffMetadataQName(Tiff::namespaceUri, "JPEGRestartInterval", 0x203, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyJPEGLosslessPredictors = TiffMetadataQName(Tiff::namespaceUri, "JPEGLosslessPredictors", 0x205, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyJPEGPointTransforms = TiffMetadataQName(Tiff::namespaceUri, "JPEGPointTransforms", 0x206, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyJPEGQTables = TiffMetadataQName(Tiff::namespaceUri, "JPEGQTables", 0x207, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyJPEGDCTables = TiffMetadataQName(Tiff::namespaceUri, "JPEGDCTables", 0x208, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyJPEGACTables = TiffMetadataQName(Tiff::namespaceUri, "JPEGACTables", 0x209, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyYCbCrCoefficients = TiffMetadataQName(Tiff::namespaceUri, "YCbCrCoefficients", 0x211, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyYCbCrSubSampling = TiffMetadataQName(Tiff::namespaceUri, "YCbCrSubSampling", 0x212, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyYCbCrPositioning = TiffMetadataQName(Tiff::namespaceUri, "YCbCrPositioning", 0x213, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyReferenceBlackWhite = TiffMetadataQName(Tiff::namespaceUri, "ReferenceBlackWhite", 0x214, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyXMP = TiffMetadataQName(Tiff::namespaceUri, "XMP", 0x2bc, Tiff::TypeByte);
TiffMetadataQName Tiff::propertyCFARepeatPatternDim = TiffMetadataQName(Tiff::namespaceUri, "CFARepeatPatternDim", 0x828d, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyCFAPattern = TiffMetadataQName(Tiff::namespaceUri, "CFAPattern", 0x828e, Tiff::TypeByte);
TiffMetadataQName Tiff::propertyBatteryLevel = TiffMetadataQName(Tiff::namespaceUri, "BatteryLevel", 0x828f, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyCopyright = TiffMetadataQName(Tiff::namespaceUri, "Copyright", 0x8298, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyExposureTime = TiffMetadataQName(Tiff::namespaceUri, "ExposureTime", 0x829a, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyFNumber = TiffMetadataQName(Tiff::namespaceUri, "FNumber", 0x829d, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyIPTCNAA = TiffMetadataQName(Tiff::namespaceUri, "IPTCNAA", 0x83bb, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyInterColorProfile = TiffMetadataQName(Tiff::namespaceUri, "InterColorProfile", 0x8773, Tiff::TypeUndefined);
TiffMetadataQName Tiff::propertyExposureProgram = TiffMetadataQName(Tiff::namespaceUri, "ExposureProgram", 0x8822, Tiff::TypeShort);
TiffMetadataQName Tiff::propertySpectralSensitivity = TiffMetadataQName(Tiff::namespaceUri, "SpectralSensitivity", 0x8824, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyGPSInfo = TiffMetadataQName(Tiff::namespaceUri, "GPSInfo", 0x8825, Tiff::TypeLong);
TiffMetadataQName Tiff::propertyISOSpeedRatings = TiffMetadataQName(Tiff::namespaceUri, "ISOSpeedRatings", 0x8827, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyOECF = TiffMetadataQName(Tiff::namespaceUri, "OECF", 0x8828, Tiff::TypeUndefined);
TiffMetadataQName Tiff::propertyInterlace = TiffMetadataQName(Tiff::namespaceUri, "Interlace", 0x8829, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyTimeZoneOffset = TiffMetadataQName(Tiff::namespaceUri, "TimeZoneOffset", 0x882a, Tiff::TypeSShort);
TiffMetadataQName Tiff::propertySelfTimerMode = TiffMetadataQName(Tiff::namespaceUri, "SelfTimerMode", 0x882b, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyDateTimeOriginal = TiffMetadataQName(Tiff::namespaceUri, "DateTimeOriginal", 0x9003, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyCompressedBitsPerPixel = TiffMetadataQName(Tiff::namespaceUri, "CompressedBitsPerPixel", 0x9102, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyShutterSpeedValue = TiffMetadataQName(Tiff::namespaceUri, "ShutterSpeedValue", 0x9201, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyApertureValue = TiffMetadataQName(Tiff::namespaceUri, "ApertureValue", 0x9202, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyBrightnessValue = TiffMetadataQName(Tiff::namespaceUri, "BrightnessValue", 0x9203, Tiff::TypeSRational);
TiffMetadataQName Tiff::propertyExposureBiasValue = TiffMetadataQName(Tiff::namespaceUri, "ExposureBiasValue", 0x9204, Tiff::TypeSRational);
TiffMetadataQName Tiff::propertyMaxApertureValue = TiffMetadataQName(Tiff::namespaceUri, "MaxApertureValue", 0x9205, Tiff::TypeRational);
TiffMetadataQName Tiff::propertySubjectDistance = TiffMetadataQName(Tiff::namespaceUri, "SubjectDistance", 0x9206, Tiff::TypeSRational);
TiffMetadataQName Tiff::propertyMeteringMode = TiffMetadataQName(Tiff::namespaceUri, "MeteringMode", 0x9207, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyLightSource = TiffMetadataQName(Tiff::namespaceUri, "LightSource", 0x9208, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyFlash = TiffMetadataQName(Tiff::namespaceUri, "Flash", 0x9209, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyFocalLength = TiffMetadataQName(Tiff::namespaceUri, "FocalLength", 0x920a, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyFlashEnergy = TiffMetadataQName(Tiff::namespaceUri, "FlashEnergy", 0x920b, Tiff::TypeRational);
TiffMetadataQName Tiff::propertySpatialFrequencyResponse = TiffMetadataQName(Tiff::namespaceUri, "SpatialFrequencyResponse", 0x920c, Tiff::TypeUndefined);
TiffMetadataQName Tiff::propertyNoise = TiffMetadataQName(Tiff::namespaceUri, "Noise", 0x920d, Tiff::TypeUndefined);
TiffMetadataQName Tiff::propertyFocalPlaneXResolution = TiffMetadataQName(Tiff::namespaceUri, "FocalPlaneXResolution", 0x920e, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyFocalPlaneYResolution = TiffMetadataQName(Tiff::namespaceUri, "FocalPlaneYResolution", 0x920f, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyFocalPlaneResolutionUnit = TiffMetadataQName(Tiff::namespaceUri, "FocalPlaneResolutionUnit", 0x9210, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyImageNumber = TiffMetadataQName(Tiff::namespaceUri, "ImageNumber", 0x9211, Tiff::TypeLong);
TiffMetadataQName Tiff::propertySecurityClassification = TiffMetadataQName(Tiff::namespaceUri, "SecurityClassification", 0x922, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertyImageHistory = TiffMetadataQName(Tiff::namespaceUri, "ImageHistory", 0x9213, Tiff::TypeAscii);
TiffMetadataQName Tiff::propertySubjectLocation = TiffMetadataQName(Tiff::namespaceUri, "SubjectLocation", 0x9214, Tiff::TypeShort);
TiffMetadataQName Tiff::propertyExposureIndex = TiffMetadataQName(Tiff::namespaceUri, "ExposureIndex", 0x9215, Tiff::TypeRational);
TiffMetadataQName Tiff::propertyTIFFEPStandardID = TiffMetadataQName(Tiff::namespaceUri, "TIFFEPStandardID", 0x9216, Tiff::TypeByte);
TiffMetadataQName Tiff::propertySensingMethod = TiffMetadataQName(Tiff::namespaceUri, "SensingMethod", 0x9217, Tiff::TypeShort);
QList<TiffMetadataQName> Tiff::_tiffMetadataQNames = QList<TiffMetadataQName>();

QList<TiffMetadataQName> Tiff::tiffMetadataQNames() {
	if(_tiffMetadataQNames.isEmpty()) {
		_tiffMetadataQNames.append(propertyNewSubfileType);
		_tiffMetadataQNames.append(propertySubfileType);
		_tiffMetadataQNames.append(propertyImageWidth);
		_tiffMetadataQNames.append(propertyImageLength);
		_tiffMetadataQNames.append(propertyBitsPerSample);
		_tiffMetadataQNames.append(propertyCompression);
		_tiffMetadataQNames.append(propertyPhotometricInterpretation);
		_tiffMetadataQNames.append(propertyImageDescription);
		_tiffMetadataQNames.append(propertyMake);
		_tiffMetadataQNames.append(propertyModel);
		_tiffMetadataQNames.append(propertyStripOffsets);
		_tiffMetadataQNames.append(propertyOrientation);
		_tiffMetadataQNames.append(propertySamplesPerPixel);
		_tiffMetadataQNames.append(propertyRowsPerStrip);
		_tiffMetadataQNames.append(propertyStripByteCounts);
		_tiffMetadataQNames.append(propertyMinSampleValue);
		_tiffMetadataQNames.append(propertyMaxSampleValue);
		_tiffMetadataQNames.append(propertyXResolution);
		_tiffMetadataQNames.append(propertyYResolution);
		_tiffMetadataQNames.append(propertyPlanarConfiguration);
		_tiffMetadataQNames.append(propertyPageName);
		_tiffMetadataQNames.append(propertyXPosition);
		_tiffMetadataQNames.append(propertyYPosition);
		_tiffMetadataQNames.append(propertyFreeOffsets);
		_tiffMetadataQNames.append(propertyFreeByteCounts);
		_tiffMetadataQNames.append(propertyGrayResponseUnit);
		_tiffMetadataQNames.append(propertyGrayResponseCurve);
		_tiffMetadataQNames.append(propertyT4Options);
		_tiffMetadataQNames.append(propertyT6Options);
		_tiffMetadataQNames.append(propertyResolutionUnit);
		_tiffMetadataQNames.append(propertyPageNumber);
		_tiffMetadataQNames.append(propertyTransferFunction);
		_tiffMetadataQNames.append(propertySoftware);
		_tiffMetadataQNames.append(propertyDateTime);
		_tiffMetadataQNames.append(propertyArtist);
		_tiffMetadataQNames.append(propertyHostComputer);
		_tiffMetadataQNames.append(propertyPredictor);
		_tiffMetadataQNames.append(propertyWhitePoint);
		_tiffMetadataQNames.append(propertyPrimaryChromaticities);
		_tiffMetadataQNames.append(propertyColorMap);
		_tiffMetadataQNames.append(propertyHalftoneHints);
		_tiffMetadataQNames.append(propertyTileWidth);
		_tiffMetadataQNames.append(propertyTileLength);
		_tiffMetadataQNames.append(propertyTileOffsets);
		_tiffMetadataQNames.append(propertyTileByteCounts);
		_tiffMetadataQNames.append(propertySubIFDs);
		_tiffMetadataQNames.append(propertyInkSet);
		_tiffMetadataQNames.append(propertyInkNames);
		_tiffMetadataQNames.append(propertyNumberOfInks);
		_tiffMetadataQNames.append(propertyDotRange);
		_tiffMetadataQNames.append(propertyTargetPrinter);
		_tiffMetadataQNames.append(propertyExtraSamples);
		_tiffMetadataQNames.append(propertySampleFormat);
		_tiffMetadataQNames.append(propertySMinSampleValue);
		_tiffMetadataQNames.append(propertySMaxSampleValue);
		_tiffMetadataQNames.append(propertyTransferFunction);
		_tiffMetadataQNames.append(propertyJPEGProc);
		_tiffMetadataQNames.append(propertyJPEGTables);
		_tiffMetadataQNames.append(propertyJPEGInterchangeFormat);
		_tiffMetadataQNames.append(propertyJPEGInterchangeFormatLength);
		_tiffMetadataQNames.append(propertyJPEGRestartInterval);
		_tiffMetadataQNames.append(propertyJPEGLosslessPredictors);
		_tiffMetadataQNames.append(propertyJPEGPointTransforms);
		_tiffMetadataQNames.append(propertyJPEGQTables);
		_tiffMetadataQNames.append(propertyJPEGDCTables);
		_tiffMetadataQNames.append(propertyJPEGACTables);
		_tiffMetadataQNames.append(propertyYCbCrCoefficients);
		_tiffMetadataQNames.append(propertyYCbCrSubSampling);
		_tiffMetadataQNames.append(propertyYCbCrPositioning);
		_tiffMetadataQNames.append(propertyReferenceBlackWhite);
		_tiffMetadataQNames.append(propertyXMP);
		_tiffMetadataQNames.append(propertyCFARepeatPatternDim);
		_tiffMetadataQNames.append(propertyCFAPattern);
		_tiffMetadataQNames.append(propertyBatteryLevel);
		_tiffMetadataQNames.append(propertyCopyright);
		_tiffMetadataQNames.append(propertyExposureTime);
		_tiffMetadataQNames.append(propertyFNumber);
		_tiffMetadataQNames.append(propertyIPTCNAA);
		_tiffMetadataQNames.append(propertyInterColorProfile);
		_tiffMetadataQNames.append(propertyExposureProgram);
		_tiffMetadataQNames.append(propertySpectralSensitivity);
		_tiffMetadataQNames.append(propertyGPSInfo);
		_tiffMetadataQNames.append(propertyISOSpeedRatings);
		_tiffMetadataQNames.append(propertyOECF);
		_tiffMetadataQNames.append(propertyInterlace);
		_tiffMetadataQNames.append(propertyTimeZoneOffset);
		_tiffMetadataQNames.append(propertySelfTimerMode);
		_tiffMetadataQNames.append(propertyDateTimeOriginal);
		_tiffMetadataQNames.append(propertyCompressedBitsPerPixel);
		_tiffMetadataQNames.append(propertyShutterSpeedValue);
		_tiffMetadataQNames.append(propertyApertureValue);
		_tiffMetadataQNames.append(propertyBrightnessValue);
		_tiffMetadataQNames.append(propertyExposureBiasValue);
		_tiffMetadataQNames.append(propertyMaxApertureValue);
		_tiffMetadataQNames.append(propertySubjectDistance);
		_tiffMetadataQNames.append(propertyMeteringMode);
		_tiffMetadataQNames.append(propertyLightSource);
		_tiffMetadataQNames.append(propertyFlash);
		_tiffMetadataQNames.append(propertyFocalLength);
		_tiffMetadataQNames.append(propertyFlashEnergy);
		_tiffMetadataQNames.append(propertySpatialFrequencyResponse);
		_tiffMetadataQNames.append(propertyNoise);
		_tiffMetadataQNames.append(propertyFocalPlaneXResolution);
		_tiffMetadataQNames.append(propertyFocalPlaneYResolution);
		_tiffMetadataQNames.append(propertyFocalPlaneResolutionUnit);
		_tiffMetadataQNames.append(propertyImageNumber);
		_tiffMetadataQNames.append(propertySecurityClassification);
		_tiffMetadataQNames.append(propertyImageHistory);
		_tiffMetadataQNames.append(propertySubjectLocation);
		_tiffMetadataQNames.append(propertyExposureIndex);
		_tiffMetadataQNames.append(propertyTIFFEPStandardID);
		_tiffMetadataQNames.append(propertySensingMethod);
	}

	return _tiffMetadataQNames;
}

QString Exif::namespaceUri = "http://ns.bergnet.org/bergphoto/rdf/exif/1.0/";
TiffMetadataQName Exif::propertyExifIFD = TiffMetadataQName(Exif::namespaceUri, "ExifIFD", 0x8769, Tiff::TypeLong);
TiffMetadataQName Exif::propertyGPSIFD = TiffMetadataQName(Exif::namespaceUri, "GPSIFD", 0x8825, Tiff::TypeLong);
TiffMetadataQName Exif::propertyExifVersion = TiffMetadataQName(Exif::namespaceUri, "ExifVersion", 0x9000, Tiff::TypeUndefined);
TiffMetadataQName Exif::propertyDateTimeDigitized = TiffMetadataQName(Exif::namespaceUri, "DateTimeDigitized", 0x9004, Tiff::TypeAscii);
TiffMetadataQName Exif::propertyComponentsConfiguration = TiffMetadataQName(Exif::namespaceUri, "ComponentsConfiguration", 0x9101, Tiff::TypeUndefined);
TiffMetadataQName Exif::propertySubjectArea = TiffMetadataQName(Exif::namespaceUri, "SubjectArea", 0x9214, Tiff::TypeShort);
TiffMetadataQName Exif::propertyMakerNote = TiffMetadataQName(Exif::namespaceUri, "MakerNote", 0x927C, Tiff::TypeUndefined);
TiffMetadataQName Exif::propertyUserComment = TiffMetadataQName(Exif::namespaceUri, "UserComment", 0x9286, Tiff::TypeUndefined);
TiffMetadataQName Exif::propertySubsecTime = TiffMetadataQName(Exif::namespaceUri, "SubsecTime", 0x9290, Tiff::TypeAscii);
TiffMetadataQName Exif::propertySubsecTimeOriginal = TiffMetadataQName(Exif::namespaceUri, "SubsecTimeOriginal", 0x9291, Tiff::TypeAscii);
TiffMetadataQName Exif::propertySubsecTimeDigitized = TiffMetadataQName(Exif::namespaceUri, "SubsecTimeDigitized", 0x9292, Tiff::TypeAscii);
TiffMetadataQName Exif::propertyFlashPixVersion = TiffMetadataQName(Exif::namespaceUri, "FlashPixVersion", 0xa000, Tiff::TypeUndefined);
TiffMetadataQName Exif::propertyColorSpace = TiffMetadataQName(Exif::namespaceUri, "ColorSpace", 0xa001, Tiff::TypeShort);
TiffMetadataQName Exif::propertyPixelXDimension = TiffMetadataQName(Exif::namespaceUri, "PixelXDimension", 0xa002, Tiff::TypeLong);
TiffMetadataQName Exif::propertyPixelYDimension = TiffMetadataQName(Exif::namespaceUri, "PixelYDimension", 0xa003, Tiff::TypeLong);
TiffMetadataQName Exif::propertyRelatedSoundFile = TiffMetadataQName(Exif::namespaceUri, "RelatedSoundFile", 0xa004, Tiff::TypeAscii);
TiffMetadataQName Exif::propertyInteroperabilityIFD = TiffMetadataQName(Exif::namespaceUri, "InteroperabilityIFD", 0xa005, Tiff::TypeLong);
TiffMetadataQName Exif::propertyFlashEnergy = TiffMetadataQName(Exif::namespaceUri, "FlashEnergy", 0xa20b, Tiff::TypeRational);
TiffMetadataQName Exif::propertySpatialFrequencyResponse = TiffMetadataQName(Exif::namespaceUri, "SpatialFrequencyResponse", 0xa20c, Tiff::TypeUndefined);
TiffMetadataQName Exif::propertyFocalPlaneXResolution = TiffMetadataQName(Exif::namespaceUri, "FocalPlaneXResolution", 0xa20e, Tiff::TypeRational);
TiffMetadataQName Exif::propertyFocalPlaneYResolution = TiffMetadataQName(Exif::namespaceUri, "FocalPlaneYResolution", 0xa20f, Tiff::TypeRational);
TiffMetadataQName Exif::propertyFocalPlaneResolutionUnit = TiffMetadataQName(Exif::namespaceUri, "FocalPlaneResolutionUnit", 0xa210, Tiff::TypeShort);
TiffMetadataQName Exif::propertySubjectLocation = TiffMetadataQName(Exif::namespaceUri, "SubjectLocation", 0xa214, Tiff::TypeShort);
TiffMetadataQName Exif::propertyExposureIndex = TiffMetadataQName(Exif::namespaceUri, "ExposureIndex", 0xa215, Tiff::TypeRational);
TiffMetadataQName Exif::propertySensingMethod = TiffMetadataQName(Exif::namespaceUri, "SensingMethod", 0xa217, Tiff::TypeShort);
TiffMetadataQName Exif::propertyFileSource = TiffMetadataQName(Exif::namespaceUri, "FileSource", 0xa300, Tiff::TypeUndefined);
TiffMetadataQName Exif::propertySceneType = TiffMetadataQName(Exif::namespaceUri, "SceneType", 0xa301, Tiff::TypeUndefined);
TiffMetadataQName Exif::propertyCFAPattern = TiffMetadataQName(Exif::namespaceUri, "CFAPattern", 0xa302, Tiff::TypeUndefined);
TiffMetadataQName Exif::propertyCustomRendered = TiffMetadataQName(Exif::namespaceUri, "CustomRendered", 0xa401, Tiff::TypeShort);
TiffMetadataQName Exif::propertyExposureMode = TiffMetadataQName(Exif::namespaceUri, "ExposureMode", 0xa402, Tiff::TypeShort);
TiffMetadataQName Exif::propertyWhiteBalance = TiffMetadataQName(Exif::namespaceUri, "WhiteBalance", 0xa403, Tiff::TypeShort);
TiffMetadataQName Exif::propertyDigitalZoomRatio = TiffMetadataQName(Exif::namespaceUri, "DigitalZoomRatio", 0xa404, Tiff::TypeRational);
TiffMetadataQName Exif::propertyFocalLengthIn35mmFilm = TiffMetadataQName(Exif::namespaceUri, "FocalLengthIn35mmFilm", 0xa405, Tiff::TypeShort);
TiffMetadataQName Exif::propertySceneCaptureType = TiffMetadataQName(Exif::namespaceUri, "SceneCaptureType", 0xa406, Tiff::TypeShort);
TiffMetadataQName Exif::propertyGainControl = TiffMetadataQName(Exif::namespaceUri, "GainControl", 0xa407, Tiff::TypeRational);
TiffMetadataQName Exif::propertyContrast = TiffMetadataQName(Exif::namespaceUri, "Contrast", 0xa408, Tiff::TypeShort);
TiffMetadataQName Exif::propertySaturation = TiffMetadataQName(Exif::namespaceUri, "Saturation", 0xa409, Tiff::TypeShort);
TiffMetadataQName Exif::propertySharpness = TiffMetadataQName(Exif::namespaceUri, "Sharpness", 0xa40a, Tiff::TypeShort);
TiffMetadataQName Exif::propertyDeviceSettingDescription = TiffMetadataQName(Exif::namespaceUri, "DeviceSettingDescription", 0xa40b, Tiff::TypeUndefined);
TiffMetadataQName Exif::propertySubjectDistanceRange = TiffMetadataQName(Exif::namespaceUri, "SubjectDistanceRange", 0xa40c, Tiff::TypeShort);
TiffMetadataQName Exif::propertyImageUniqueID = TiffMetadataQName(Exif::namespaceUri, "ImageUniqueID", 0xa420, Tiff::TypeAscii);
QList<TiffMetadataQName> Exif::_tiffMetadataQNames = QList<TiffMetadataQName>();

QList<TiffMetadataQName> Exif::tiffMetadataQNames() {
	if(_tiffMetadataQNames.isEmpty()) {
		_tiffMetadataQNames.append(propertyExifIFD);
		_tiffMetadataQNames.append(propertyGPSIFD);
		_tiffMetadataQNames.append(propertyExifVersion);
		_tiffMetadataQNames.append(propertyDateTimeDigitized);
		_tiffMetadataQNames.append(propertyComponentsConfiguration);
		_tiffMetadataQNames.append(propertySubjectArea);
		_tiffMetadataQNames.append(propertyMakerNote);
		_tiffMetadataQNames.append(propertyUserComment);
		_tiffMetadataQNames.append(propertySubsecTime);
		_tiffMetadataQNames.append(propertySubsecTimeOriginal);
		_tiffMetadataQNames.append(propertySubsecTimeDigitized);
		_tiffMetadataQNames.append(propertyFlashPixVersion);
		_tiffMetadataQNames.append(propertyColorSpace);
		_tiffMetadataQNames.append(propertyPixelXDimension);
		_tiffMetadataQNames.append(propertyPixelYDimension);
		_tiffMetadataQNames.append(propertyRelatedSoundFile);
		_tiffMetadataQNames.append(propertyInteroperabilityIFD);
		_tiffMetadataQNames.append(propertyFlashEnergy);
		_tiffMetadataQNames.append(propertySpatialFrequencyResponse);
		_tiffMetadataQNames.append(propertyFocalPlaneXResolution);
		_tiffMetadataQNames.append(propertyFocalPlaneYResolution);
		_tiffMetadataQNames.append(propertyFocalPlaneResolutionUnit);
		_tiffMetadataQNames.append(propertySubjectLocation);
		_tiffMetadataQNames.append(propertyExposureIndex);
		_tiffMetadataQNames.append(propertySensingMethod);
		_tiffMetadataQNames.append(propertyFileSource);
		_tiffMetadataQNames.append(propertySceneType);
		_tiffMetadataQNames.append(propertyCFAPattern);
		_tiffMetadataQNames.append(propertyCustomRendered);
		_tiffMetadataQNames.append(propertyExposureMode);
		_tiffMetadataQNames.append(propertyWhiteBalance);
		_tiffMetadataQNames.append(propertyDigitalZoomRatio);
		_tiffMetadataQNames.append(propertyFocalLengthIn35mmFilm);
		_tiffMetadataQNames.append(propertySceneCaptureType);
		_tiffMetadataQNames.append(propertyGainControl);
		_tiffMetadataQNames.append(propertyContrast);
		_tiffMetadataQNames.append(propertySaturation);
		_tiffMetadataQNames.append(propertySharpness);
		_tiffMetadataQNames.append(propertyDeviceSettingDescription);
		_tiffMetadataQNames.append(propertySubjectDistanceRange);
		_tiffMetadataQNames.append(propertyImageUniqueID);
	}

	return _tiffMetadataQNames;
}

QMap<quint16, MetadataQName> TiffReader::_tiffTagMetadataQNameMap = QMap<quint16, MetadataQName>();
QMap<quint16, MetadataQName> TiffReader::_exifTagMetadataQNameMap = QMap<quint16, MetadataQName>();

TiffMetadataQName::TiffMetadataQName() {
}

TiffMetadataQName::TiffMetadataQName(QString namespaceUri, QString localName, quint16 tag, quint16 defaultType) : MetadataQName(namespaceUri, localName) {
	_tag = tag;
	_defaultType = defaultType;
}

quint16 TiffMetadataQName::tag() {
	return _tag;
}

quint16 TiffMetadataQName::defaultType() {
	return _defaultType;
}

bool TiffMetadataQName::operator !=(const TiffMetadataQName other) const {
	return MetadataQName::operator !=(other);
}

bool TiffMetadataQName::operator <(const TiffMetadataQName other) const {
	return MetadataQName::operator <(other);
}

bool TiffMetadataQName::operator ==(const MetadataQName other) const {
	return MetadataQName::operator ==(other);
}

bool TiffMetadataQName::operator ==(const TiffMetadataQName other) const {
	return MetadataQName::operator ==(other);
}

uint qHash(const TiffMetadataQName& qName) {
	return qHash((MetadataQName)qName);
}

int Tiff::typeSize(Type type) {
	switch(type) {
		case TypeByte:
		case TypeAscii:
			return 1;
		case TypeShort:
			return 2;
		case TypeLong:
			return 4;
		case TypeRational:
			return 8;
		case TypeSByte:
		case TypeUndefined:
			return 1;
		case TypeSShort:
			return 2;
		case TypeSLong:
			return 4;
		case TypeSRational:
			return 8;
		case TypeFloat:
			return 4;
		case TypeDouble:
			return 8;
		default:
			return 1;
	}
}

double Tiff::lightSourceTemperature(quint16 value) {
	if(value & 0x8000)
		return (double)(value & 0x7fff);

	switch(value) {
		case 1:
		case 4:
		case 9:
		case 18:
		case 20:
			return 5500.0;

		case 2:
		case 14:
			return 4200.0;

		case 3:
		case 17:
			return 2850.0;

		case 10:
		case 19:
		case 21:
			return 6500.0;

		case 11:
		case 22:
			return 7500.0;

		case 12:
			return 6400.0;

		case 13:
		case 23:
			return 5000.0;

		case 15:
			return 3450.0;

		case 24:
			return 3200.0;

		default:
			return 0.0;
	}
}

TiffEntry::TiffEntry(TiffMetadataQName qName, QVariant value) {
	QList<QVariant> values;

	values << value;

	_tag = qName.tag();
	_type = (Tiff::Type)qName.defaultType();
	_values = values;
}

TiffEntry::TiffEntry(TiffMetadataQName qName, QList<QVariant> values) {
	_tag = qName.tag();
	_type = (Tiff::Type)qName.defaultType();
	_values = values;
}

TiffEntry::TiffEntry(quint16 tag, Tiff::Type type, QVariant value) {
	QList<QVariant> values;

	values << value;

	_tag = tag;
	_type = type;
	_values = values;
}

TiffEntry::TiffEntry(quint16 tag, Tiff::Type type, QList<QVariant> values) {
	_tag = tag;
	_type = type;
	_values = values;
}

quint16 TiffEntry::tag() {
	return _tag;
}

Tiff::Type TiffEntry::type() {
	return _type;
}

QList<QVariant> TiffEntry::values() {
	return _values;
}

quint32 TiffEntry::valueCount() {
	switch(_type) {
		case Tiff::TypeAscii:
			return _values.at(0).toString().size() + 1;
		case Tiff::TypeUndefined:
			return _values.at(0).toByteArray().size();
		default:
			return _values.size();
	}
}

quint32 TiffEntry::valueSize() {
	return valueCount() * Tiff::typeSize(_type);
}

void TiffDirectory::addTag(TiffEntry* entry) {
	_entries.insert(entry->tag(), entry);
}

QMap<quint16, TiffEntry*> TiffDirectory::entryMap() {
	return _entries;
}

quint32 TiffDirectory::headerSize() {
	return _entries.size()*12 + 6;
}

quint32 TiffDirectory::valueSize() {
	quint32 size = 0;

	foreach(TiffEntry* entry, _entries) {
		quint32 entryValueSize = entry->valueSize();

		if(entryValueSize > 4)
			size += entryValueSize;
	}

	return size;
}

quint32 TiffDirectory::size() {
	return headerSize() + valueSize();
}

QMap<quint16, MetadataQName> TiffReader::tiffTagMetadataQNameMap() {
	if(_tiffTagMetadataQNameMap.isEmpty()) {
		foreach(TiffMetadataQName qName, Tiff::tiffMetadataQNames()) {
			_tiffTagMetadataQNameMap.insert(qName.tag(), qName);
		}
	}

	return _tiffTagMetadataQNameMap;
}

QMap<quint16, MetadataQName> TiffReader::exifTagMetadataQNameMap() {
	if(_exifTagMetadataQNameMap.isEmpty()) {
		foreach(TiffMetadataQName qName, Exif::tiffMetadataQNames()) {
			_exifTagMetadataQNameMap.insert(qName.tag(), qName);
		}
	}

	return _exifTagMetadataQNameMap;
}

TiffReader::TiffReader(QIODevice* device) {
	_dataStream = new QDataStream(device);
	_document = 0;
}

TiffReader::~TiffReader() {
	delete _dataStream;
}

MetadataNode* TiffReader::read(MetadataNode* parent) {
	Q_UNUSED(parent);

	_document = new MetadataDocument();

	parseHeader(_document);
	parseDirectories(_document, 4);

	return _document;
}

MetadataDocument* TiffReader::document() {
	return _document;
}

void TiffReader::detectByteOrder() {
	quint16 byteOrder;

	*_dataStream >> byteOrder;

	if(byteOrder == 0x4949)
		_dataStream->setByteOrder(QDataStream::LittleEndian);
	else if(byteOrder == 0x4d4d)
		_dataStream->setByteOrder(QDataStream::BigEndian);
}

quint8 TiffReader::readByte() {
	quint8 value;
	*_dataStream >> value;

	return value;
}

quint16 TiffReader::readShort() {
	quint16 value;
	*_dataStream >> value;

	return value;
}

quint32 TiffReader::readLong() {
	quint32 value;
	*_dataStream >> value;

	return value;
}

Rational TiffReader::readRational() {
	qlonglong numerator = readLong();
	qlonglong denominator = readLong();

	return Rational(numerator, denominator);
}

qint8 TiffReader::readSByte() {
	qint8 value;
	*_dataStream >> value;

	return value;
}

qint16 TiffReader::readSShort() {
	qint16 value;
	*_dataStream >> value;

	return value;
}

qint32 TiffReader::readSLong() {
	qint32 value;
	*_dataStream >> value;

	return value;
}

Rational TiffReader::readSRational() {
	qlonglong numerator = readSLong();
	qlonglong denominator = readSLong();

	return Rational(numerator, denominator);
}

float TiffReader::readFloat() {
	float value;
	*_dataStream >> value;

	return value;
}

double TiffReader::readDouble() {
	double value;
	*_dataStream >> value;

	return value;
}

QList<QVariant> TiffReader::readValues(quint16 tag, quint16 type, quint32 count) {
	QList<QVariant> values;

	if(type == Tiff::TypeAscii) {
		values.append(tagValue(tag, QVariant(QString(_dataStream->device()->read(count)))));
	} else if(type == Tiff::TypeUndefined || type == Tiff::TypeByte || type == Tiff::TypeSByte) {
		values.append(tagValue(tag, QVariant(_dataStream->device()->read(count))));
	} else {
		for(quint32 i=0; i<count; i++) {
			QVariant value;

			switch(type) {
				case Tiff::TypeByte:
					value = QVariant(readByte());
					break;
				case Tiff::TypeShort:
					value = QVariant(readShort());
					break;
				case Tiff::TypeLong:
					value = QVariant(readLong());
					break;
				case Tiff::TypeRational:
					value.setValue(readRational());
					break;
				case Tiff::TypeSByte:
					value = QVariant(readSByte());
					break;
				case Tiff::TypeSShort:
					value = QVariant(readSShort());
					break;
				case Tiff::TypeSLong:
					value = QVariant(readSLong());
					break;
				case Tiff::TypeSRational:
					value.setValue(readSRational());
					break;
				case Tiff::TypeFloat:
					value = QVariant(readFloat());
					break;
				case Tiff::TypeDouble:
					value = QVariant(readDouble());
					break;
				default:
					qDebug("tiff type not supported");
					break;
			}

			if(!value.isNull())
				values.append(tagValue(tag, value));
		}
	}

	return values;
}

void TiffReader::parseEntry(MetadataNode* parent, quint32 offset) {
	_dataStream->device()->seek(offset);

	quint16 tag, type;
	quint32 count;

	*_dataStream >> tag;
	*_dataStream >> type;
	*_dataStream >> count;

	int dataSize = Tiff::typeSize((Tiff::Type)type) * count;

	if(tag == Exif::propertyMakerNote.tag()) {
		MetadataProperty* makerNoteProperty = new MetadataProperty(parent, Exif::propertyMakerNote);
		parseMakerNote(makerNoteProperty, readLong());

		return;
	}

	if(dataSize > 4)
		_dataStream->device()->seek(readLong());

	QList<QVariant> values = readValues(tag, type, count);

	MetadataQName qName = tagQName(tag);

	if(tag == Tiff::propertySubIFDs.tag()) {
		foreach(QVariant value, values) {
			MetadataProperty* property = new MetadataProperty(parent, Tiff::propertySubIFDs);
			parseDirectory(property, value.toUInt());
		}

		return;
	} else if(tag == Exif::propertyExifIFD.tag()) {
		MetadataProperty* exifIfdProperty = new MetadataProperty(parent, Exif::propertyExifIFD);
		parseExif(exifIfdProperty, values.at(0).toUInt());

		return;
	}

	MetadataProperty* property = new MetadataProperty(parent, qName);
	property->setValues(values);
}

void TiffReader::parseMakerNote(MetadataNode* parent, quint32 offset) {
	Q_UNUSED(parent);
	Q_UNUSED(offset);
}

void TiffReader::parseExif(MetadataNode* parent, quint32 offset) {
	parseDirectory(parent, offset);
}

void TiffReader::parseDirectory(MetadataNode* parent, quint32 offset) {
	_dataStream->device()->seek(offset);
	quint16 directorySize;
	*_dataStream >> directorySize;

	for(int i=0; i<directorySize; i++)
		parseEntry(parent, offset+2+i*12);
}

void TiffReader::parseDirectories(MetadataNode* parent, quint32 offset) {
	_dataStream->device()->seek(offset);
	quint32 directoryOffset;
	*_dataStream >> directoryOffset;
	int directoryNumber = 0;

	while(directoryOffset != 0) {
		_dataStream->device()->seek(directoryOffset);
		quint16 directorySize;
		*_dataStream >> directorySize;

		MetadataResource* resource = new MetadataResource(parent, QString("tiffdir%1").arg(directoryNumber));

		parseDirectory(resource, directoryOffset);

		_dataStream->device()->seek(directoryOffset+directorySize*12+2);
		*_dataStream >> directoryOffset;
		directoryNumber++;
	}
}

void TiffReader::parseHeader(MetadataNode* parent, quint32 offset) {
	Q_UNUSED(parent);

	_dataStream->device()->seek(offset);

	detectByteOrder();
}

MetadataQName TiffReader::tagQName(quint16 tag) {
	if(tiffTagMetadataQNameMap().contains(tag))
		return tiffTagMetadataQNameMap().value(tag);
	else if(exifTagMetadataQNameMap().contains(tag))
		return exifTagMetadataQNameMap().value(tag);
	else
		return MetadataQName(Tiff::namespaceUri, QString("tag%1").arg(QString::number(tag, 16)));
}

QVariant TiffReader::tagValue(quint16 tag, QVariant value) {
	Q_UNUSED(tag);

	/*if(tag == Tiff::propertyCompression.tag()) {
		switch(value.toInt()) {
			case 6:
				return QVariant("JPEG");
			default:
				return value;
		}
	} else if(tag == Tiff::propertyResolutionUnit.tag()) {
		switch(value.toInt()) {
			case 1:
				return QVariant("none");
			case 2:
				return QVariant("inches");
			case 3:
				return QVariant("cm");
			default:
				return value;
		}
	} else if(tag == Tiff::propertyExposureProgram.tag()) {
		switch(value.toInt()) {
			case 0:
				return QVariant("not defined");
			case 1:
				return QVariant("manual");
			case 2:
				return QVariant("program ae");
			case 3:
				return QVariant("aperture-priority ae");
			case 4:
				return QVariant("shutter speed priority ae");
			case 5:
				return QVariant("creative (slow speed)");
			case 6:
				return QVariant("action (high speed)");
			case 7:
				return QVariant("portrait");
			case 8:
				return QVariant("landscape");
			default:
				return value;
		}
	} else if(tag == Exif::propertyExifVersion.tag() || tag == Exif::propertyFlashPixVersion.tag()) {
		return QVariant(QString(value.toByteArray()));
	}*/

	return value;
}


TiffImageReader::TiffImageReader(QIODevice* device) {
	_device = device;
}

Image* TiffImageReader::read(MetadataResource* resource, Image* image, ImageFormat format) {
	int width = MetadataQuery::typedValue<int>(resource, Tiff::propertyImageWidth);
	int height = MetadataQuery::typedValue<int>(resource, Tiff::propertyImageLength);
	int compression = MetadataQuery::typedValue<int>(resource, Tiff::propertyCompression);

	if(MetadataQuery::property(resource, Tiff::propertyTileWidth) != 0) {
		int tileWidth = MetadataQuery::typedValue<int>(resource, Tiff::propertyTileWidth);
		int tilesPerLine = (width+tileWidth-1) / tileWidth;
		int fullWidth = tilesPerLine * tileWidth;
		int tileLength = MetadataQuery::typedValue<int>(resource, Tiff::propertyTileLength);
		int tilesPerRow = (height+tileLength-1) / tileLength;
		int fullHeight = tilesPerRow * tileLength;
		QList<quint32> offsets = MetadataQuery::typedValues<quint32>(resource, Tiff::propertyTileOffsets);
		QList<quint32> lengths = MetadataQuery::typedValues<quint32>(resource, Tiff::propertyTileByteCounts);

		if(image == 0)
			image = createImage(format, fullWidth, fullHeight);

		if(compression == 7)
			readLosslessJpegTiled(offsets, lengths, tileWidth, tileLength, tilesPerLine, image, format);

		fixImage(image, format);
	} else {
		quint32 offset = MetadataQuery::typedValue<quint32>(resource, Tiff::propertyStripOffsets);
		quint32 length = MetadataQuery::typedValue<quint32>(resource, Tiff::propertyStripByteCounts);

		if(image == 0)
			image = createImage(format, width, height);

		if(compression == 7)
			readLosslessJpeg(offset, length, image);

		fixImage(image, format);
	}

	return image;
}

Image* TiffImageReader::read(quint32 offset, quint32 length, Image* image, ImageFormat format) {
	Q_UNUSED(offset);
	Q_UNUSED(length);
	Q_UNUSED(image);
	Q_UNUSED(image);
	Q_UNUSED(format);

	return 0;
}

Image* TiffImageReader::readLosslessJpeg(quint32 offset, quint32 length, Image* image, ImageFormat format) {
	_device->seek(offset);
	QByteArray data = _device->read(length);
	QBuffer buffer(&data);

	buffer.open(QIODevice::ReadOnly);

	LosslessJpegDecoder losslessJpegDecoder(&buffer);
	losslessJpegDecoder.decodeHeader();

	int width = losslessJpegDecoder.samples() * losslessJpegDecoder.components();
	int height = losslessJpegDecoder.lines();

	if(image == 0)
		image = createImage(format, width, height);

	int stripeSize = image->stripeSize();

	// workaround for some dngs
	if(image->width()*2 == width)
		stripeSize *= 2;

	losslessJpegDecoder.decodeImage((quint16*)image->data(), stripeSize);

	BayerImage16* bayerImage16 = qobject_cast<BayerImage16*>(image);

	if(bayerImage16 != 0)
		bayerImage16->setPrecision(losslessJpegDecoder.precision());

	buffer.close();

	fixImage(image, format);

	return image;
}

Image* TiffImageReader::readLosslessJpegTiled(QList<quint32> offsets, QList<quint32> lengths, int tileWidth, int tileHeight, int tileColumns, Image* image, ImageFormat format) {
	Q_UNUSED(format);

	if(image == 0) {
		qDebug("can't detect image size if tiled");
		return 0;
	}

	for(int i=0; i<offsets.size(); i++) {
		quint32 offset = offsets.at(i);
		quint32 length = lengths.at(i);

		_device->seek(offset);
		QByteArray data = _device->read(length);
		QBuffer buffer(&data);

		buffer.open(QIODevice::ReadOnly);

		LosslessJpegDecoder losslessJpegDecoder(&buffer);
		losslessJpegDecoder.decodeHeader();

		quint16* imageOffset = (quint16*)image->dataOffset((i%tileColumns)*tileWidth, (i/tileColumns)*tileHeight);

		losslessJpegDecoder.decodeImage(imageOffset, image->stripeSize());

		buffer.close();
	}

	return image;
}

QImage TiffImageReader::readQImage(quint32 offset, quint32 length, const char* format) {
	_device->seek(offset);
	QByteArray data = _device->read(length);
	QBuffer buffer(&data);

	buffer.open(QIODevice::ReadOnly);

	QImage image;
	image.load(&buffer, format);

	buffer.close();

	return image;
}

Image* TiffImageReader::createImage(ImageFormat format, int width, int height) {
	Image* image = 0;

	int targetWidth = qMax(format.width(), width);
	int targetHeight = qMax(format.height(), height);

	if(format.type() == BayerImage16::type())
		image = new BayerImage16(targetWidth, targetHeight);

	return image;
}

void TiffImageReader::fixImage(Image* image, ImageFormat format) {
	if(format.width() > image->width() || format.height() > image->height())
		qDebug("can't upscale image");

	if(format.width() != -1)
		image->setWidth(format.width());

	if(format.height() != -1)
		image->setHeight(format.height());
}

QString TiffWriter::name = "TiffWriter";

QMap<MetadataQName, TiffMetadataQName> TiffWriter::_tiffMetadataQNameMap = QMap<MetadataQName, TiffMetadataQName>();
QMap<MetadataQName, TiffMetadataQName> TiffWriter::_exifMetadataQNameMap = QMap<MetadataQName, TiffMetadataQName>();

TiffWriter::TiffWriter() {
}

TiffWriter::~TiffWriter() {
}

bool TiffWriter::accepts(QString format) {
	return format == RgbImage48::type();
}

bool TiffWriter::write(QIODevice* device, Image* image, MetadataResource* resource, BergPhoto::Parameters parameters) {
	Q_UNUSED(parameters);

	RgbImage48* rgbImage48;
	BayerImage16* bayerImage16;

	if((rgbImage48 = qobject_cast<RgbImage48*>(image)) != 0) {
		QDataStream* dataStream = new QDataStream(device);
		dataStream->setByteOrder(QDataStream::LittleEndian);

		QList<QVariant> valuesBitsPerSample;
		valuesBitsPerSample << 16 << 16 << 16;

		writeHeader(dataStream);

		TiffDirectory directory;
		directory.addTag(new TiffEntry(Tiff::propertyImageWidth, QVariant(rgbImage48->width())));
		directory.addTag(new TiffEntry(Tiff::propertyImageLength, QVariant(rgbImage48->height())));
		directory.addTag(new TiffEntry(Tiff::propertyBitsPerSample, valuesBitsPerSample));
		directory.addTag(new TiffEntry(Tiff::propertyCompression, QVariant(1)));
		directory.addTag(new TiffEntry(Tiff::propertyPhotometricInterpretation, QVariant(2)));
		directory.addTag(new TiffEntry(Tiff::propertySamplesPerPixel, QVariant(3)));

		if(image->colorProfile() != 0)
			new MetadataProperty(resource, Tiff::propertyInterColorProfile, QVariant(image->colorProfile()->toByteArray()));

		addTiffMetadataTags(&directory, resource);

		TiffDirectory exifDirectory;
		addExifMetadataTags(&exifDirectory, resource);

		quint32 exifIdfOffset = (quint32)dataStream->device()->pos() + 4;
		directory.addTag(new TiffEntry(Exif::propertyExifIFD, QVariant(exifIdfOffset)));

		quint32 firstIdfOffset = exifIdfOffset + exifDirectory.size();

		*dataStream << firstIdfOffset;

		writeIdf(dataStream, &exifDirectory);

		QByteArray imageData;

		for(int y=0; y<rgbImage48->height(); y++)
			imageData.append(QByteArray::fromRawData((const char*)rgbImage48->dataStripe(y), rgbImage48->width()*6));

		writeIdf(dataStream, &directory, imageData);

		delete dataStream;

		return true;
	} else if((bayerImage16 = qobject_cast<BayerImage16*>(image)) != 0) {
		QDataStream* dataStream = new QDataStream(device);
		dataStream->setByteOrder(QDataStream::LittleEndian);

		writeHeader(dataStream);

		TiffDirectory directory;
		directory.addTag(new TiffEntry(Tiff::propertyImageWidth, QVariant(bayerImage16->width())));
		directory.addTag(new TiffEntry(Tiff::propertyImageLength, QVariant(bayerImage16->height())));
		directory.addTag(new TiffEntry(Tiff::propertyBitsPerSample, QVariant(16)));
		directory.addTag(new TiffEntry(Tiff::propertyCompression, QVariant(1)));
		directory.addTag(new TiffEntry(Tiff::propertyPhotometricInterpretation, QVariant(1)));
		directory.addTag(new TiffEntry(Tiff::propertySamplesPerPixel, QVariant(1)));

		addTiffMetadataTags(&directory, resource);

		quint32 firstIdfOffset = (quint32)dataStream->device()->pos() + 4;

		*dataStream << firstIdfOffset;

		QByteArray imageData;

		for(int y=0; y<bayerImage16->height(); y++)
			imageData.append(QByteArray::fromRawData((const char*)bayerImage16->dataStripe(y), bayerImage16->width()*2));

		writeIdf(dataStream, &directory, imageData);

		delete dataStream;

		return true;
	}

	return false;
}

void TiffWriter::writeHeader(QDataStream* dataStream) {
	if(dataStream->byteOrder() == QDataStream::LittleEndian)
		*dataStream << (quint16)Tiff::ByteOrderLittleEndian;
	else
		*dataStream << (quint16)Tiff::ByteOrderBigEndian;

	*dataStream << (quint16)Tiff::MagicNumber;
}

void TiffWriter::writeIdf(QDataStream* dataStream, TiffDirectory* directory, QByteArray data, quint32 nextIdf) {
	if(data.size() != 0) {
		directory->addTag(new TiffEntry(Tiff::propertyStripByteCounts, QVariant((quint32)data.size())));
		directory->addTag(new TiffEntry(Tiff::propertyStripOffsets, QVariant(0)));
	}

	quint32 currentOffset = (quint32)dataStream->device()->pos();
	quint32 valuesOffset = currentOffset + directory->headerSize();
	quint32 imageOffset = currentOffset + directory->size();

	QMap<quint16, TiffEntry*> entries = directory->entryMap();
	*dataStream << (quint16)entries.size();

	QList<quint16> sortedTags = entries.keys();
	qSort(sortedTags);

	foreach(quint16 tag, sortedTags) {
		TiffEntry* entry = entries.value(tag);
		quint32 entryValueSize = entry->valueSize();

		*dataStream << tag;
		*dataStream << (quint16)entry->type();
		*dataStream << entry->valueCount();

		if(tag == Tiff::propertyStripOffsets.tag()) {
			*dataStream << imageOffset;
		} else if(entryValueSize <= 4) {
			writeValues(dataStream, entry);

			for(quint32 i=entryValueSize; i<4; i++)
				*dataStream << (quint8)0;
		} else {
			*dataStream << valuesOffset;
			valuesOffset += entryValueSize;
		}
	}

	*dataStream << nextIdf;

	foreach(quint16 tag, sortedTags) {
		TiffEntry* entry = entries.value(tag);

		if(entry->valueSize() > 4)
			writeValues(dataStream, entry);
	}

	dataStream->device()->write(data);
}

void TiffWriter::writeValues(QDataStream* dataStream, TiffEntry* entry) {
	Tiff::Type type = entry->type();

	if(type == Tiff::TypeAscii) {
		dataStream->device()->write(entry->values().at(0).toString().toAscii());
		*dataStream << (quint8)0;
	} else if(type == Tiff::TypeUndefined) {
		dataStream->device()->write(entry->values().at(0).toByteArray());
	} else {
		foreach(QVariant value, entry->values()) {
			value = tagValue(entry->tag(), value);

			switch(type) {
				case Tiff::TypeByte:
					*dataStream << value.value<quint8>();
					break;
				case Tiff::TypeShort:
					*dataStream << value.value<quint16>();
					break;
				case Tiff::TypeLong:
					*dataStream << value.value<quint32>();
					break;
				case Tiff::TypeRational:
					writeRational(dataStream, value.value<Rational>());
					break;
				case Tiff::TypeSByte:
					*dataStream << value.value<qint8>();
					break;
				case Tiff::TypeSShort:
					*dataStream << value.value<qint16>();
					break;
				case Tiff::TypeSLong:
					*dataStream << value.value<qint32>();
					break;
				case Tiff::TypeSRational:
					writeSRational(dataStream, value.value<Rational>());
					break;
				case Tiff::TypeFloat:
					*dataStream << value.value<float>();
					break;
				case Tiff::TypeDouble:
					*dataStream << value.value<double>();
					break;
				default:
					break;
			}
		}
	}
}

void TiffWriter::writeRational(QDataStream* dataStream, Rational value) {
	*dataStream << (quint32)value.numerator();
	*dataStream << (quint32)value.denominator();
}

void TiffWriter::writeSRational(QDataStream* dataStream, Rational value) {
	*dataStream << (qint32)value.numerator();
	*dataStream << (qint32)value.denominator();
}

QMap<MetadataQName, TiffMetadataQName> TiffWriter::tiffMetadataQNameMap() {
	if(_tiffMetadataQNameMap.isEmpty()) {
		foreach(TiffMetadataQName qName, Tiff::tiffMetadataQNames()) {
			_tiffMetadataQNameMap.insert((MetadataQName)qName, qName);
		}
	}

	return _tiffMetadataQNameMap;
}

QMap<MetadataQName, TiffMetadataQName> TiffWriter::exifMetadataQNameMap() {
	if(_exifMetadataQNameMap.isEmpty()) {
		foreach(TiffMetadataQName qName, Exif::tiffMetadataQNames()) {
			_exifMetadataQNameMap.insert((MetadataQName)qName, qName);
		}
	}

	return _exifMetadataQNameMap;
}

void TiffWriter::addTiffMetadataTags(TiffDirectory* directory, MetadataResource* resource) {
	if(resource == 0)
		return;

	foreach(MetadataNode* node, resource->children()) {
		if(!node->isProperty())
			continue;

		MetadataProperty* property = node->toProperty();

		if(property->qName().namespaceUri() != Tiff::namespaceUri)
			continue;

		if(tiffMetadataQNameMap().contains(property->qName())) {
			TiffMetadataQName tiffQName = tiffMetadataQNameMap().value(property->qName());

			if(!directory->entryMap().contains(tiffQName.tag()))
				directory->addTag(new TiffEntry(tiffQName, property->values()));
		}
	}
}

void TiffWriter::addExifMetadataTags(TiffDirectory* directory, MetadataResource* resource) {
	if(resource == 0)
		return;

	MetadataProperty* exifIfdProperty = MetadataQuery::property(resource, Exif::propertyExifIFD);

	foreach(MetadataNode* node, exifIfdProperty->children()) {
		if(!node->isProperty())
			continue;

		MetadataProperty* property = node->toProperty();

		if(property->qName().namespaceUri() != Exif::namespaceUri)
			continue;

		if(exifMetadataQNameMap().contains(property->qName())) {
			TiffMetadataQName tiffQName = exifMetadataQNameMap().value(property->qName());

			if(!directory->entryMap().contains(tiffQName.tag()))
				directory->addTag(new TiffEntry(tiffQName, property->values()));
		}
	}
}

QVariant TiffWriter::tagValue(int tag, QVariant value) {
	Q_UNUSED(tag);

	/*QString stringValue = value.toString();

	if(tag == Tiff::propertyCompression.tag()) {
		if(stringValue == "JPEG")
			return QVariant(6);
		else
			return value;
	} else if(tag == Tiff::propertyResolutionUnit.tag()) {
		if(stringValue == "none")
			return QVariant(1);
		else if(stringValue == "inches")
			return QVariant(2);
		else if(stringValue == "cm")
			return QVariant(3);
	} else if(tag == Exif::propertyExposureProgram.tag()) {
		if(stringValue == "not defined")
			return QVariant(0);
		else if(stringValue == "manual")
			return QVariant(1);
		else if(stringValue == "program ae")
			return QVariant(2);
		else if(stringValue == "aperture-priority ae")
			return QVariant(3);
		else if(stringValue == "shutter speed priority ae")
			return QVariant(4);
		else if(stringValue == "creative (slow speed)")
			return QVariant(5);
		else if(stringValue == "action (high speed)")
			return QVariant(6);
		else if(stringValue == "portrait")
			return QVariant(7);
		else if(stringValue == "landscape")
			return QVariant(8);
		else
			return value;
	}*/

	return value;
}

TiffSimpleMetadataTransform::~TiffSimpleMetadataTransform() {
}

MetadataNode* TiffSimpleMetadataTransform::transform(MetadataNode* source) {
	if(!source->isResource())
		return 0;

	QList<MetadataQName> propertyToKeepTiff;
	propertyToKeepTiff << Tiff::propertyDateTime << Tiff::propertyMake << Tiff::propertyModel << Tiff::propertyOrientation << Exif::propertyExifIFD;

	MetadataResource* resource = new MetadataResource(0);
	MetadataResource* sourceResource = source->toResource();

	foreach(MetadataNode* sourceChild, sourceResource->children()) {
		if(!sourceChild->isProperty())
			continue;

		MetadataProperty* sourceProperty = sourceChild->toProperty();

		if(sourceProperty->qName().namespaceUri() == Tiff::namespaceUri && !propertyToKeepTiff.contains(sourceProperty->qName()))
			continue;

		sourceProperty->clone(resource);
	}

	return resource;
}
