/* Copyright 2008 Thomas Bergwinkl
 *
 * This file is part of bergphoto.
 *
 * bergphoto is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * bergphoto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with bergphoto.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef TIFF_H_
#define TIFF_H_

#include <QByteArray>
#include <QDataStream>
#include <QIODevice>
#include <QList>
#include <QMap>
#include <QString>
#include <QVariant>

#include <Image.h>
#include <ImageFilter.h>
#include <Metadata.h>

// TODO: GPS Info

class TiffMetadataQName : public MetadataQName {
public:
	TiffMetadataQName();
	TiffMetadataQName(QString namespaceUri, QString localName, quint16 tag, quint16 defaultType);

	quint16 tag();
	quint16 defaultType();

	bool operator !=(const TiffMetadataQName other) const;
	bool operator <(const TiffMetadataQName other) const;
	bool operator ==(const MetadataQName other) const;
	bool operator ==(const TiffMetadataQName other) const;

private:
	quint16 _tag;
	quint16 _defaultType;
};

uint qHash(const TiffMetadataQName& qName);

class Tiff {
public:
	static QString namespaceUri;

	static TiffMetadataQName propertyNewSubfileType;
	static TiffMetadataQName propertySubfileType;
	static TiffMetadataQName propertyImageWidth;
	static TiffMetadataQName propertyImageLength;
	static TiffMetadataQName propertyBitsPerSample;
	static TiffMetadataQName propertyCompression;
	static TiffMetadataQName propertyPhotometricInterpretation;
	static TiffMetadataQName propertyThreshholding;
	static TiffMetadataQName propertyCellWidth;
	static TiffMetadataQName propertyCellLength;
	static TiffMetadataQName propertyFillOrder;
	static TiffMetadataQName propertyDocumentName;
	static TiffMetadataQName propertyImageDescription;
	static TiffMetadataQName propertyMake;
	static TiffMetadataQName propertyModel;
	static TiffMetadataQName propertyStripOffsets;
	static TiffMetadataQName propertyOrientation;
	static TiffMetadataQName propertySamplesPerPixel;
	static TiffMetadataQName propertyRowsPerStrip;
	static TiffMetadataQName propertyStripByteCounts;
	static TiffMetadataQName propertyMinSampleValue;
	static TiffMetadataQName propertyMaxSampleValue;
	static TiffMetadataQName propertyXResolution;
	static TiffMetadataQName propertyYResolution;
	static TiffMetadataQName propertyPlanarConfiguration;
	static TiffMetadataQName propertyPageName;
	static TiffMetadataQName propertyXPosition;
	static TiffMetadataQName propertyYPosition;
	static TiffMetadataQName propertyFreeOffsets;
	static TiffMetadataQName propertyFreeByteCounts;
	static TiffMetadataQName propertyGrayResponseUnit;
	static TiffMetadataQName propertyGrayResponseCurve;
	static TiffMetadataQName propertyT4Options;
	static TiffMetadataQName propertyT6Options;
	static TiffMetadataQName propertyResolutionUnit;
	static TiffMetadataQName propertyPageNumber;
	static TiffMetadataQName propertyTransferFunction;
	static TiffMetadataQName propertySoftware;
	static TiffMetadataQName propertyDateTime;
	static TiffMetadataQName propertyArtist;
	static TiffMetadataQName propertyHostComputer;
	static TiffMetadataQName propertyPredictor;
	static TiffMetadataQName propertyWhitePoint;
	static TiffMetadataQName propertyPrimaryChromaticities;
	static TiffMetadataQName propertyColorMap;
	static TiffMetadataQName propertyHalftoneHints;
	static TiffMetadataQName propertyTileWidth;
	static TiffMetadataQName propertyTileLength;
	static TiffMetadataQName propertyTileOffsets;
	static TiffMetadataQName propertyTileByteCounts;
	static TiffMetadataQName propertySubIFDs;
	static TiffMetadataQName propertyInkSet;
	static TiffMetadataQName propertyInkNames;
	static TiffMetadataQName propertyNumberOfInks;
	static TiffMetadataQName propertyDotRange;
	static TiffMetadataQName propertyTargetPrinter;
	static TiffMetadataQName propertyExtraSamples;
	static TiffMetadataQName propertySampleFormat;
	static TiffMetadataQName propertySMinSampleValue;
	static TiffMetadataQName propertySMaxSampleValue;
	static TiffMetadataQName propertyTransferRange;
	static TiffMetadataQName propertyJPEGTables;
	static TiffMetadataQName propertyJPEGProc;
	static TiffMetadataQName propertyJPEGInterchangeFormat;
	static TiffMetadataQName propertyJPEGInterchangeFormatLength;
	static TiffMetadataQName propertyJPEGRestartInterval;
	static TiffMetadataQName propertyJPEGLosslessPredictors;
	static TiffMetadataQName propertyJPEGPointTransforms;
	static TiffMetadataQName propertyJPEGQTables;
	static TiffMetadataQName propertyJPEGDCTables;
	static TiffMetadataQName propertyJPEGACTables;
	static TiffMetadataQName propertyYCbCrCoefficients;
	static TiffMetadataQName propertyYCbCrSubSampling;
	static TiffMetadataQName propertyYCbCrPositioning;
	static TiffMetadataQName propertyReferenceBlackWhite;
	static TiffMetadataQName propertyXMP;
	static TiffMetadataQName propertyCFARepeatPatternDim;
	static TiffMetadataQName propertyCFAPattern;
	static TiffMetadataQName propertyBatteryLevel;
	static TiffMetadataQName propertyCopyright;
	static TiffMetadataQName propertyExposureTime;
	static TiffMetadataQName propertyFNumber;
	static TiffMetadataQName propertyIPTCNAA;
	static TiffMetadataQName propertyInterColorProfile;
	static TiffMetadataQName propertyExposureProgram;
	static TiffMetadataQName propertySpectralSensitivity;
	static TiffMetadataQName propertyGPSInfo;
	static TiffMetadataQName propertyISOSpeedRatings;
	static TiffMetadataQName propertyOECF;
	static TiffMetadataQName propertyInterlace;
	static TiffMetadataQName propertyTimeZoneOffset;
	static TiffMetadataQName propertySelfTimerMode;
	static TiffMetadataQName propertyDateTimeOriginal;
	static TiffMetadataQName propertyCompressedBitsPerPixel;
	static TiffMetadataQName propertyShutterSpeedValue;
	static TiffMetadataQName propertyApertureValue;
	static TiffMetadataQName propertyBrightnessValue;
	static TiffMetadataQName propertyExposureBiasValue;
	static TiffMetadataQName propertyMaxApertureValue;
	static TiffMetadataQName propertySubjectDistance;
	static TiffMetadataQName propertyMeteringMode;
	static TiffMetadataQName propertyLightSource;
	static TiffMetadataQName propertyFlash;
	static TiffMetadataQName propertyFocalLength;
	static TiffMetadataQName propertyFlashEnergy;
	static TiffMetadataQName propertySpatialFrequencyResponse;
	static TiffMetadataQName propertyNoise;
	static TiffMetadataQName propertyFocalPlaneXResolution;
	static TiffMetadataQName propertyFocalPlaneYResolution;
	static TiffMetadataQName propertyFocalPlaneResolutionUnit;
	static TiffMetadataQName propertyImageNumber;
	static TiffMetadataQName propertySecurityClassification;
	static TiffMetadataQName propertyImageHistory;
	static TiffMetadataQName propertySubjectLocation;
	static TiffMetadataQName propertyExposureIndex;
	static TiffMetadataQName propertyTIFFEPStandardID;
	static TiffMetadataQName propertySensingMethod;

	enum ByteOrder {
		ByteOrderLittleEndian = 0x4949,
		ByteOrderBigEndian = 0x4d4d
	};

	enum {
		MagicNumber = 42
	};

	enum Type {
		TypeByte = 1,
		TypeAscii = 2,
		TypeShort = 3,
		TypeLong = 4,
		TypeRational = 5,
		TypeSByte = 6,
		TypeUndefined = 7,
		TypeSShort = 8,
		TypeSLong = 9,
		TypeSRational = 10,
		TypeFloat = 11,
		TypeDouble = 12
	};

	static QList<TiffMetadataQName> tiffMetadataQNames();

	static int typeSize(Type type);

	static double lightSourceTemperature(quint16 value);

private:
	static QList<TiffMetadataQName> _tiffMetadataQNames;
};

class Exif {
public:
	static QString namespaceUri;
	static TiffMetadataQName propertyExifIFD;
	static TiffMetadataQName propertyGPSIFD;
	static TiffMetadataQName propertyExifVersion;
	static TiffMetadataQName propertyDateTimeDigitized;
	static TiffMetadataQName propertyComponentsConfiguration;
	static TiffMetadataQName propertySubjectArea;
	static TiffMetadataQName propertyMakerNote;
	static TiffMetadataQName propertyUserComment;
	static TiffMetadataQName propertySubsecTime;
	static TiffMetadataQName propertySubsecTimeOriginal;
	static TiffMetadataQName propertySubsecTimeDigitized;
	static TiffMetadataQName propertyFlashPixVersion;
	static TiffMetadataQName propertyColorSpace;
	static TiffMetadataQName propertyPixelXDimension;
	static TiffMetadataQName propertyPixelYDimension;
	static TiffMetadataQName propertyRelatedSoundFile;
	static TiffMetadataQName propertyInteroperabilityIFD;
	static TiffMetadataQName propertyFlashEnergy;
	static TiffMetadataQName propertySpatialFrequencyResponse;
	static TiffMetadataQName propertyFocalPlaneXResolution;
	static TiffMetadataQName propertyFocalPlaneYResolution;
	static TiffMetadataQName propertyFocalPlaneResolutionUnit;
	static TiffMetadataQName propertySubjectLocation;
	static TiffMetadataQName propertyExposureIndex;
	static TiffMetadataQName propertySensingMethod;
	static TiffMetadataQName propertyFileSource;
	static TiffMetadataQName propertySceneType;
	static TiffMetadataQName propertyCFAPattern;
	static TiffMetadataQName propertyCustomRendered;
	static TiffMetadataQName propertyExposureMode;
	static TiffMetadataQName propertyWhiteBalance;
	static TiffMetadataQName propertyDigitalZoomRatio;
	static TiffMetadataQName propertyFocalLengthIn35mmFilm;
	static TiffMetadataQName propertySceneCaptureType;
	static TiffMetadataQName propertyGainControl;
	static TiffMetadataQName propertyContrast;
	static TiffMetadataQName propertySaturation;
	static TiffMetadataQName propertySharpness;
	static TiffMetadataQName propertyDeviceSettingDescription;
	static TiffMetadataQName propertySubjectDistanceRange;
	static TiffMetadataQName propertyImageUniqueID;

	static QList<TiffMetadataQName> tiffMetadataQNames();

private:
	static QList<TiffMetadataQName> _tiffMetadataQNames;
};

class GPSInfo {
 /*
GPSVersionID			0	0	BYTE		4
GPSLatitudeRef			1	1	ASCII		2
GPSLatitude				2	2	RATIONAL	3
GPSLongitudeRef			3	3	ASCII		2
GPSLongitude			4	4	RATIONAL	3
GPSAltitudeRef			5	5	BYTE		1
GPSAltitude				6	6	RATIONAL	1
GPSTimeStamp			7	7	RATIONAL	3
GPSSatellites			8	8	ASCII		Any
GPSStatus				9	9	ASCII		2
GPSMeasureMode			10	A	ASCII		2
GPSDOP					11	B	RATIONAL	1
GPSSpeedRef				12	C	ASCII		2
GPSSpeed				13	D	RATIONAL	1
GPSTrackRef				14	E	ASCII		2
GPSTrack				15	F	RATIONAL	1
GPSImgDirectionRef		16	10	ASCII		2
GPSImgDirection			17	11	RATIONAL	1
GPSMapDatum				18	12	ASCII		Any
GPSDestLatitudeRef		19	13	ASCII		2
GPSDestLatitude			20	14	RATIONAL	3
GPSDestLongitudeRef		21	15	ASCII		2
GPSDestLongitude		22	16	RATIONAL	3
GPSDestBearingRef		23	17	ASCII		2
GPSDestBearing			24	18	RATIONAL	1
GPSDestDistanceRef		25	19	ASCII		2
GPSDestDistance			26	1A	RATIONAL	1
GPSProcessingMethod		27	1B	UNDEFINED	Any
GPSAreaInformation		28	1C	UNDEFINED	Any
GPSDateStamp			29	1D	ASCII		11
GPSDifferential			30	1E	SHORT		1
 */
};

class TiffEntry {
public:
	TiffEntry(TiffMetadataQName qName, QVariant value);
	TiffEntry(TiffMetadataQName qName, QList<QVariant> values);
	TiffEntry(quint16 tag, Tiff::Type type, QVariant value);
	TiffEntry(quint16 tag, Tiff::Type type, QList<QVariant> values);

	quint16 tag();
	Tiff::Type type();
	QList<QVariant> values();

	quint32 valueCount();
	quint32 valueSize();

private:
	quint16 _tag;
	Tiff::Type _type;
	QList<QVariant> _values;
};

class TiffDirectory {
public:
	void addTag(TiffEntry* entry);

	QMap<quint16, TiffEntry*> entryMap();

	quint32 headerSize();
	quint32 valueSize();
	quint32 size();

private:
	QMap<quint16, TiffEntry*> _entries;
};

class TiffReader : public MetadataReader {
public:
	TiffReader(QIODevice* device);
	virtual ~TiffReader();

	virtual MetadataNode* read(MetadataNode* parent=0);

	MetadataDocument* document();

protected:
	QDataStream* _dataStream;
	MetadataDocument* _document;

	void detectByteOrder();

	quint8 readByte();
	quint16 readShort();
	quint32 readLong();
	Rational readRational();
	qint8 readSByte();
	qint16 readSShort();
	qint32 readSLong();
	Rational readSRational();
	float readFloat();
	double readDouble();

	QList<QVariant> readValues(quint16 tag, quint16 type, quint32 count);

	virtual void parseEntry(MetadataNode* parent, quint32 offset);
	virtual void parseExif(MetadataNode* parent, quint32 offset);
	virtual void parseDirectory(MetadataNode* parent, quint32 offset);
	virtual void parseDirectories(MetadataNode* parent, quint32 offset);
	virtual void parseHeader(MetadataNode* parent, quint32 offset=0);

	virtual MetadataQName tagQName(quint16 tag);
	virtual QVariant tagValue(quint16 tag, QVariant value);

	virtual void parseMakerNote(MetadataNode* parent, quint32 offset);

	static QMap<quint16, MetadataQName> _tiffTagMetadataQNameMap;
	static QMap<quint16, MetadataQName> tiffTagMetadataQNameMap();
	static QMap<quint16, MetadataQName> _exifTagMetadataQNameMap;
	static QMap<quint16, MetadataQName> exifTagMetadataQNameMap();
};

class TiffImageReader {
public:
	TiffImageReader(QIODevice* device);

	Image* read(MetadataResource* resource, Image* image=0, ImageFormat format=ImageFormat());
	Image* read(quint32 offset, quint32 length, Image* image=0, ImageFormat format=ImageFormat());
	Image* readLosslessJpeg(quint32 offset, quint32 length, Image* image=0, ImageFormat format=ImageFormat());
	Image* readLosslessJpegTiled(QList<quint32> offsets, QList<quint32> lengths, int tileWidth, int tileHeight, int tileColumns, Image* image=0, ImageFormat format=ImageFormat());

	QImage readQImage(quint32 offset, quint32 length, const char* format=0);

private:
	QIODevice* _device;

	Image* createImage(ImageFormat format, int width, int height);
	void fixImage(Image* image, ImageFormat format);
};

class TiffWriter : public ImageWriter {
public:
	TiffWriter();
	virtual ~TiffWriter();

	virtual bool accepts(QString format);
	virtual bool write(QIODevice* device, Image* image, MetadataResource* resource, BergPhoto::Parameters parameters);

	static QString name;

protected:
	void writeHeader(QDataStream* dataStream);
	void writeIdf(QDataStream* dataStream, TiffDirectory* directory, QByteArray data=QByteArray(), quint32 nextIdf=0);
	void writeValues(QDataStream* dataStream, TiffEntry* entry);
	void writeRational(QDataStream* dataStream, Rational value);
	void writeSRational(QDataStream* dataStream, Rational value);

	void addTiffMetadataTags(TiffDirectory* directory, MetadataResource* resource);
	void addExifMetadataTags(TiffDirectory* directory, MetadataResource* resource);

	virtual QVariant tagValue(int tag, QVariant value);

private:
	static QMap<MetadataQName, TiffMetadataQName> _tiffMetadataQNameMap;
	static QMap<MetadataQName, TiffMetadataQName> tiffMetadataQNameMap();
	static QMap<MetadataQName, TiffMetadataQName> _exifMetadataQNameMap;
	static QMap<MetadataQName, TiffMetadataQName> exifMetadataQNameMap();
};

class TiffSimpleMetadataTransform : public MetadataTransform {
public:
	virtual ~TiffSimpleMetadataTransform();

	virtual MetadataNode* transform(MetadataNode* source);
};

#endif /* TIFF_H_ */
